/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	sysace_interface												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module sysace_interface(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [6:0]		ADDR_IN,		/* Address bus from Sysace Bridge */
	input				BRDY,			/* Buffer Ready from System ACE IC */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input				CLK_IN,		/* System ACE Clock */
	input [15:0]	DATA_IN,		/* Data bus from Sysace Bridge */
	input				DEBUG,		/* Debug Enable */
	input [15:0]	DQ_IN,		/* Data bus from System ACE IC */
	input				EN,			/* Enable */
	input				IRQ,			/* System ACE IRQ */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input [3:0]		TDELAY,		/* Delay Amount */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [6:0]	ADDR_OUT,	/* Address bus to System ACE IC */
	output			BRDY_SA,		/* Buffer Ready to Sysace Bridge */
	output			BUSY,			/* Interface Busy */
	output [15:0]	DATA_OUT,	/* Data bus to Sysace Bridge */
	output			DIR,			/* Data bus direction */
	output [15:0]	DQ_OUT,		/* Data bus to System ACE IC */
	output			END,			/* Interface Enabled */
	output			IRQ_SA,		/* System ACE IRQ to Sysace Bridge */
	output			NCE,			/* Chipselect */
	output			NOE,			/* Output Enable to System ACE IC */
	output			NWE			/* Write Enable to System ACE IC */
	);

/*****Registers****************************************************************/

	reg [6:0]	addr_r;			/* Address */
	reg [6:0]	addrr;			/* Address System ACE domain */
	reg			brdy_r;			/* Buffer Ready */
	reg [15:0]	data_in_r;		/* Data In */
	reg [15:0]	data_out_r;		/* Data Out */
	reg [15:0]	datain_r;		/* Data In System ACE Domain */
	reg [15:0]	dataout_r;		/* Data Out System ACE Domain */
	reg			debug_r;			/* Debug Enable */
	reg [3:0]	delay_count;	/* Delay Counter */
	reg			enn;				/* Enable */
	reg			irq_r;			/* System ACE IRQ */
	reg			nce_r;			/* Chipselect to System ACE IC */
	reg			noe_r;			/* Output Enable to System ACE IC */
	reg			nwe_r;			/* Output Write to System ACE IC */
	reg			oel_r;			/* Output Enabled */
	reg			oen;				/* Output Enable System ACE Domain */
	reg [3:0]	tdelay_r;		/* Delay Amount */
	reg			wel_r;			/* Write Enabled */
	reg			wen;				/* Write Enable System ACE Domain */

/*****Initialization****************************************************************/

	initial
	begin
		addr_r = 7'b0;
		addrr = 7'b0;
		brdy_r = 1'b0;
		datain_r = 16'b0;
		dataout_r = 16'b0;
		data_in_r = 16'b0;
		data_out_r = 16'b0;
		debug_r = 1'b0;
		delay_count = 4'b1111;
		enn = 1'b0;
		irq_r = 1'b0;
		nce_r = 1'b1;
		noe_r = 1'b1;
		nwe_r = 1'b1;
		oel_r = 1'b0;
		oen = 1'b0;
		tdelay_r = 4'b0;
		wel_r = 1'b0;
		wen = 1'b0;
	end

/*********************************************************************/

	assign ADDR_OUT = addr_r;
	assign BRDY_SA = brdy_r;
	assign BUSY = ~(nwe_r & noe_r) | (delay_count < tdelay_r);
	assign DATA_OUT = dataout_r;
	assign DIR = nwe_r;
	assign DQ_OUT = data_in_r;
	assign END = ~nce_r;
	assign IRQ_SA = irq_r;
	assign NCE = nce_r;
	assign NOE = noe_r;
	assign NWE = nwe_r;

/*********************************************************************/

	/* System ACE Domain */
	always@(posedge CLK_IN)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			addrr = 7'b0;
			datain_r = 16'b0;
			debug_r = 1'b0;
			enn = 1'b0;
			oen = 1'b0;
			tdelay_r = 4'b0;
			wen = 1'b0;
		end
		else
		begin
			/* Latch Signals */
			addrr = ADDR_IN;
			datain_r = DATA_IN;
			debug_r = DEBUG;
			enn = EN;
			oen = OE;
			tdelay_r = TDELAY;
			wen = WE;
		end
	end

	/* System Domain */
	always@(posedge CLK)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			brdy_r = 1'b0;
			dataout_r = 16'b0;
			irq_r = 1'b0;
		end
		else
		begin
			/* Latch Signals */
			brdy_r = BRDY;
			dataout_r = data_out_r;
			irq_r = IRQ;
		end
	end

	always@(posedge CLK_IN)
	begin
		if (RESET == 0)
		begin
			/* Reset */
			addr_r = 7'b0;
			data_in_r = 16'b0;
			data_out_r = 16'b0;
			delay_count = 4'b1111;
			nce_r = 1'b1;
			noe_r = 1'b1;
			nwe_r = 1'b1;
			oel_r = 1'b0;
			wel_r = 1'b0;
		end
		else
		begin
			if (enn & nce_r)
			begin
				/* Enable */
				nce_r = 1'b0;
				data_in_r = 16'b0;
				data_out_r = 16'b0;
			end
			if (~nce_r)
			begin
				/* Wait for Delay */
				if (delay_count >= 4'b11)
				begin
					if (nwe_r & noe_r)
					begin
						if (~enn) nce_r = 1'b1;
					end
					if (~wen & ~nwe_r)
					begin
						nwe_r = 1'b1;
						delay_count = 4'b0;
					end
					if (~oen & ~noe_r)
					begin
						/* Capture Data */
						noe_r = 1'b1;
						if (debug_r)
						begin
							/* Debug Mode */
							data_out_r = 16'h1234;
						end
						else
						begin
							data_out_r = DQ_IN;
						end
						delay_count = 4'b0;
					end
				end
				else
				begin
					if (~noe_r)
					begin
						/* Capture Data */
						if (debug_r)
						begin
							/* Debug Mode */
							data_out_r = 16'h5678;
						end
						else
						begin
							data_out_r = DQ_IN;
						end
					end
				end
				if (delay_count < 4'b1111)
				begin
					delay_count = delay_count + 1;
				end
			end
			if (~nce_r & wen & nwe_r & ~wel_r)
			begin
				wel_r = 1'b1;
				addr_r = addrr;
				data_in_r = datain_r;
				delay_count = 4'b0;
			end
			if (wel_r)
			begin
				if (delay_count >= 4'b10)
				begin
					if (~nce_r & nwe_r)
					begin
						nwe_r = 1'b0;
						wel_r = 1'b0;
						delay_count = 4'b0;
					end
				end
			end
			if (~nce_r & oen & noe_r & ~oel_r)
			begin
				oel_r = 1'b1;
				addr_r = addrr;
				delay_count = 4'b0;
			end
			if (oel_r)
			begin
				if (delay_count >= 4'b10)
				begin
					if (~nce_r & noe_r)
					begin
						noe_r = 1'b0;
						oel_r = 1'b0;
						delay_count = 4'b0;
					end
				end
			end
		end
	end

endmodule
