/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	timer_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module timer_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				SRT,			/* Start Transaction */
	input				TMRCLK,		/* Timer Clock 10 MHZ */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output [31:0]	STATUS,		/* Control Register */
	output			TDIRQ,		/* Down Counter Interrupt */
	output			TFIRQ,		/* Free Counter Interrupt */
	/* IO PORTS */
	inout				CLK_EXT		/* External Clock */
	);

/*****Signals****************************************************************/

	wire [3:0]	BM;			/* Byte Selects */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire			MEN;			/* Transaction Enabled for Timer Configuration Space */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* Configuration Register Data Write Mask */
	wire [31:0] BASE_ADDR;	/* */
	wire [31:0] MAX_ADDR;	/* */

/*****Registers****************************************************************/
	
	reg [31:0]	basetagr;		/* BASETAG Register */
	reg			clr;				/* Clear Interrupt */
	reg [31:0]	control;			/* Control Register */
	reg [31:0]	data_in_m_reg;	/* Masked Incoming Data */
	reg [3:0]	dnecnt;			/* Transaction Progress Counter */
	reg			dner;				/* Configuration Space Transaction Done */
	reg [63:0]	downcount;		/* Down Counter */
	reg [63:0]	freecount;		/* Free Counter */
	reg			irq_td;			/* Down Counter Interrupt */
	reg			irq_tf;			/* Free Counter Interrupt */
	reg [63:0]	loaddown;		/* Load Down Counter */
	reg [63:0]	loadfree;		/* Load Free Counter */
	reg			men_reg;			/* Transaction Enabled for Timer Configuration Space */
	reg			ovf_td;			/* Down Counter Overflow */
	reg			ovr_td;			/* Down Counter Overrun */
	reg [31:0]	reg_m_0_reg;	/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;		/* Masked Configuration Register Data */
	reg			rst;				/* Reset Counters */
	reg			updateh_td;		/* Update Down Counter High */
	reg			updateh_tf;		/* Update Free Counter High */
	reg			updatel_td;		/* Update Down Counter Low */
	reg			updatel_tf;		/* Update Free Counter Low */

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =		16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =		16'h0004;	/* Control Register Relative Address */
	parameter	FREECOUNT_HIGH =	16'h0008;	/* Free Counter High Register Relative Address */
	parameter	FREECOUNT_LOW =	16'h000C;	/* Free Counter Low Register Relative Address */
	parameter	DOWNCOUNT_HIGH =	16'h0010;	/* Down Counter High Register Relative Address */
	parameter	DOWNCOUNT_LOW =	16'h0014;	/* Down Counter Low Register Relative Address */
	parameter	RESERVED0REG =	16'h0018;
	parameter	RESERVED1REG = 	16'h001c;

	/* BASETAG Register */
	parameter	BASEl =				31;			/* BASE Bit Range Left */
	parameter	BASEr =				16;			/* BASE Bit Range Right */
	parameter 	TAGl =				15;			/* TAG Bit Range Left */
	parameter	TAGr =				0;				/* TAG Bit Range Right */
	parameter	TAG =					7;				/* TAG Value */
	
	/* Control Register */
	parameter	TD_OVERRUN =		11;			/* Down Counter Overrun */
	parameter	TD_OVERFLOW =		10;			/* Down Counter Overflow */
	parameter	TF_INT =				9;				/* Free Counter Interrupt */
	parameter	TD_INT =				8;				/* Down Counter Interrupt */
	parameter	T_CLKSELl =			7;				/* Clock Select Bit Range Left */
	parameter	T_CLKSELr =			6;				/* Clock Select Bit Range Right */
	parameter	TF_INT_ENABLE =	5;				/* Free Counter Interrupt Enable */
	parameter	T_RESET =			4;				/* Soft Reset */
	parameter	T_CLKO =				3;				/* Output Clock */
	parameter	TD_INT_ENABLE =	1;				/* Down Counter Interrupt Enable */ 
	parameter	T_ENABLE =			0;				/* Timer Enable */

	parameter	SIZE = 32;
	
/*****Initialization****************************************************************/

	initial
	begin
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		clr = 1'b0;
		control = 32'b0;
		dnecnt = 4'b1111;
		dner = 1'b1;
		downcount = 64'b0;
		freecount = 64'b0;
		irq_td = 1'b0;
		irq_tf = 1'b0;
		loaddown = 64'b0;
		loadfree = 64'b0;
		men_reg = 1'b0;
		ovf_td = 1'b0;
		ovr_td = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		rst = 1'b1;
		updatel_td = 1'b0;
		updatel_tf = 1'b0;
		updateh_td = 1'b0;
		updateh_tf = 1'b0;
	end

/*********************************************************************/
	
	assign BASETAG = basetagr;
	assign CLK_EXT = (control[T_CLKO])?TMRCLK:1'bz;
	assign DNE = dner;
	assign STATUS = control;
	assign TDIRQ = irq_td & control[T_ENABLE] & control[TD_INT_ENABLE] & ~clr;
	assign TFIRQ = irq_tf & control[T_ENABLE] & control[TF_INT_ENABLE] & ~clr;

/*****Address Decoding****************************************************************/

	assign BASE_ADDR = CADDR;
	assign MAX_ADDR = CADDR + SIZE;
	assign MEN = (ADDR_IN >= BASE_ADDR) & (ADDR_IN < MAX_ADDR);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG = {{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN = {{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M = DATA_IN_C & WMASK_IN;
	assign REG_M = reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/
	
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CONTROLREG)?control:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == FREECOUNT_HIGH))?freecount[63:32]:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == FREECOUNT_LOW))?freecount[31:0]:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == FREECOUNT_HIGH))?loadfree[63:32]:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == FREECOUNT_LOW))?loadfree[31:0]:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == DOWNCOUNT_HIGH))?downcount[63:32]:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == DOWNCOUNT_LOW))?downcount[31:0]:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == DOWNCOUNT_HIGH))?loaddown[63:32]:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == DOWNCOUNT_LOW))?loaddown[31:0]:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == RESERVED0REG)?32'b0:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == RESERVED1REG)?32'b0:32'bz;

	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*********************************************************************/
		
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,control[T_RESET],dnecnt[3],dner})
			4'b0xxx	:		begin
									/* Reset */
									basetagr = 32'b0;
									basetagr[TAGl:TAGr] = TAG;
									clr = 1'b1;
									control = 32'b0;
									dner = 1'b1;
									loaddown = 64'b0;
									loadfree = 64'b0;
									updateh_td = 1'b0;
									updateh_tf = 1'b0;
									updatel_td = 1'b0;
									updatel_tf = 1'b0;
									rst = 1'b1;
								end
			4'b11xx	:		begin
									/* Soft Reset */
									basetagr = 32'b0;
									basetagr[TAGl:TAGr] = TAG;
									clr = 1'b1;
									control = 32'b0;
									dner = 1'b1;
									loaddown = 64'b0;
									loadfree = 64'b0;
									updateh_td = 1'b0;
									updateh_tf = 1'b0;
									updatel_td = 1'b0;
									updatel_tf = 1'b0;
									rst = 1'b1;
								end
			4'b1000	:		begin
									/* Update Registers */
									dner = 1'b1;
									if (men_reg & OE)
									begin
										casex(ADDR_IN[BASEr-1:0])
											CONTROLREG	:	begin
																	/* Clear Interrupt */
																	if (irq_td || irq_tf) clr = 1'b1;
																end
											default		:	begin
																	/* Do Nothing */
																	/* Address not Recognized */
																end
										endcase
									end
									if (men_reg & WE)
									begin
										casex(ADDR_IN[BASEr-1:0])
											BASETAGREG			:	begin
																			/* Update Basetag Register */
																			basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
																		end
											CONTROLREG 			:	begin
																			/* Update Control Register */
																			control = {(data_in_m_reg[31:12] | reg_m_reg[31:12]),control[11:8],(data_in_m_reg[7:0] | reg_m_reg[7:0])};
																		end
											DOWNCOUNT_HIGH		:	begin
																			/* Update Down Counter High Register */
																			loaddown[63:32] = data_in_m_reg | reg_m_reg;
																			updateh_td = 1'b1;
																		end
											DOWNCOUNT_LOW		:	begin
																			/* Update Down Counter Low Register */
																			loaddown[31:0] = data_in_m_reg | reg_m_reg;
																			updatel_td = 1'b1;
																		end
											FREECOUNT_HIGH		:	begin
																			/* Update Free Counter High Register */
																			loadfree[63:32] = data_in_m_reg | reg_m_reg;
																			updateh_tf = 1'b1;
																		end
											FREECOUNT_LOW		:	begin
																			/* Update Free Counter Low Register */
																			loadfree[31:0] = data_in_m_reg | reg_m_reg;
																			updatel_tf = 1'b1;
																		end
											default				:	begin
																			/* Do Nothing */
																			/* Address not Recognized */
																		end
										endcase
									end
								end
			4'b10x1	:		begin
									/* Wait for incoming request */
									if (SRT & men_reg)
									begin
										/* Request Recognized */
										dner = 1'b0;
									end
									if (clr && (~irq_td && ~irq_tf))
									begin
										/* Interrupt Cleared */
										clr = 1'b0;
									end
									if ((downcount == 64'b0) && (freecount == 64'b0) && (rst))
									begin
										/* Reset Counters */
										rst = 1'b0;
									end
									if (downcount[31:0] == loaddown[31:0])
									begin
										/* Down Counter Low Register Updated */
										updatel_td = 1'b0;
									end
									if (freecount[31:0] == loadfree[31:0])
									begin
										/* Free Counter Low Register Updated */
										updatel_tf = 1'b0;
									end
									if (downcount[63:32] == loaddown[63:32])
									begin
										/* Down Counter High Register Updated */
										updateh_td = 1'b0;
									end
									if (freecount[63:32] == loadfree[63:32])
									begin
										/* Free Counter High Register Updated */
										updateh_tf = 1'b0;
									end
									if (irq_td)
									begin
										/* Down Counter Interrupt */
										control[TD_INT] = 1'b1;
									end
									else
									begin
										control[TD_INT] = 1'b0;
									end
									if (irq_tf)
									begin
										/* Free Counter Interrupt */
										control[TF_INT] = 1'b1;
									end
									else
									begin
										control[TF_INT] = 1'b0;
									end
									if (ovf_td)
									begin
										/* Down Counter Overflow */
										control[TD_OVERFLOW] = 1'b1;
									end
									else
									begin
										control[TD_OVERFLOW] = 1'b0;
									end
									if (ovr_td)
									begin
										/* Down Counter Overrun */
										control[TD_OVERRUN] = 1'b1;
									end
									else
									begin
										control[TD_OVERRUN] = 1'b0;
									end
								end
			default	:		begin
									/* Do Nothing */
								end
		endcase
	end
	
	/* Free Counter */
	always@(posedge TMRCLK)
	begin
		if (rst)
		begin
			/* Reset */
			irq_tf = 1'b0;
			freecount = 64'b0;
		end
		else
		begin
			if (updatel_tf || updateh_tf)
			begin
				/* Update Counter */
				if (updatel_tf) freecount[31:0] = loadfree[31:0];
				if (updateh_tf) freecount[63:32] = loadfree[63:32];
			end
			else
			begin
				if (control[T_ENABLE])
				begin
					if ((clr) && (irq_tf))
					begin
						/* Clear Interrupt */
						irq_tf = 1'b0;
					end
					if (freecount == 64'hffffffffffffffff)
					begin
						/* Set Interrupt */
						irq_tf = 1'b1;
					end
					/* Increment Counter */
					freecount = freecount + 1;
				end
			end
		end
	end
	
	/* Down Counter */
	always@(posedge TMRCLK)
	begin
		if (rst)
		begin
			/* Reset */
			irq_td = 1'b0;
			ovr_td = 1'b0;
			ovf_td = 1'b0;
			downcount = 64'b0;
		end
		else
		begin
			if (updatel_td || updateh_td)
			begin
				/* Update Counter */
				if (updatel_td) downcount[31:0] = loaddown[31:0];
				if (updateh_td) downcount[63:32] = loaddown[63:32];
			end
			else
			begin
				if (control[T_ENABLE])
				begin
					if ((clr) && (irq_td))
					begin
						/* Clear Interrupt */
						irq_td = 1'b0;
						ovf_td = 1'b0;
						ovr_td = 1'b0;
					end
					if (ovf_td & (downcount == loaddown))
					begin
						/* Set Overrun */
						ovr_td = 1'b1;
					end
					if (irq_td & (downcount == loaddown))
					begin
						/* Set Overflow */
						ovf_td = 1'b1;
					end
					if (downcount == 64'b0)
					begin
						/* Set Interrupt */
						irq_td = 1'b1;
					end
					/* Decrement Counter */
					downcount = downcount - 1;
				end
			end
		end
	end
	
endmodule
