/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	tocp0																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module tocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	/* OUTPUT PORTS */
	output			RESET_CP0,			/* Reset Coprocessor0 */
	/* PA INPUT PORTS */
	input				EXTSTALL_PA,		/* Stall for Extension */
	input				GR_EXT0_PA,			/* Extension 0 Grant Pipeline Resources */
	input				GR_EXT1_PA,			/* Extension 1 Grant Pipeline Resources */
	/* PA OUTPUT PORTS */
	output			EXTSTALL_CP0,		/* Stall for Extension */
	/* IF INPUT PORTS */
	input				EXC_IF,				/* Instruction Fetch Exception */
	input [31:0]	EXCADD_IF,			/* Instruction Fetch Exception Address/Data */
	input [6:0]		EXCCODE_IF,			/* Instruction Fetch Exception Code */
	input [31:0]	EXCPC_IF,			/* Instruction Fetch Exception PC */
	/* IF OUTPUT PORTS */
	output			EXC_IF_CP0,			/* Instruction Fetch Exception */
	output [31:0]	EXCADD_IF_CP0,		/* Instruction Fetch Exception Address/Data */
	output [6:0]	EXCCODE_IF_CP0,	/* Instruction Fetch Exception Code */
	output [31:0]	EXCPC_IF_CP0,		/* Instruction Fetch Exception PC */
	/* ID INPUT PORTS */
	input				BRANCH_ID,			/* Instruction Decode Branch Delay Instruction */
	input				BUBBLE_ID,			/* Bubble in Instruction Decode */
	input				EXC_ID,				/* Instruction Decode Exception */
	input [31:0]	EXCADD_ID,			/* Instruction Decode Exception Address/Data */
	input [6:0]		EXCCODE_ID,			/* Instruction Decode Exception Code */
	input [31:0]	EXCPC_ID,			/* Instruction Decode Exception PC */
	input				FLUSH_ID,			/* Instruction Fetch Flush */
	/* ID OUTPUT PORTS */
	output			BD_ID_CP0,			/* Instruction Decode Branch Delay Instruction */
	output			BUBBLE_ID_CP0,		/* Bubble in Instruction Decode */
	output			EXC_ID_CP0,			/* Instruction Decode Exception */
	output [31:0]	EXCADD_ID_CP0,		/* Instruction Decode Exception Address/Data */
	output [6:0]	EXCCODE_ID_CP0,	/* Instruction Decode Exception Code */
	output [31:0]	EXCPC_ID_CP0,		/* Instruction Decode Exception PC */
	output			FLUSH_CP0,			/* Instruction Fetch Flush */
	/* EX INPUT PORTS */
	input				BRANCH_EX,			/* Execute Branch Delay Instruction */
	input				BUBBLE_EX,			/* Bubble in Execute */
	input [4:0]		CP0RDREG_EX,		/* CP0 Register Read Port 1 Register Number Execute Phase */
	input				CP0REGWRITE_EX,	/* Coprocessor Register Write Enable Execute Phase */
	input [4:0]		CP0WRREG_EX,		/* CP0 Register Write Port Register Number  */
	input [31:0]	CP0WRREGDATA_EX,	/* CP0 Register Write Port Data Execute Phase */
	input				EXC_EX,				/* Execute Exception */
	input [31:0]	EXCADD_EX,			/* Execute Exception Address/Data */
	input [6:0]		EXCCODE_EX,			/* Execute Exception Code */
	input [31:0]	EXCPC_EX,			/* Execute Exception PC */
	input				RFE_EX,				/* Return from Exception */
	/* EX OUTPUT PORTS */
	output			BD_EX_CP0,			/* Execute Branch Delay Instruction */
	output			BUBBLE_EX_CP0,		/* Bubble in Execute */
	output [4:0]	CP0RDREG_CP0,		/* CP0 Register Read Port 1 Register Number */
	output			CP0REGWRITE_CP0,	/* Coprocessor Register Write Enable */
	output [4:0]	CP0WRREG_CP0,		/* CP0 Register Write Port Register Number */
	output [31:0]	CP0WRREGDATA_CP0,	/* CP0 Register Write Port Data */
	output			EXC_EX_CP0,			/* Execute Exception */
	output [31:0]	EXCADD_EX_CP0,		/* Execute Exception Address/Data */
	output [6:0]	EXCCODE_EX_CP0,	/* Execute Exception Code */
	output [31:0]	EXPC_EX_CP0,		/* Execute Exception PC */
	output			RFE_CP0,				/* Return from Exception */
	/* MA INPUT PORTS */
	input				BRANCH_MA,			/* Memory Access Branch Delay Instruction */
	input				BUBBLE_MA,			/* Bubble in Memory Access */
	input				EXC_MA,				/* Memory Access Exception */
	input [31:0]	EXCADD_MA,			/* Memory Access Exception Address/Data */
	input [6:0]		EXCCODE_MA,			/* Memory Access Exception Code */
	input [31:0]	EXCPC_MA,			/* Memory Access Exception PC */
	/* MA OUTPUT PORTS */
	output			BD_MA_CP0,			/* Memory Access Branch Delay Instruction */
	output			BUBBLE_MA_CP0,		/* Bubble in Memory Access */
	output			EXC_MA_CP0,			/* Memory Access Exception */
	output [31:0]	EXCADD_MA_CP0,		/* Memory Access Exception Address/Data */
	output [6:0]	EXCCODE_MA_CP0,	/* Memory Access Exception Code */
	output [31:0]	EXPC_MA_CP0,		/* Memory Access Exception PC */
	/* WB INPUT PORTS */
	input				BRANCH_WB,			/* Writeback Branch Delay Instruction */
	/* WB OUTPUT PORTS */
	output			BD_WB_CP0,			/* Writeback Branch Delay Instruction */
	/* EXT0 INPUT PORTS */
	input				BD_EXT0,					/* Extension Branch Delay Instruction */
	input [4:0]		CP0RDREG_EXT0,			/* CP0 Register Read Port 1 Register Number Extension 0 */
	input				CP0REGWRITE_EXT0,		/* Coprocessor Register Write Enable Extension 0 */
	input [4:0]		CP0WRREG_EXT0,			/* CP0 Register Write Port Register Number Extension 0 */
	input [31:0]	CP0WRREGDATA_EXT0,	/* CP0 Register Write Port Data Extension 0 */
	input				EXC_EXT0,				/* Extension 0 Exception */
	input [31:0]	EXCADD_EXT0,			/* Extension 0 Exception Address/Data */
	input [6:0]		EXCCODE_EXT0,			/* Extension 0 Exception Code */
	input [31:0]	EXCPC_EXT0,				/* Extension 0 Exception PC */
	input				REMA_EXT0,				/* Extension 0 Re-enter at Memory Access */
	input				REWB_EXT0,				/* Extension 0 Re-enter at Writeback */
	input				RFE_EXT0,				/* Return from Exception */
	/* EXT1 INPUT PORTS */
	input				BD_EXT1,					/* Extension Branch Delay Instruction */
	input [4:0]		CP0RDREG_EXT1,			/* CP0 Register Read Port 1 Register Number Extension 1 */
	input				CP0REGWRITE_EXT1,		/* Coprocessor Register Write Enable Extension 1 */
	input [4:0]		CP0WRREG_EXT1,			/* CP0 Register Write Port Register Number Extension 1 */
	input [31:0]	CP0WRREGDATA_EXT1,	/* CP0 Register Write Port Data Extension 1 */
	input				EXC_EXT1,				/* Extension 1 Exception */
	input [31:0]	EXCADD_EXT1,			/* Extension 1 Exception Address/Data */
	input [6:0]		EXCCODE_EXT1,			/* Extension 1 Exception Code */
	input [31:0]	EXCPC_EXT1,				/* Extension 1 Exception PC */
	input				REMA_EXT1,				/* Extension 1 Re-enter at Memory Access */
	input				REWB_EXT1,				/* Extension 1 Re-enter at Writeback */
	input				RFE_EXT1,				/* Return from Exception */
	/* EXT OUTPUT PORTS */
	output			BD_EXT_CP0,				/* Extension Branch Delay Instruction */
	output			EXC_EXT_CP0,			/* Extension Exception */
	output [31:0]	EXCADD_EXT_CP0,		/* Extension Exception Address/Data */
	output [6:0]	EXCCODE_EXT_CP0,		/* Extension Exception Code */
	output [31:0]	EXPC_EXT_CP0,			/* Extension Exception PC */
	output			REMA_CP0,				/* Extension Re-enter at Memory Access */
	output			REWB_CP0,				/* Extension Re-enter at Writeback */
	/* HZ INPUT PORTS */
	input				STALL_HZ,				/* Pipeline Stall */
	/* HZ OUTPUT PORTS */
	output			STALL_CP0,				/* Pipeline Stall */
	/* MEM INPUT PORTS */
	input				IRQ0_MEM,				/* Interrupt */
	/* MEM OUTPUT PORTS */
	output			IRQ0_CP0					/* Interrupt */
	);

/*****Signals****************************************************************/

	wire GR_EX;		/* Extension Grant Pipeline Resources */

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Coprocessor0 */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_CP0 = reset_reg;

/*********************************************************************/

	or2_1 or0(
		.A(GR_EXT0_PA),
		.B(GR_EXT1_PA),
		.C(GR_EX)
		);

	or2_1 or1(
		.A(REMA_EXT0),
		.B(REMA_EXT1),
		.C(REMA_CP0)
		);

	or2_1 or2(
		.A(REWB_EXT0),
		.B(REWB_EXT1),
		.C(REWB_CP0)
		);

/*********************************************************************/

	assign BD_EXT_CP0 =			(GR_EX)?			1'bz:		1'b0;
	assign EXC_EXT_CP0 =			(GR_EX)?			1'bz:		1'b0;
	assign EXCADD_EXT_CP0 =		(GR_EX)?			32'bz:	32'b0;
	assign EXCCODE_EXT_CP0 = 	(GR_EX)?			7'bz:		7'h7f;
	assign EXPC_EXT_CP0 =		(GR_EX)?			32'bz:	32'b0;

/*****PA -> CP0****************************************************************/

	patocp0 pa(
		.EXTSTALL_CP0(EXTSTALL_CP0),
		.EXTSTALL_PA(EXTSTALL_PA)
		);

/*****IF -> CP0****************************************************************/

	iftocp0 i_f(
		.EXC_IF(EXC_IF),
		.EXC_IF_CP0(EXC_IF_CP0),
		.EXCADD_IF(EXCADD_IF),
		.EXCADD_IF_CP0(EXCADD_IF_CP0),
		.EXCCODE_IF(EXCCODE_IF),
		.EXCCODE_IF_CP0(EXCCODE_IF_CP0),
		.EXCPC_IF(EXCPC_IF),
		.EXCPC_IF_CP0(EXCPC_IF_CP0)
		);

/*****ID -> CP0****************************************************************/

	idtocp0 id(
		.BD_ID_CP0(BD_ID_CP0),
		.BRANCH_ID(BRANCH_ID),
		.BUBBLE_ID(BUBBLE_ID),
		.BUBBLE_ID_CP0(BUBBLE_ID_CP0),
		.EXC_ID(EXC_ID),
		.EXC_ID_CP0(EXC_ID_CP0),
		.EXCADD_ID(EXCADD_ID),
		.EXCADD_ID_CP0(EXCADD_ID_CP0),
		.EXCCODE_ID(EXCCODE_ID),
		.EXCCODE_ID_CP0(EXCCODE_ID_CP0),
		.EXCPC_ID(EXCPC_ID),
		.EXCPC_ID_CP0(EXCPC_ID_CP0),
		.FLUSH_ID(FLUSH_ID),
		.FLUSH_CP0(FLUSH_CP0)
		);

/*****EX -> CP0****************************************************************/

	extocp0 ex(
		.BD_EX_CP0(BD_EX_CP0),
		.BRANCH_EX(BRANCH_EX),
		.BUBBLE_EX(BUBBLE_EX),
		.BUBBLE_EX_CP0(BUBBLE_EX_CP0),
		.CP0RDREG_CP0(CP0RDREG_CP0),
		.CP0RDREG_EX(CP0RDREG_EX),
		.CP0REGWRITE_CP0(CP0REGWRITE_CP0),
		.CP0REGWRITE_EX(CP0REGWRITE_EX),
		.CP0WRREG_CP0(CP0WRREG_CP0),
		.CP0WRREG_EX(CP0WRREG_EX),
		.CP0WRREGDATA_CP0(CP0WRREGDATA_CP0),
		.CP0WRREGDATA_EX(CP0WRREGDATA_EX),
		.EXC_EX(EXC_EX),
		.EXC_EX_CP0(EXC_EX_CP0),
		.EXCADD_EX(EXCADD_EX),
		.EXCADD_EX_CP0(EXCADD_EX_CP0),
		.EXCCODE_EX(EXCCODE_EX),
		.EXCCODE_EX_CP0(EXCCODE_EX_CP0),
		.EXCPC_EX(EXCPC_EX),
		.EXPC_EX_CP0(EXPC_EX_CP0),
		.GR(GR_EX),
		.RFE_CP0(RFE_CP0),
		.RFE_EX(RFE_EX)
		);

/*****MA -> CP0****************************************************************/

	matocp0 ma(
		.BD_MA_CP0(BD_MA_CP0),
		.BRANCH_MA(BRANCH_MA),
		.BUBBLE_MA(BUBBLE_MA),
		.BUBBLE_MA_CP0(BUBBLE_MA_CP0),
		.EXC_MA(EXC_MA),
		.EXC_MA_CP0(EXC_MA_CP0),
		.EXCADD_MA(EXCADD_MA),
		.EXCADD_MA_CP0(EXCADD_MA_CP0),
		.EXCCODE_MA(EXCCODE_MA),
		.EXCCODE_MA_CP0(EXCCODE_MA_CP0),
		.EXCPC_MA(EXCPC_MA),
		.EXPC_MA_CP0(EXPC_MA_CP0)
		);

/*****WB -> CP0****************************************************************/

	wbtocp0 wb(
		.BD_WB_CP0(BD_WB_CP0),
		.BRANCH_WB(BRANCH_WB)
		);

/*****EXT0 -> CP0****************************************************************/

	exttocp0 ext0(
		.BD_EXT(BD_EXT0),
		.BD_EXT_CP0(BD_EXT_CP0),
		.CP0RDREG_CP0(CP0RDREG_CP0),
		.CP0RDREG_EXT(CP0RDREG_EXT0),
		.CP0REGWRITE_EX(CP0REGWRITE_EX),
		.CP0REGWRITE_CP0(CP0REGWRITE_CP0),
		.CP0REGWRITE_EXT(CP0REGWRITE_EXT0),
		.CP0WRREG_CP0(CP0WRREG_CP0),
		.CP0WRREG_EXT(CP0WRREG_EXT0),
		.CP0WRREGDATA_CP0(CP0WRREGDATA_CP0),
		.CP0WRREGDATA_EXT(CP0WRREGDATA_EXT0),
		.EXC_EXT(EXC_EXT0),
		.EXC_EXT_CP0(EXC_EXT_CP0),
		.EXCADD_EXT(EXCADD_EXT0),
		.EXCADD_EXT_CP0(EXCADD_EXT_CP0),
		.EXCCODE_EXT(EXCCODE_EXT0),
		.EXCCODE_EXT_CP0(EXCCODE_EXT_CP0),
		.EXCPC_EXT(EXCPC_EXT0),
		.EXPC_EXT_CP0(EXPC_EXT_CP0),
		.GR(GR_EXT0_PA),
		.RFE_CP0(RFE_CP0),
		.RFE_EX(RFE_EX),
		.RFE_EXT(RFE_EXT0)
		);

/*****EXT1 -> CP0****************************************************************/

	exttocp0 ext1(
		.BD_EXT(BD_EXT1),
		.BD_EXT_CP0(BD_EXT_CP0),
		.CP0RDREG_CP0(CP0RDREG_CP0),
		.CP0RDREG_EXT(CP0RDREG_EXT1),
		.CP0REGWRITE_CP0(CP0REGWRITE_CP0),
		.CP0REGWRITE_EX(CP0REGWRITE_EX),
		.CP0REGWRITE_EXT(CP0REGWRITE_EXT1),
		.CP0WRREG_CP0(CP0WRREG_CP0),
		.CP0WRREG_EXT(CP0WRREG_EXT1),
		.CP0WRREGDATA_CP0(CP0WRREGDATA_CP0),
		.CP0WRREGDATA_EXT(CP0WRREGDATA_EXT1),
		.EXC_EXT(EXC_EXT1),
		.EXC_EXT_CP0(EXC_EXT_CP0),
		.EXCADD_EXT(EXCADD_EXT1),
		.EXCADD_EXT_CP0(EXCADD_EXT_CP0),
		.EXCCODE_EXT(EXCCODE_EXT1),
		.EXCCODE_EXT_CP0(EXCCODE_EXT_CP0),
		.EXCPC_EXT(EXCPC_EXT1),
		.EXPC_EXT_CP0(EXPC_EXT_CP0),
		.GR(GR_EXT1_PA),
		.RFE_CP0(RFE_CP0),
		.RFE_EX(RFE_EX),
		.RFE_EXT(RFE_EXT1)
		);

/*****HZ -> CP0****************************************************************/

	hztocp0 hz(
		.STALL_CP0(STALL_CP0),
		.STALL_HZ(STALL_HZ)
		);

/*****MEM -> CP0****************************************************************/

	memtocp0 mem(
		.IRQ0_CP0(IRQ0_CP0),
		.IRQ0_MEM(IRQ0_MEM)
		);

/*********************************************************************/
		
	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****PA -> CP0****************************************************************/

module patocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		EXTSTALL_PA,		/* Stall for Extension */
	/* OUTPUT PORTS */
	output	EXTSTALL_CP0		/* Stall for Extension */
	);

/*********************************************************************/
	
	assign EXTSTALL_CP0	=	EXTSTALL_PA;
	
endmodule

/*****IF -> CP0****************************************************************/

module iftocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				EXC_IF,				/* Instruction Fetch Exception */
	input [31:0]	EXCADD_IF,			/* Instruction Fetch Exception Address/Data */
	input [6:0]		EXCCODE_IF,			/* Instruction Fetch Exception Code */
	input [31:0]	EXCPC_IF,			/* Instruction Fetch Exception PC */
	/* OUTPUT PORTS */
	output			EXC_IF_CP0,			/* Instruction Fetch Exception */
	output [31:0]	EXCADD_IF_CP0,		/* Instruction Fetch Exception Address/Data */
	output [6:0]	EXCCODE_IF_CP0,	/* Instruction Fetch Exception Code */
	output [31:0]	EXCPC_IF_CP0		/* Instruction Fetch Exception PC */
	);

/*********************************************************************/
	
	assign EXC_IF_CP0 		=	EXC_IF;
	assign EXCADD_IF_CP0	 	=	EXCADD_IF;
	assign EXCCODE_IF_CP0 	=	EXCCODE_IF;
	assign EXCPC_IF_CP0 		=	EXCPC_IF;

endmodule

/*****ID -> CP0****************************************************************/

module idtocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BRANCH_ID,			/* Instruction Decode Branch Delay Instruction */
	input				BUBBLE_ID,			/* Bubble in Instruction Decode */
	input				EXC_ID,				/* Instruction Decode Exception */
	input [31:0]	EXCADD_ID,			/* Instruction Decode Exception Address/Data */
	input [6:0]		EXCCODE_ID,			/* Instruction Decode Exception Code */
	input [31:0]	EXCPC_ID,			/* Instruction Decode Exception PC */
	input				FLUSH_ID,			/* Instruction Fetch Flush */
	/* OUTPUT PORTS */
	output			BD_ID_CP0,			/* Instruction Decode Branch Delay Instruction */
	output			BUBBLE_ID_CP0,		/* Bubble in Instruction Decode */
	output			EXC_ID_CP0,			/* Instruction Decode Exception */
	output [31:0]	EXCADD_ID_CP0,		/* Instruction Decode Exception Address/Data */
	output [6:0]	EXCCODE_ID_CP0,	/* Instruction Decode Exception Code */
	output [31:0]	EXCPC_ID_CP0,		/* Instruction Decode Exception PC */
	output			FLUSH_CP0			/* Instruction Fetch Flush */
	);

/*********************************************************************/
	
	assign BD_ID_CP0 			=	BRANCH_ID;
	assign BUBBLE_ID_CP0 	=	BUBBLE_ID;
	assign EXC_ID_CP0 		=	EXC_ID;
	assign EXCADD_ID_CP0 	=	EXCADD_ID;
	assign EXCCODE_ID_CP0 	= 	EXCCODE_ID;
	assign EXCPC_ID_CP0 		= 	EXCPC_ID;
	assign FLUSH_CP0 			= 	FLUSH_ID;

endmodule

/*****EX -> CP0****************************************************************/

module extocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			BRANCH_EX,			/* Execute Branch Delay Instruction */
	input 			BUBBLE_EX,			/* Bubble in Execute */
	input [4:0]		CP0RDREG_EX,		/* CP0 Register Read Port 1 Register Number */
	input 			CP0REGWRITE_EX,	/* Coprocessor Register Write Enable */
	input [4:0] 	CP0WRREG_EX,		/* CP0 Register Write Port Register Number */
	input [31:0]	CP0WRREGDATA_EX,	/* CP0 Register Write Port Data */
	input 			EXC_EX,				/* Execute Exception */
	input [31:0] 	EXCADD_EX,			/* Execute Exception Address/Data */
	input [6:0] 	EXCCODE_EX,			/* Execute Exception Code */
	input [31:0] 	EXCPC_EX,			/* Execute Exception PC */
	input 			GR,					/* Extension Grant Pipeline Resources */
	input 			RFE_EX,				/* Return from Exception */
	/* OUTPUT PORTS */
	output 			BD_EX_CP0,			/* Execute Branch Delay Instruction */
	output 			BUBBLE_EX_CP0,		/* Bubble in Execute */
	output [4:0] 	CP0RDREG_CP0,		/* CP0 Register Read Port 1 Register Number */
	output 			CP0REGWRITE_CP0,	/* Coprocessor Register Write Enable */
	output [4:0] 	CP0WRREG_CP0,		/* CP0 Register Write Port Register Number */
	output [31:0]	CP0WRREGDATA_CP0,	/* CP0 Register Write Port Data */
	output 			EXC_EX_CP0,			/* Execute Exception */
	output [31:0] 	EXCADD_EX_CP0,		/* Execute Exception Address/Data */
	output [6:0] 	EXCCODE_EX_CP0,	/* Execute Exception Code */
	output [31:0] 	EXPC_EX_CP0,		/* Execute Exception PC */
	output 			RFE_CP0				/* Return from Exception */
	);

/*********************************************************************/
	
	assign BD_EX_CP0 			= 	BRANCH_EX;
	assign BUBBLE_EX_CP0 	=	BUBBLE_EX;
	assign CP0RDREG_CP0 		= 	(GR)?							5'bz:		CP0RDREG_EX;
	assign CP0REGWRITE_CP0 	= 	(GR & ~CP0REGWRITE_EX)?	1'bz:		CP0REGWRITE_EX;
	assign CP0WRREG_CP0 		= 	(GR & ~CP0REGWRITE_EX)?	5'bz:		CP0WRREG_EX;
	assign CP0WRREGDATA_CP0 = 	(GR & ~CP0REGWRITE_EX)?	32'bz:	CP0WRREGDATA_EX;
	assign EXC_EX_CP0 		=	EXC_EX;
	assign EXCADD_EX_CP0 	= 	EXCADD_EX;
	assign EXCCODE_EX_CP0 	= 	EXCCODE_EX;
	assign EXPC_EX_CP0 		= 	EXCPC_EX;
	assign RFE_CP0 			= 	(GR & ~RFE_EX)?			1'bz:		RFE_EX;

endmodule

/*****MA -> CP0****************************************************************/

module matocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BRANCH_MA,			/* Memory Access Branch Delay Instruction */
	input				BUBBLE_MA,			/* Bubble in Memory Access */
	input				EXC_MA,				/* Memory Access Exception */
	input [31:0]	EXCADD_MA,			/* Memory Access Exception Address/Data */
	input [6:0]		EXCCODE_MA,			/* Memory Access Exception Code */
	input [31:0]	EXCPC_MA,			/* Memory Access Exception PC */
	/* OUTPUT PORTS */
	output			BD_MA_CP0,			/* Memory Access Branch Delay Instruction */
	output			BUBBLE_MA_CP0,		/* Bubble in Memory Access */
	output			EXC_MA_CP0,			/* Memory Access Exception */
	output [31:0]	EXCADD_MA_CP0,		/* Memory Access Exception Address/Data */
	output [6:0]	EXCCODE_MA_CP0,	/* Memory Access Exception Code */
	output [31:0]	EXPC_MA_CP0			/* Memory Access Exception PC */
	);

/*********************************************************************/
	
	assign BD_MA_CP0 			=	BRANCH_MA;
	assign BUBBLE_MA_CP0 	=	BUBBLE_MA;
	assign EXC_MA_CP0 		=	EXC_MA;
	assign EXCADD_MA_CP0 	=	EXCADD_MA;
	assign EXCCODE_MA_CP0 	=	EXCCODE_MA;
	assign EXPC_MA_CP0 		=	EXCPC_MA;

endmodule

/*****WB -> CP0****************************************************************/

module wbtocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		BRANCH_WB,		/* Writeback Branch Delay Instruction */
	/* OUTPUT PORTS */
	output	BD_WB_CP0		/* Writeback Branch Delay Instruction */
	);

/*********************************************************************/
	
	assign BD_WB_CP0	=	BRANCH_WB;

endmodule

/*****EXT -> CP0****************************************************************/

module exttocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BD_EXT,				/* Extension Branch Delay Instruction */
	input [4:0]		CP0RDREG_EXT,		/* CP0 Register Read Port 1 Register Number */
	input				CP0REGWRITE_EX,	/* Coprocessor Register Write Enable Execute Phase */
	input				CP0REGWRITE_EXT,	/* Coprocessor Register Write Enable Extension */
	input [4:0]		CP0WRREG_EXT,		/* CP0 Register Write Port Register Number */
	input [31:0]	CP0WRREGDATA_EXT,	/* CP0 Register Write Port Data */
	input				EXC_EXT,				/* Extension Exception */
	input [31:0]	EXCADD_EXT,			/* Extension Exception Address/Data */
	input [6:0]		EXCCODE_EXT,		/* Extension Exception Code */
	input [31:0]	EXCPC_EXT,			/* Extension Exception PC */
	input				GR,					/* Extension Grant Pipeline Resources */
	input				RFE_EX,				/* Return from Exception Execute Phase */
	input				RFE_EXT,				/* Return from Exception Extension */
	/* OUTPUT PORTS */
	output			BD_EXT_CP0,			/* Extension Branch Delay Instruction */
	output [4:0]	CP0RDREG_CP0,		/* CP0 Register Read Port 1 Register Number */
	output			CP0REGWRITE_CP0,	/* Coprocessor Register Write Enable */
	output [4:0]	CP0WRREG_CP0,		/* CP0 Register Write Port Register Number */
	output [31:0]	CP0WRREGDATA_CP0,	/* CP0 Register Write Port Data */
	output			EXC_EXT_CP0,		/* Extension Exception */
	output [31:0]	EXCADD_EXT_CP0,	/* Extension Exception Address/Data */
	output [6:0]	EXCCODE_EXT_CP0,	/* Extension Exception Code */
	output [31:0]	EXPC_EXT_CP0,		/* Extension Exception PC */
	output			RFE_CP0				/* Return from Exception */
	);
	
/*********************************************************************/

	assign BD_EXT_CP0 = 			(~GR)?							1'bz:		BD_EXT;
	assign CP0RDREG_CP0 = 		(~GR)?							5'bz:		CP0RDREG_EXT;
	assign CP0REGWRITE_CP0 = 	(~GR | CP0REGWRITE_EX)?		1'bz:		CP0REGWRITE_EXT;
	assign CP0WRREG_CP0 = 		(~GR | CP0REGWRITE_EX)?		5'bz:		CP0WRREG_EXT;
	assign CP0WRREGDATA_CP0 = 	(~GR | CP0REGWRITE_EX)?		32'bz:	CP0WRREGDATA_EXT;
	assign EXC_EXT_CP0 =			(~GR)?							1'bz:		EXC_EXT;
	assign EXCADD_EXT_CP0 = 	(~GR)?							32'bz:	EXCADD_EXT;
	assign EXCCODE_EXT_CP0 = 	(~GR)?							7'bz:		EXCCODE_EXT;
	assign EXPC_EXT_CP0 = 		(~GR)?							32'bz:	EXCPC_EXT;
	assign RFE_CP0 = 				(~GR | RFE_EX)?				1'bz:		RFE_EXT;

endmodule

/*****HZ -> CP0****************************************************************/

module hztocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		STALL_HZ,		/* Pipeline Stall */
	/* OUTPUT PORTS */
	output	STALL_CP0		/* Pipeline Stall */
	);

/*********************************************************************/

	assign STALL_CP0	=	STALL_HZ;

endmodule

/*****MEM -> CP0****************************************************************/

module memtocp0(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		IRQ0_MEM,		/* Interrupt */
	/* OUTPUT PORTS */
	output	IRQ0_CP0			/* Interrupt */
	);

/*********************************************************************/

	assign IRQ0_CP0	=	IRQ0_MEM;

endmodule
