/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:		todf															*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module todf(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	/* OUTPUT PORTS */
	output			RESET_DF,			/* Reset Dataforward */
	/* ID INPUT PORTS */
	input [4:0]		RS_ID,				/* Instruction Decode TISA Operand Register Number 1 */
	input [4:0]		RT_ID,				/* Instruction Decode TISA Operand Register Number 2 */
	/* ID OUTPUT PORTS */
	output [4:0]	RS_ID_DF,			/* Instruction Decode TISA Operand Register Number 1 */
	output [4:0]	RT_ID_DF,			/* Instruction Decode TISA Operand Register Number 2 */
	/* EX INPUT PORTS */
	input [4:0]		RS_EX,				/* Execute TISA Operand Register Number 1 */
	input [4:0]		RT_EX,				/* Execute TISA Operand Register Number 2 */
	/* EX OUTPUT PORTS */
	output [4:0]	RS_EX_DF,			/* Execute TISA Operand Register Number 1 */
	output [4:0]	RT_EX_DF,			/* Execute TISA Operand Register Number 2 */
	/* MA INPUT PORTS */
	input				REGWRITE_MA,		/* Memory Access Register Write */
	input [4:0]		WRREG_MA,			/* Memory Access TISA Destination Register Number */
	/* MA OUTPUT PORTS */
	output [4:0]	RD_MA_DF,			/* Memory Access TISA Destination Register Number */
	output			REGWRITE_MA_DF,	/* Memory Access Register Write */
	/* WB INPUT PORTS */
	input				REGWRITE_WB,		/* Writeback Register Write */
	input [4:0]		WRREG_WB,			/* Writeback TISA Destination Register Number */
	/* WB OUTPUT PORTS */
	output [4:0]	RD_WB_DF,			/* Writeback TISA Destination Register Number */
	output			REGWRITE_WB_DF		/* Writeback Register Write */
	);

/*****Registers****************************************************************/
	
	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Dataforward */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_DF = reset_reg;

/*****ID -> DF****************************************************************/

	idtodf id(
		.RS_ID(RS_ID),
		.RS_ID_DF(RS_ID_DF),
		.RT_ID(RT_ID),
		.RT_ID_DF(RT_ID_DF)
		);

/*****EX -> DF****************************************************************/

	extodf ex(
		.RS_EX(RS_EX),
		.RS_EX_DF(RS_EX_DF),
		.RT_EX(RT_EX),
		.RT_EX_DF(RT_EX_DF)
		);

/*****MA -> DF****************************************************************/

	matodf ma(
		.REGWRITE_MA(REGWRITE_MA),
		.REGWRITE_MA_DF(REGWRITE_MA_DF),
		.RD_MA_DF(RD_MA_DF),
		.WRREG_MA(WRREG_MA)
		);

/*****WB -> DF****************************************************************/

	wbtodf wb(
		.REGWRITE_WB(REGWRITE_WB),
		.REGWRITE_WB_DF(REGWRITE_WB_DF),
		.RD_WB_DF(RD_WB_DF),
		.WRREG_WB(WRREG_WB)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end
	
endmodule

/*****ID -> DF****************************************************************/

module idtodf(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [4:0] RS_ID,		/* Instruction Decode TISA Operand Register Number 1 */
	input [4:0] RT_ID,		/* Instruction Decode TISA Operand Register Number 2 */
	/* OUTPUT PORTS */
	output [4:0] RS_ID_DF,	/* Instruction Decode TISA Operand Register Number 1 */
	output [4:0] RT_ID_DF	/* Instruction Decode TISA Operand Register Number 2 */
	);

/*********************************************************************/
	
	assign RS_ID_DF	=	RS_ID;
	assign RT_ID_DF	=	RT_ID;

endmodule

/*****EX -> DF****************************************************************/

module extodf(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [4:0] RS_EX,		/* Execute TISA Operand Register Number 1 */
	input [4:0] RT_EX,		/* Execute TISA Operand Register Number 2 */
	/* OUTPUT PORTS */
	output [4:0] RS_EX_DF,	/* Execute TISA Operand Register Number 1 */
	output [4:0] RT_EX_DF	/* Execute TISA Operand Register Number 2 */
	);

/*********************************************************************/
	
	assign RS_EX_DF	=	RS_EX;
	assign RT_EX_DF	=	RT_EX;

endmodule

/*****MA -> DF****************************************************************/

module matodf(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				REGWRITE_MA,	/* Memory Access Register Write */
	input [4:0]		WRREG_MA,		/* Memory Access TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [4:0]	RD_MA_DF,		/* Memory Access TISA Destination Register Number */
	output			REGWRITE_MA_DF	/* Memory Access Register Write */
	);

/*********************************************************************/
	
	assign RD_MA_DF			=	WRREG_MA;
	assign REGWRITE_MA_DF	=	REGWRITE_MA;

endmodule

/*****WB -> DF****************************************************************/

module wbtodf(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				REGWRITE_WB,	/* Writeback Register Write */
	input [4:0]		WRREG_WB,		/* Writeback TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [4:0]	RD_WB_DF,		/* Writeback TISA Destination Register Number */
	output			REGWRITE_WB_DF	/* Writeback Register Write */
	);

/*********************************************************************/

	assign RD_WB_DF			=	WRREG_WB;
	assign REGWRITE_WB_DF	=	REGWRITE_WB;

endmodule
