/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	toex																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module toex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				PCLK,					/* Pipeline Clock */
	input				EXCE,					/* Instruction Decode Exception Flush */
	input				RESET,				/* System Reset */
	input				STALL,				/* Pipeline Stall */
	/* OUTPUT PORTS */
	output			RESET_EX,			/* Reset Execute */
	/* ID INPUT PORTS */
	input [5:0]		ALUOP_ID,			/* ALU Opcode */
	input				ALUSRC_ID,			/* ALU Operand B Switch */
	input				BLS_ID,				/* Byte Load/Store */
	input				BRANCH_ID,			/* Branch */
	input				BUBBLE_ID,			/* Bubble in Execute */
	input				CMOV_ID,				/* Conditional Move */
	input				COND_ID,				/* Conditional Operation */
	input				CP0_ID,				/* Coprocessor 0 Operation */
	input				CPREGWRITE_ID,		/* Coprocessor Register Write Enable */
	input				EXTNOP_ID,			/* Extension NOP */
	input				HLS_ID,				/* Halfword Load/Store */
	input [31:0]	IMM_ID,				/* Data Immediate */
	input				LC_ID,				/* Load/Store Conditional */
	input				LINK_ID,				/* Linked Branch */
	input				LUI_ID,				/* Load Upper Immediate */
	input				MEMREAD_ID,			/* Memory Read */
	input				MEMTOREG_ID,		/* Memory to Regiser */
	input				MEMWRITE_ID,		/* Memory Write */
	input				MEXT_ID,				/* Sign Extend Memory Data */
	input [31:0]	PC_ID,				/* Currect PC */
	input [31:0]	PC4_ID,				/* Currect PC plus 4*/
	input [4:0]		RD_ID,				/* Destination Register */
	input [31:0]	REG1DATA_ID,		/* Data from Register Read Port 1 */
	input [31:0]	REG2DATA_ID,		/* Data from Register Read Port 2 */
	input				REGDST_ID,			/* Destination Register Switch */
	input				REGWRITE_ID,		/* Register Write */
	input				RFE_ID,				/* Return From Exception */
	input				RNL_ID,				/* Right/Left Unaligned Load/Store */
	input [4:0]		RS_ID,				/* TISA Operand Register Number 1 */
	input [4:0]		RT_ID,				/* TISA Operand Register Number 2 */
	/* ID OUTPUT PORTS */
	output [5:0]	ALUOP_EX,			/* ALU Opcode */
	output			ALUSRC_EX,			/* ALU Operand B Switch */
	output			BLS_EX,				/* Byte Load/Store */
	output			BRANCH_EX,			/* Branch */
	output			BUBBLE_EX,			/* Bubble in Execute */
	output			CMOV_EX,				/* Conditional Move */
	output			COND_EX,				/* Conditional Operation */
	output			CP0_EX,				/* Coprocessor 0 Operation */
	output			CPREGWRITE_EX,		/* Coprocessor Register Write Enable */
	output			EXTNOP_EX,			/* Extension NOP */
	output			HLS_EX,				/* Halfword Load/Store */
	output [31:0]	IMM_EX,				/* Data Immediate */
	output			LC_EX,				/* Load/Store Conditional */
	output			LINK_EX,				/* Linked Branch */
	output			LUI_EX,				/* Load Upper Immediate */
	output			MEMREAD_EX,			/* Memory Read */
	output			MEMTOREG_EX,		/* Memory to Regiser */
	output			MEMWRITE_EX,		/* Memory Write */
	output			MEXT_EX,				/* Sign Extend Memory Data */
	output [31:0]	PC_EX,				/* Currect PC */
	output [31:0]	PC4_EX,				/* Currect PC plus 4*/
	output [4:0]	RD_EX,				/* Destination Register */
	output [31:0]	REG1DATA_EX,		/* Data from Register Read Port 1 */
	output [31:0]	REG2DATA_EX,		/* Data from Register Read Port 2 */
	output			REGDST_EX,			/* Destination Register Switch */
	output			REGWRITE_EX,		/* Register Write */
	output			RFE_EX,				/* Return From Exception */
	output			RNL_EX,				/* Right/Left Unaligned Load/Store */
	output [4:0]	RS_EX,				/* TISA Operand Register Number 1 */
	output [4:0]	RT_EX,				/* TISA Operand Register Number 2 */
	/* MA INPUT PORTS */
	input [31:0]	ALURESULT_MA,		/* Data forwarded from Memory Access Phase */
	/* MA OUTPUT PORTS */
	output [31:0]	RDDATA_MA_EX,		/* Data forwarded from Memory Access Phase */
	/* WB INPUT PORTS */
	input [31:0]	WRREGDATA_WB,		/* Data forwarded from Writeback Phase */
	/* WB OUTPUT PORTS */
	output [31:0]	RDDATA_WB_EX,		/* Data forwarded from Writeback Phase */
	/* DF INPUT PORTS */
	input [1:0]		FORWARD_A_DF,		/* Data Forward Switch Operand A */
	input [1:0]		FORWARD_B_DF,		/* Data Forward Switch Operand B */
	/* DF OUTPUT PORTS */
	output [1:0]	FORWARD_A_EX,		/* Data Forward Switch Operand A */
	output [1:0]	FORWARD_B_EX,		/* Data Forward Switch Operand B */
	/* CP0 INPUT PORTS */
	input [31:0]	CP0RDDATA1_CP0,	/* Data from CP0 Register Read Port 1 */
	input				EXC_CP0,				/* Execute Exception Flush */
	/* CP0 OUTPUT PORTS */
	output [31:0]	CP0RDDATA1_EX,		/* Data from CP0 Register Read Port 1 */
	output			EXCE_EX				/* Execute Exception Flush */
	);
	
/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Execute */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_EX = reset_reg;

/*****ID -> EX****************************************************************/

	idtoex id(
		.ALUOP_EX(ALUOP_EX),
		.ALUOP_ID(ALUOP_ID),
		.ALUSRC_EX(ALUSRC_EX),
		.ALUSRC_ID(ALUSRC_ID),
		.BLS_EX(BLS_EX),
		.BLS_ID(BLS_ID),
		.BRANCH_EX(BRANCH_EX),
		.BRANCH_ID(BRANCH_ID),
		.BUBBLE_EX(BUBBLE_EX),
		.BUBBLE_ID(BUBBLE_ID),
		.CLK(CLK),
		.CMOV_EX(CMOV_EX),
		.CMOV_ID(CMOV_ID),
		.COND_EX(COND_EX),
		.COND_ID(COND_ID),
		.CP0_EX(CP0_EX),
		.CP0_ID(CP0_ID),
		.CPREGWRITE_EX(CPREGWRITE_EX),
		.CPREGWRITE_ID(CPREGWRITE_ID),
		.EXCE(EXCE),
		.EXTNOP_EX(EXTNOP_EX),
		.EXTNOP_ID(EXTNOP_ID),
		.HLS_EX(HLS_EX),
		.HLS_ID(HLS_ID),
		.IMM_EX(IMM_EX),
		.IMM_ID(IMM_ID),
		.LC_EX(LC_EX),
		.LC_ID(LC_ID),
		.LINK_EX(LINK_EX),
		.LINK_ID(LINK_ID),
		.LUI_EX(LUI_EX),
		.LUI_ID(LUI_ID),
		.MEMREAD_EX(MEMREAD_EX),
		.MEMREAD_ID(MEMREAD_ID),
		.MEMTOREG_EX(MEMTOREG_EX),
		.MEMTOREG_ID(MEMTOREG_ID),
		.MEMWRITE_EX(MEMWRITE_EX),
		.MEMWRITE_ID(MEMWRITE_ID),
		.MEXT_EX(MEXT_EX),
		.MEXT_ID(MEXT_ID),
		.PC_EX(PC_EX),
		.PC_ID(PC_ID),
		.PC4_EX(PC4_EX),
		.PC4_ID(PC4_ID),
		.PCLK(PCLK),
		.RD_EX(RD_EX),
		.RD_ID(RD_ID),
		.REG1DATA_EX(REG1DATA_EX),
		.REG1DATA_ID(REG1DATA_ID),
		.REG2DATA_EX(REG2DATA_EX),
		.REG2DATA_ID(REG2DATA_ID),
		.REGDST_EX(REGDST_EX),
		.REGDST_ID(REGDST_ID),
		.REGWRITE_EX(REGWRITE_EX),
		.REGWRITE_ID(REGWRITE_ID),
		.RESET(RESET_EX),
		.RFE_EX(RFE_EX),
		.RFE_ID(RFE_ID),
		.RNL_EX(RNL_EX),
		.RNL_ID(RNL_ID),
		.RS_EX(RS_EX),
		.RS_ID(RS_ID),
		.RT_EX(RT_EX),
		.RT_ID(RT_ID),
		.STALL(STALL)
		);

/*****MA -> EX****************************************************************/

	matoex ma(
		.ALURESULT_MA(ALURESULT_MA),
		.RDDATA_MA_EX(RDDATA_MA_EX)
		);

/*****WB -> EX****************************************************************/

	wbtoex wb(
		.RDDATA_WB_EX(RDDATA_WB_EX),
		.WRREGDATA_WB(WRREGDATA_WB)
		);

/*****DF -> EX****************************************************************/

	dftoex df(
		.FORWARD_A_DF(FORWARD_A_DF),
		.FORWARD_A_EX(FORWARD_A_EX),
		.FORWARD_B_DF(FORWARD_B_DF),
		.FORWARD_B_EX(FORWARD_B_EX)
		);

/*****CP0 -> EX****************************************************************/

	cp0toex cp0(
		.CP0RDDATA1_CP0(CP0RDDATA1_CP0),
		.EXC_CP0(EXC_CP0),
		.CP0RDDATA1_EX(CP0RDDATA1_EX),
		.EXCE_EX(EXCE_EX)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****ID -> EX****************************************************************/

module idtoex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [5:0]		ALUOP_ID,			/* ALU Opcode */
	input 			ALUSRC_ID,			/* ALU Operand B Switch */
	input 			BLS_ID,				/* Byte Load/Store */
	input 			BRANCH_ID,			/* Branch */
	input				BUBBLE_ID,			/* Bubble in Execute */
	input 			CLK,					/* System Clock 50 - 100 MHZ */
	input 			CMOV_ID,				/* Conditional Move */
	input 			COND_ID,				/* Conditional Operation */
	input 			CP0_ID,				/* Coprocessor 0 Operation */
	input 			CPREGWRITE_ID,		/* Coprocessor Register Write Enable */
	input				EXCE,					/* Instruction Decode Exception Flush */
	input				EXTNOP_ID,			/* Extension NOP */
	input				HLS_ID,				/* Halfword Load/Store */
	input [31:0]	IMM_ID,				/* Data Immediate */
	input				LC_ID,				/* Load/Store Conditional */
	input 			LINK_ID,				/* Linked Branch */
	input 			LUI_ID,				/* Load Upper Immediate */
	input 			MEMREAD_ID,			/* Memory Read */
	input 			MEMTOREG_ID,		/* Memory to Regiser */
	input 			MEMWRITE_ID,		/* Memory Write */
	input 			MEXT_ID,				/* Sign Extend Memory Data */
	input [31:0]	PC_ID,				/* Currect PC */
	input [31:0]	PC4_ID,				/* Currect PC plus 4*/
	input 			PCLK,					/* Pipeline Clock */
	input [4:0]		RD_ID,				/* Destination Register */
	input [31:0]	REG1DATA_ID,		/* Data from Register Read Port 1 */
	input [31:0]	REG2DATA_ID,		/* Data from Register Read Port 2 */
	input 			REGDST_ID,			/* Destination Register Switch */
	input 			REGWRITE_ID,		/* Register Write */
	input 			RESET,				/* System Reset */
	input				RFE_ID,				/* Return From Exception */
	input 			RNL_ID,				/* Right/Left Unaligned Load/Store */
	input [4:0]		RS_ID,				/* TISA Operand Register Number 1 */
	input [4:0]		RT_ID,				/* TISA Operand Register Number 2 */
	input 			STALL,				/* Pipeline Stall */
	/* OUTPUT PORTS */
	output [5:0]	ALUOP_EX,			/* ALU Opcode */
	output 			ALUSRC_EX,			/* ALU Operand B Switch */
	output 			BLS_EX,				/* Byte Load/Store */
	output 			BRANCH_EX,			/* Branch */
	output 			BUBBLE_EX,			/* Bubble in Execute */
	output 			CMOV_EX,				/* Conditional Move */
	output 			COND_EX,				/* Conditional Operation */
	output 			CP0_EX,				/* Coprocessor 0 Operation */
	output 			CPREGWRITE_EX,		/* Coprocessor Register Write Enable */
	output 			EXTNOP_EX,			/* Extension NOP */
	output 			HLS_EX,				/* Halfword Load/Store */
	output [31:0]	IMM_EX,				/* Data Immediate */
	output 			LC_EX,				/* Load/Store Conditional */
	output 			LINK_EX,				/* Linked Branch */
	output 			LUI_EX,				/* Load Upper Immediate */
	output 			MEMREAD_EX,			/* Memory Read */
	output 			MEMTOREG_EX,		/* Memory to Regiser */
	output 			MEMWRITE_EX,		/* Memory Write */
	output 			MEXT_EX,				/* Sign Extend Memory Data */
	output [31:0]	PC_EX,				/* Currect PC */
	output [31:0]	PC4_EX,				/* Currect PC plus 4*/
	output [4:0]	RD_EX,				/* Destination Register */
	output [31:0]	REG1DATA_EX,		/* Data from Register Read Port 1 */
	output [31:0]	REG2DATA_EX,		/* Data from Register Read Port 2 */
	output 			REGDST_EX,			/* Destination Register Switch */
	output 			REGWRITE_EX,		/* Register Write */
	output 			RFE_EX,				/* Return From Exception */
	output 			RNL_EX,				/* Right/Left Unaligned Load/Store */
	output [4:0]	RS_EX,				/* TISA Operand Register Number 1 */
	output [4:0]	RT_EX					/* TISA Operand Register Number 2 */
	);

/*****Registers****************************************************************/

	reg [201:0]	id_ex;	/* ID -> EX Pipeline Register */
	reg [1:0]	pclkcnt;	/* Pipeline Clock edge detection */

/*****Initialization****************************************************************/

	initial
	begin
		id_ex = {10'b0,1'b1,3'b0,1'b1,187'b0};
		pclkcnt = 2'b0;
	end

/*********************************************************************/

	assign BUBBLE_EX		=	id_ex[201];			//BUBBLE;
	assign EXTNOP_EX		=	id_ex[200];			//EXTNOP_ID;
	assign RFE_EX			=	id_ex[199];			//RFE_ID;
	assign LC_EX			=	id_ex[198];			//LC_ID;
	assign BRANCH_EX		=	id_ex[197];			//BRANCH_ID;
	assign BLS_EX			=	id_ex[196];			//BLS_ID;
	assign HLS_EX			=	id_ex[195];			//HLS_ID;
	assign MEXT_EX			=	id_ex[194];			//MEXT_ID;
	assign RNL_EX 			=	id_ex[193];			//RNL_ID;
	assign LUI_EX			=	id_ex[192];			//LUI_ID;
	assign LINK_EX			=	id_ex[191];			//LINK_ID;
	assign CP0_EX			=	id_ex[190];			//CP0_ID;
	assign CPREGWRITE_EX	=	id_ex[189];			//CPREGWRITE_ID;
	assign COND_EX			=	id_ex[188];			//COND_ID;
	assign CMOV_EX			=	id_ex[187];			//CMOV_ID;
	assign REGWRITE_EX	=	id_ex[186];			//REGWRITE_ID;
	assign MEMTOREG_EX	=	id_ex[185];			//MEMTOREG_ID;
	assign MEMREAD_EX		=	id_ex[184];			//MEMREAD_ID;
	assign MEMWRITE_EX	=	id_ex[183];			//MEMWRITE_ID;
	assign REGDST_EX		=	id_ex[182];			//REGDST_ID;
	assign ALUSRC_EX		=	id_ex[181];			//ALUSRC_ID;
	assign RS_EX			=	id_ex[180:176];	//RS_ID;
	assign RT_EX			=	id_ex[175:171];	//RT_ID;
	assign RD_EX			=	id_ex[170:166];	//RD_ID;
	assign ALUOP_EX		=	id_ex[165:160];	//ALUOP_ID;
	assign PC_EX			=	id_ex[159:128];	//PC_ID;
	assign PC4_EX			=	id_ex[127:96];		//PC4_ID;
	assign IMM_EX			=	id_ex[95:64];		//IMM_ID;
	assign REG1DATA_EX	=	id_ex[63:32];		//REG1DATA_ID;
	assign REG2DATA_EX	=	id_ex[31:0];		//REG2DATA_ID;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		casex({pclkcnt,RESET,EXCE,STALL})
			5'bxx0xx	:	begin
								/* Reset */
								id_ex[201] <= 1'b1;
								id_ex[200:0] <= 
									{9'b0,1'b1,3'b0,1'b1,
									187'b0};
							end
			5'b0111x	:	begin
								/* Exception in Pipeline, Flush */
								id_ex[199:198] <= 2'b0;
								id_ex[196:0] <= 
									{5'b0,1'b1,3'b0,1'b1,
									187'b0};
								id_ex[201] <= 1'b1;
							end
			5'b01100	:	begin
								/* Pipeline Stall, Flush */
								id_ex[199:198] <= 2'b0;
								id_ex[196:0] <= 
									{5'b0,1'b1,3'b0,1'b1,
									187'b0};
								id_ex[201] <= 1'b1;
							end
			5'b01101	:	begin
								/* Clocking the Pipeline */
								id_ex <= {BUBBLE_ID,EXTNOP_ID,RFE_ID,
									LC_ID,BRANCH_ID,BLS_ID,
									HLS_ID,MEXT_ID,RNL_ID,
									LUI_ID,LINK_ID,CP0_ID,
									CPREGWRITE_ID,COND_ID,
									CMOV_ID,REGWRITE_ID,
									MEMTOREG_ID,MEMREAD_ID,
									MEMWRITE_ID,REGDST_ID,
									ALUSRC_ID,RS_ID,RT_ID,
									RD_ID,ALUOP_ID,PC_ID,
									PC4_ID,IMM_ID,
									REG1DATA_ID,
									REG2DATA_ID};
							end
			default	:	begin
							end
		endcase
	end
endmodule

/*****MA -> EX****************************************************************/

module matoex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_MA,	/* Data forwarded from Memory Access Phase */
	/* OUTPUT PORTS */
	output [31:0]	RDDATA_MA_EX	/* Data forwarded from Memory Access Phase */
	);

/*********************************************************************/
	
	assign RDDATA_MA_EX	=	ALURESULT_MA;
	
endmodule

/*****WB -> EX****************************************************************/

module wbtoex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	WRREGDATA_WB,	/* Data forwarded from Writeback Phase */
	/* OUTPUT PORTS */
	output [31:0]	RDDATA_WB_EX	/* Data forwarded from Writeback Phase */
	);

/*********************************************************************/
	
	assign RDDATA_WB_EX	=	WRREGDATA_WB;
	
endmodule

/*****DF -> EX****************************************************************/

module dftoex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [1:0]		FORWARD_A_DF,	/* Data Forward Switch Operand A */
	input [1:0]		FORWARD_B_DF,	/* Data Forward Switch Operand B */
	/* OUTPUT PORTS */
	output [1:0]	FORWARD_A_EX,	/* Data Forward Switch Operand A */
	output [1:0]	FORWARD_B_EX	/* Data Forward Switch Operand B */
	);

/*********************************************************************/

	assign FORWARD_A_EX	=	FORWARD_A_DF;
	assign FORWARD_B_EX	=	FORWARD_B_DF;
	
endmodule

/*****CP0 -> EX****************************************************************/

module cp0toex(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	CP0RDDATA1_CP0,	/* Data from CP0 Register Read Port 1 */
	input 			EXC_CP0,				/* Execute Exception Flush */
	/* OUTPUT PORTS */
	output [31:0]	CP0RDDATA1_EX,		/* Data from CP0 Register Read Port 1 */
	output 			EXCE_EX				/* Execute Exception Flush */
	);

/*********************************************************************/

	assign CP0RDDATA1_EX	=	CP0RDDATA1_CP0;
	assign EXCE_EX			=	EXC_CP0;

endmodule
