/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	toext																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Peripheral Extensions, Bug Fixes, see Manual		*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module toext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,						/* System Clock 50 - 100 MHZ */
	input				PCLK,						/* Pipeline Clock */
	input				RESET,					/* System Reset */
	/* OUTPUT PORTS */
	output			PCLK_EXT,				/* Pipeline Clock */
	output			RESET_EXT,				/* Reset Extension */
	/* PA INPUT PORTS */
	input				EN_PA,					/* Extension Enable */
	input				GR_PA,					/* Extension Grant Pipeline Resources */
	/* PA OUTPUT PORTS */
	output			EN_EXT,					/* Extension Enable */
	output			GR_EXT,					/* Extension Grant Pipeline Resources */
	/* ID INPUT PORTS */
	input [31:0]	INSTR_ID,				/* Current Instruction */
	input [31:0]	PC_ID,					/* Current PC */
	/* ID OUTPUT PORTS */
	output [31:0]	INSTR_EXT,				/* Current Instruction */
	output [31:0]	PC_EXT,					/* Current PC */
	/* EX INPUT PORTS */
	input				EXTNOP_EX,				/* Extension Bubble in Execute Phase */
	/* EX OUTPUT PORTS */
	output			EXTNOP_EX_EXT,			/* Extension Bubble in Execute Phase */
	/* MA INPUT PORTS */
	input				EXTNOP_MA,				/* Extension Bubble in Memory Access Phase */
	/* MA OUTPUT PORTS */
	output			EXTNOP_MA_EXT,			/* Extension Bubble in Memory Access Phase */
	/* RG INPUT PORTS */
	input				EMPTY_RG,				/* Register Write Buffer Empty */
	input				FULL_RG,					/* Register Write Buffer Full */
	input [31:0]	RDREG1DATA_RG,			/* Register Read Port 1 Register Data */
	input [31:0]	RDREG2DATA_RG,			/* Register Read Port 2 Register Data */
	input [31:0]	RDREG3DATA_RG,			/* Register Read Port 3 Register Data */
	input [31:0]	RDREG4DATA_RG,			/* Register Read Port 4 Register Data */
	input				REGRDY_RG,				/* Register Write Buffer Ready */
	input				REGWRITE1_RG,			/* Register Write Port 1 Write Enable Monitor */
	input				REGWRITE2_RG,			/* Register Write Port 2 Write Enable Monitor */
	input [31:0]	WRDATA1_RG,				/* Register Write Port 1 Data Monitor */
	input [31:0]	WRDATA2_RG,				/* Register Write Port 2 Data Monitor */
	input [4:0]		WRREG1_RG,				/* Register Write Port 1 Register Number Monitor */
	input [4:0]		WRREG2_RG,				/* Register Write Port 2 Register Number Monitor */
	/* RG OUTPUT PORTS */
	output			EMPTY_RG_EXT,			/* Register Write Buffer Empty */
	output			FULL_RG_EXT,			/* Register Write Buffer Full */
	output [31:0]	RDREG1DATA_EXT,		/* Register Read Port 1 Register Data */
	output [31:0]	RDREG2DATA_EXT,		/* Register Read Port 2 Register Data */
	output [31:0]	RDREG3DATA_EXT,		/* Register Read Port 3 Register Data */
	output [31:0]	RDREG4DATA_EXT,		/* Register Read Port 4 Register Data */
	output			REGRDY_EXT,				/* Register Write Buffer Ready */
	output			REGWRITE1_RG_EXT,		/* Register Write Port 1 Write Enable Monitor */
	output			REGWRITE2_RG_EXT,		/* Register Write Port 2 Write Enable Monitor */
	output [31:0]	WRDATA1_RG_EXT,		/* Register Write Port 1 Data Monitor */
	output [31:0]	WRDATA2_RG_EXT,		/* Register Write Port 2 Data Monitor */
	output [4:0]	WRREG1_RG_EXT,			/* Register Write Port 1 Register Number Monitor */
	output [4:0]	WRREG2_RG_EXT,			/* Register Write Port 2 Register Number Monitor */
	/* CP0 INPUT PORTS */
	input				ENEXT_CP0,				/* Extension Execution Enable */
	input				EXCEXT_CP0,				/* Extension Exception Flush */
	input				LDEXT_CP0,				/* Extension Loaded */
	input				PER_EXT_IN,				/* Extension Peripheral */
	input				PRIV_EXT_IN,			/* Extension Priveleged */
	input				EXTCLK_EN_IN,
	/* CP0 OUTPUT PORTS */
	output			ENEXT_EXT,				/* Extension Execution Enable */
	output			EXCEXT_EXT,				/* Extension Exception Flush */
	output			LDEXT_EXT,				/* Extension Loaded */
	output			EXTCLK_EN_OUT,			/* */
	/* MEM INPUT PORTS */
	input				BAT_EN_IN,				/* */
	input [31:0]	EXT_MDATA_IN,			/* Memory Data In */
	input				EXT_MDATAVLD_IN,		/* Memory Data Valid */
	input [31:0]	PER_MADDR_IN,			/* Peripheral Memory Addresss */
	input [31:0]	PER_MDATA_IN,			/* Peripheral Memory Data In */
	input				PER_MOE_IN,				/* Peripheral Memory Output Enable */
	input				PER_MWE_IN,				/* Peripheral Memory Write Enable */
	input [31:0]	SNOOP_MADDR_IN,		/* Memory Addresss Monitor */
	input [31:0]	SNOOP_MDATA_IN,		/* Memory Data In Monitor */
	input				SNOOP_MDATAVLD_IN,	/* Memory Data Valid Monitor */
	input				SNOOP_MOE_IN,			/* Memory Output Enable Monitor */
	input				SNOOP_MWE_IN,			/* Memory Write Enable Monitor */
	input				SRT_IN,	/* */
	input [1:0]		BYTES_IN,	/* */
	/* MEM OUTPUT PORTS */
	output			BAT_EN_OUT,				/* */
	output [31:0]	EXT_ADDR_OUT,			/* Memory Addresss */
	output [31:0]	EXT_DATA_OUT,			/* Memory Data */
	output			EXT_DATAVLD_OUT,		/* Memory Data Valid */
	output			EXT_OE_OUT,				/* Memory Output Enable */
	output			EXT_WE_OUT,				/* Memory Write Enable */
	output			SRT_OUT,	/* */
	output [1:0]	BYTES_OUT		/* */
	);

/*****Signals****************************************************************/

	wire ENLDEXT;	/* Extension Loaded and Enabled */
	wire LDDEXT;

/*****Registers****************************************************************/

	reg			ldbit;
	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			perbit;
	reg			reset_reg;	/* Reset Extension */

/*****Initialization****************************************************************/

	initial
	begin
		ldbit = 1'b0;
		pclkcnt = 2'b0;
		perbit = 1'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign ENLDEXT		=	ENEXT_CP0 & LDEXT_CP0 & EXTCLK_EN_IN;
	assign LDDEXT 		=	LDEXT_CP0 & EXTCLK_EN_IN;
	assign PCLK_EXT	=	PCLK;
	assign RESET_EXT	=	reset_reg;

/*****PA -> EXT****************************************************************/

	patoext pa(
		.EN_EXT(EN_EXT),
		.EN_PA(EN_PA),
		.ENLDEXT(ENLDEXT),
		.GR_EXT(GR_EXT),
		.GR_PA(GR_PA)
		);

/*****ID -> EXT****************************************************************/

	idtoext id(
		.ENLDEXT(LDDEXT),
		.INSTR_EXT(INSTR_EXT),
		.INSTR_ID(INSTR_ID),
		.PC_EXT(PC_EXT),
		.PC_ID(PC_ID)
		);

/*****EX -> EXT****************************************************************/

	extoext ex(
		.ENLDEXT(ENLDEXT),
		.EXTNOP_EX(EXTNOP_EX),
		.EXTNOP_EX_EXT(EXTNOP_EX_EXT)
		);

/*****MA -> EXT****************************************************************/

	matoext ma(
		.ENLDEXT(ENLDEXT),
		.EXTNOP_MA(EXTNOP_MA),
		.EXTNOP_MA_EXT(EXTNOP_MA_EXT)
		);

/*****RG -> EXT****************************************************************/

	rgtoext rg(
		.EMPTY_RG(EMPTY_RG),
		.EMPTY_RG_EXT(EMPTY_RG_EXT),
		.ENLDEXT(ENLDEXT),
		.FULL_RG(FULL_RG),
		.FULL_RG_EXT(FULL_RG_EXT),
		.RDREG1DATA_EXT(RDREG1DATA_EXT),
		.RDREG1DATA_RG(RDREG1DATA_RG),
		.RDREG2DATA_EXT(RDREG2DATA_EXT),
		.RDREG2DATA_RG(RDREG2DATA_RG),
		.RDREG3DATA_EXT(RDREG3DATA_EXT),
		.RDREG3DATA_RG(RDREG3DATA_RG),
		.RDREG4DATA_EXT(RDREG4DATA_EXT),
		.RDREG4DATA_RG(RDREG4DATA_RG),
		.REGRDY_EXT(REGRDY_EXT),
		.REGRDY_RG(REGRDY_RG),
		.REGWRITE1_RG(REGWRITE1_RG),
		.REGWRITE1_RG_EXT(REGWRITE1_RG_EXT),
		.REGWRITE2_RG(REGWRITE2_RG),
		.REGWRITE2_RG_EXT(REGWRITE2_RG_EXT),
		.WRDATA1_RG(WRDATA1_RG),
		.WRDATA1_RG_EXT(WRDATA1_RG_EXT),
		.WRDATA2_RG(WRDATA2_RG),
		.WRDATA2_RG_EXT(WRDATA2_RG_EXT),
		.WRREG1_RG(WRREG1_RG),
		.WRREG1_RG_EXT(WRREG1_RG_EXT),
		.WRREG2_RG(WRREG2_RG),
		.WRREG2_RG_EXT(WRREG2_RG_EXT)
		);

/*****CP0 -> EXT****************************************************************/

	cp0toext cp0(
		.ENEXT_CP0(ENEXT_CP0),
		.ENEXT_EXT(ENEXT_EXT),
		.ENLDEXT(ENLDEXT),
		.EXCEXT_CP0(EXCEXT_CP0),
		.EXCEXT_EXT(EXCEXT_EXT),
		.LDEXT_CP0(LDEXT_CP0),
		.LDEXT_EXT(LDEXT_EXT),
		.EXTCLK_EN_IN(EXTCLK_EN_IN),
		.EXTCLK_EN_OUT(EXTCLK_EN_OUT)
		);

/*****MEM -> EXT****************************************************************/

	mem_to_ext memtoext(
		.BAT_EN_IN(BAT_EN_IN),
		.BAT_EN_OUT(BAT_EN_OUT),
		/*.RESET_IN(RESET),*/
		.EXT_ADDR_OUT(EXT_ADDR_OUT),
		.EXT_DATA_OUT(EXT_DATA_OUT),
		.EXT_DATAVLD_OUT(EXT_DATAVLD_OUT),
		.EXT_EN_IN(ENEXT_CP0),
		.EXT_MDATA_IN(EXT_MDATA_IN),
		.EXT_MDATAVLD_IN(EXT_MDATAVLD_IN), 
		.EXT_OE_OUT(EXT_OE_OUT), 
		.EXT_WE_OUT(EXT_WE_OUT),
		.PER_EXT_IN(PER_EXT_IN),
		.PER_MADDR_IN(PER_MADDR_IN),
		.PER_MDATA_IN(PER_MDATA_IN),
		.PER_MOE_IN(PER_MOE_IN), 
		.PER_MWE_IN(PER_MWE_IN),
		.PRIV_EXT_IN(PRIV_EXT_IN),
		.SNOOP_MADDR_IN(SNOOP_MADDR_IN),
		.SNOOP_MDATA_IN(SNOOP_MDATA_IN), 
		.SNOOP_MOE_IN(SNOOP_MOE_IN),
		.SNOOP_MWE_IN(SNOOP_MWE_IN),
		.SNOOP_MDATAVLD_IN(SNOOP_MDATAVLD_IN),
		.SRT_IN(SRT_IN),
		.SRT_OUT(SRT_OUT),
		.BYTES_IN(BYTES_IN),
		.BYTES_OUT(BYTES_OUT)	
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								if (EXTCLK_EN_IN)
								begin
									if (LDEXT_EXT & ~ldbit)
									begin
										ldbit <= 1'b1;
										reset_reg <= RESET;
									end
									else if (~LDEXT_EXT & ldbit & perbit)
									begin
										ldbit <= 1'b0;
										reset_reg <= RESET;
									end
									else if (~LDEXT_EXT & ldbit)
									begin
										ldbit <= 1'b0;
										reset_reg <= 1'b0;
									end
									else
									begin
										reset_reg <= RESET & EXTCLK_EN_IN;	
									end
									if (PER_EXT_IN & ~perbit)
									begin
										perbit <= 1'b1;
									end
									else if (~PER_EXT_IN & perbit & ~LDEXT_EXT)
									begin
										perbit <= 1'b0;
									end
								end
								else
								begin
									perbit <= 1'b0;
									ldbit <= 1'b0;
									reset_reg <= 1'b0;
								end
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****PA -> EXT****************************************************************/

module patoext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		EN_PA,		/* Extension Enable */
	input		ENLDEXT,		/* Extension Loaded and Enabled */
	input		GR_PA,		/* Extension Grant Pipeline Resources */
	/* OUTPUT PORTS */
	output	EN_EXT,		/* Extension Enable */
	output	GR_EXT		/* Extension Grant Pipeline Resources */
	);

/*********************************************************************/
	
	assign EN_EXT	=	(ENLDEXT == 1'b0)?	1'b0:	EN_PA;
	assign GR_EXT	=	(ENLDEXT == 1'b0)?	1'b0:	GR_PA;
	
endmodule

/*****ID -> EXT****************************************************************/

module idtoext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				ENLDEXT,			/* Extension Loaded and Enabled */
	input [31:0]	INSTR_ID,		/* Current Instruction */
	input [31:0]	PC_ID,			/* Current PC */
	/* OUTPUT PORTS */
	output [31:0]	INSTR_EXT,		/* Current Instruction */
	output [31:0]	PC_EXT			/* Current PC */
	);
	
/*********************************************************************/

	assign INSTR_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	INSTR_ID;
	assign PC_EXT		=	(ENLDEXT == 1'b0)?	32'b0:	PC_ID;
	
endmodule

/*****EX -> EXT****************************************************************/

module extoext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		ENLDEXT,			/* Extension Loaded and Enabled */
	input		EXTNOP_EX,		/* Extension Bubble in Execute Phase */
	/* OUTPUT PORTS */
	output	EXTNOP_EX_EXT	/* Extension Bubble in Execute Phase */
	);

/*********************************************************************/

	assign EXTNOP_EX_EXT	=	(ENLDEXT == 1'b0)?	1'b0:	EXTNOP_EX;
	
endmodule

/*****MA -> EXT****************************************************************/

module matoext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		ENLDEXT,			/* Extension Loaded and Enabled */
	input		EXTNOP_MA,		/* Extension Bubble in Memory Access Phase */
	/* OUTPUT PORTS */
	output	EXTNOP_MA_EXT	/* Extension Bubble in Memory Access Phase */
	);
	
/*********************************************************************/

	assign EXTNOP_MA_EXT	=	(ENLDEXT == 1'b0)?	1'b0:	EXTNOP_MA;

endmodule

/*****RG -> EXT****************************************************************/

module rgtoext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				EMPTY_RG,			/* Register Write Buffer Empty */
	input				ENLDEXT,				/* Extension Loaded and Enabled */
	input				FULL_RG,				/* Register Write Buffer Full */
	input [31:0]	RDREG1DATA_RG,		/* Register Read Port 1 Register Data */
	input [31:0]	RDREG2DATA_RG,		/* Register Read Port 2 Register Data */
	input [31:0]	RDREG3DATA_RG,		/* Register Read Port 3 Register Data */
	input [31:0]	RDREG4DATA_RG,		/* Register Read Port 4 Register Data */
	input				REGRDY_RG,			/* Register Write Buffer Ready */
	input				REGWRITE1_RG,		/* Register Write Port 1 Write Enable Monitor */
	input				REGWRITE2_RG,		/* Register Write Port 2 Write Enable Monitor */
	input [31:0]	WRDATA1_RG,			/* Register Write Port 1 Data Monitor */
	input [31:0]	WRDATA2_RG,			/* Register Write Port 2 Data Monitor */
	input [4:0]		WRREG1_RG,			/* Register Write Port 1 Register Number Monitor */
	input [4:0]		WRREG2_RG,			/* Register Write Port 2 Register Number Monitor */
	/* OUTPUT PORTS */
	output			EMPTY_RG_EXT,		/* Register Write Buffer Empty */
	output			FULL_RG_EXT,		/* Register Write Buffer Full */
	output [31:0]	RDREG1DATA_EXT,	/* Register Read Port 1 Register Data */
	output [31:0]	RDREG2DATA_EXT,	/* Register Read Port 2 Register Data */
	output [31:0]	RDREG3DATA_EXT,	/* Register Read Port 3 Register Data */
	output [31:0]	RDREG4DATA_EXT,	/* Register Read Port 4 Register Data */
	output			REGRDY_EXT,			/* Register Write Buffer Ready */
	output			REGWRITE1_RG_EXT,	/* Register Write Port 1 Write Enable Monitor */
	output			REGWRITE2_RG_EXT,	/* Register Write Port 2 Write Enable Monitor */
	output [31:0]	WRDATA1_RG_EXT,	/* Register Write Port 1 Data Monitor */
	output [31:0]	WRDATA2_RG_EXT,	/* Register Write Port 2 Data Monitor */
	output [4:0]	WRREG1_RG_EXT,		/* Register Write Port 1 Register Number Monitor */
	output [4:0]	WRREG2_RG_EXT		/* Register Write Port 2 Register Number Monitor */
	);

/*********************************************************************/
	
	assign EMPTY_RG_EXT		=	(ENLDEXT == 1'b0)?	1'b0:		EMPTY_RG;
	assign FULL_RG_EXT		=	(ENLDEXT == 1'b0)?	1'b0:		FULL_RG;
	assign RDREG1DATA_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	RDREG1DATA_RG;
	assign RDREG2DATA_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	RDREG2DATA_RG;
	assign RDREG3DATA_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	RDREG3DATA_RG;
	assign RDREG4DATA_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	RDREG4DATA_RG;
	assign REGRDY_EXT			=	(ENLDEXT == 1'b0)?	1'b0:		REGRDY_RG;
	assign REGWRITE1_RG_EXT	=	(ENLDEXT == 1'b0)?	1'b0:		REGWRITE1_RG;
	assign REGWRITE2_RG_EXT	=	(ENLDEXT == 1'b0)?	1'b0:		REGWRITE2_RG;
	assign WRDATA1_RG_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	WRDATA1_RG;
	assign WRDATA2_RG_EXT	=	(ENLDEXT == 1'b0)?	32'b0:	WRDATA2_RG;
	assign WRREG1_RG_EXT		=	(ENLDEXT == 1'b0)?	5'b0:		WRREG1_RG;
	assign WRREG2_RG_EXT		=	(ENLDEXT == 1'b0)?	5'b0:		WRREG2_RG;

endmodule

/*****CP0 -> EXT****************************************************************/

module cp0toext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		ENEXT_CP0,		/* Extension Execution Enable */
	input		ENLDEXT,			/* Extension Loaded and Enabled */
	input		EXCEXT_CP0,		/* Extension Exception Flush */
	input		LDEXT_CP0,		/* Extension Loaded */
	input		EXTCLK_EN_IN,	/* */
	/* OUTPUT PORTS */
	output	ENEXT_EXT,		/* Execution Enable */
	output	EXCEXT_EXT,		/* Extension Exception Flush */
	output	LDEXT_EXT,		/* Extension Loaded */
	output	EXTCLK_EN_OUT	/* */
	);

/*********************************************************************/
	
	assign LDEXT_EXT	=	LDEXT_CP0;
	assign ENEXT_EXT	=	ENEXT_CP0;
	assign EXTCLK_EN_OUT =	EXTCLK_EN_IN;
	assign EXCEXT_EXT	=	(ENLDEXT == 1'b0)?	1'b0:	EXCEXT_CP0;

endmodule

/*****MEM -> EXT****************************************************************/

module mem_to_ext(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			BAT_EN_IN,				/* */
	input				EXT_EN_IN,				/* */
	input [31:0]	EXT_MDATA_IN,			/* Memory Data In */
	input				EXT_MDATAVLD_IN,		/* Memory Data Valid */
	input				PER_EXT_IN,				/* Extension Peripheral */
	input [31:0]	PER_MADDR_IN,			/* Peripheral Memory Addresss */
	input [31:0]	PER_MDATA_IN,			/* Peripheral Memory Data In */
	input				PER_MOE_IN,				/* Peripheral Memory Output Enable */
	input				PER_MWE_IN,				/* Peripheral Memory Write Enable */
	input				PRIV_EXT_IN,			/* Extension Priveleged */
	/*input				RESET_IN,				/* Sytem Reset */
	input [31:0]	SNOOP_MADDR_IN,		/* Memory Addresss Monitor */
	input [31:0]	SNOOP_MDATA_IN,		/* Memory Data Monitor */
	input				SNOOP_MDATAVLD_IN,	/* Memory Data Valid Monitor */
	input				SNOOP_MOE_IN,			/* Memory Output Enable Monitor */
	input				SNOOP_MWE_IN,			/* Memory Write Enable Monitor */
	input				SRT_IN,	/* */
	input [1:0]		BYTES_IN,	/* */
	/* OUTPUT PORTS */
	output			BAT_EN_OUT,				/* */
	output [31:0]	EXT_ADDR_OUT,			/* Memory Addresss */
	output [31:0]	EXT_DATA_OUT,			/* Memory Data */
	output			EXT_DATAVLD_OUT,		/* Memory Data Valid */
	output			EXT_OE_OUT,				/* Memory Output Enable */
	output			EXT_WE_OUT,				/* Memory Write Enable */
	output			SRT_OUT,	/* */
	output [1:0] 	BYTES_OUT
	);

/*****Signals****************************************************************/

	wire	W_EXT_PER;	/* */

/*********************************************************************/

	assign BYTES_OUT 			=	(W_EXT_PER)?			BYTES_IN:	2'b0;
	assign SRT_OUT 			=	(W_EXT_PER)?			SRT_IN:		1'b0;
	assign BAT_EN_OUT			=	BAT_EN_IN;
	assign EXT_ADDR_OUT		=	(W_EXT_PER)?			PER_MADDR_IN:
										(PRIV_EXT_IN)?			SNOOP_MADDR_IN:
																	32'b0;							
	assign EXT_OE_OUT			=	(W_EXT_PER)?			PER_MOE_IN:
										(PRIV_EXT_IN)?			SNOOP_MOE_IN:
																	1'b0;
	assign EXT_WE_OUT			=	(W_EXT_PER)?			PER_MWE_IN:
										(PRIV_EXT_IN)?			SNOOP_MWE_IN:
																	1'b0;
	assign EXT_DATAVLD_OUT	=	(PRIV_EXT_IN)?			SNOOP_MDATAVLD_IN:
										(EXT_EN_IN)?			EXT_MDATAVLD_IN:
																	1'b0;
	assign EXT_DATA_OUT		=	(W_EXT_PER)?			PER_MDATA_IN:
										(PRIV_EXT_IN)?			SNOOP_MDATA_IN:
										(EXT_EN_IN)?			EXT_MDATA_IN:
																	32'b0;
	assign W_EXT_PER			=	PER_EXT_IN | BAT_EN_IN;

endmodule
