/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	tohz																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module tohz(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	/* OUTPUT PORTS */
	output			RESET_HZ,			/* Reset Hazard */
	/* PA INPUT PORTS */
	input				EXTSTALL_PA,		/* Stall for Extension */
	/* PA OUTPUT PORTS */
	output			EXTSTALL_HZ,		/* Stall for Extension */
	/* ID INPUT PORTS */
	input				BRANCH_ID,			/* Instruction Decode Branch */
	input				MULT_ID,				/* Instruction Decode Multiplication Unit Operation */
	input [4:0]		RS_ID,				/* Instruction Decode TISA Operand Register Number 1 */
	input [4:0]		RT_ID,				/* Instruction Decode TISA Operand Register Number 2 */
	/* ID OUTPUT PORTS */
	output			BRANCH_ID_HZ,		/* Instruction Decode Branch */
	output			MULT_HZ,				/* Instruction Decode Multiplication Unit Operation */
	output [4:0]	RS_ID_HZ,			/* Instruction Decode TISA Operand Register Number 1 */
	output [4:0]	RT_ID_HZ,			/* Instruction Decode TISA Operand Register Number 2 */
	/* EX INPUT PORTS */
	input				MEMREAD_EX,			/* Execute Memory Read */
	input				MULTRDY_EX,			/* Instruction Decode Multiplication Unit Operation */
	input [4:0]		RTD_EX,				/* Execute TISA Destination Register Number */
	input				LC_EX,
	/* EX OUTPUT PORTS */
	output			MEMREAD_EX_HZ,		/* Execute Memory Read */
	output			MULTRDY_HZ,			/* Instruction Decode Multiplication Unit Operation */
	output [4:0]	RTD_EX_HZ,			/* Execute TISA Destination Register Number */
	output			LC_EX_HZ,
	/* MA INPUT PORTS */
	input				MEMREAD_MA,			/* Memory Access Memory Read */
	input [4:0]		RTD_MA,				/* Memory Access TISA Destination Register Number */
	/* MA OUTPUT PORTS */
	output			MEMREAD_MA_HZ,		/* Memory Access Memory Read */
	output [4:0]	RTD_MA_HZ,			/* Memory Access TISA Destination Register Number */
	/* MEM INPUT PORTS */
	input				IRQ0_MEM,			/* Interrupt */
	/* MEM OUTPUT PORTS */
	output			IRQ0_HZ				/* Interrupt */
	);

/*****Registers****************************************************************/
	
	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Hazard */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_HZ = reset_reg;

/*****PA -> HZ****************************************************************/

	patohz pa(
		.EXTSTALL_HZ(EXTSTALL_HZ),
		.EXTSTALL_PA(EXTSTALL_PA)
		);

/*****ID -> HZ****************************************************************/

	idtohz id(
		.BRANCH_ID(BRANCH_ID),
		.BRANCH_ID_HZ(BRANCH_ID_HZ),
		.MULT_HZ(MULT_HZ),
		.MULT_ID(MULT_ID),
		.RS_ID(RS_ID),
		.RS_ID_HZ(RS_ID_HZ),
		.RT_ID(RT_ID),
		.RT_ID_HZ(RT_ID_HZ)
		);

/*****EX -> HZ****************************************************************/

	extohz ex(
		.MEMREAD_EX(MEMREAD_EX),
		.MEMREAD_EX_HZ(MEMREAD_EX_HZ),
		.MULTRDY_EX(MULTRDY_EX),
		.MULTRDY_HZ(MULTRDY_HZ),
		.RTD_EX(RTD_EX),
		.RTD_EX_HZ(RTD_EX_HZ),
		.LC_EX(LC_EX),
		.LC_EX_HZ(LC_EX_HZ)
		);

/*****MA -> HZ****************************************************************/

	matohz ma(
		.MEMREAD_MA(MEMREAD_MA),
		.MEMREAD_MA_HZ(MEMREAD_MA_HZ),
		.RTD_MA(RTD_MA),
		.RTD_MA_HZ(RTD_MA_HZ)
		);

/*****MEM -> HZ****************************************************************/

	memtohz hz(
		.IRQ0_HZ(IRQ0_HZ),
		.IRQ0_MEM(IRQ0_MEM)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****PA -> HZ****************************************************************/

module patohz(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		EXTSTALL_PA,	/* Stall for Extension */
	/* OUTPUT PORTS */
	output	EXTSTALL_HZ		/* Stall for Extension */
	);
	
/*********************************************************************/

	assign EXTSTALL_HZ	=	EXTSTALL_PA;

endmodule

/*****ID -> HZ****************************************************************/

module idtohz(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			BRANCH_ID,		/* Instruction Decode Branch */
	input 			MULT_ID,			/* Instruction Decode Multiplication Unit Operation */
	input [4:0]		RS_ID,			/* Instruction Decode TISA Operand Register Number 1 */
	input [4:0]		RT_ID,			/* Instruction Decode TISA Operand Register Number 2 */
	/* OUTPUT PORTS */
	output 			BRANCH_ID_HZ,	/* Instruction Decode Branch */
	output 			MULT_HZ,			/* Instruction Decode Multiplication Unit Operation */
	output [4:0]	RS_ID_HZ,		/* Instruction Decode TISA Operand Register Number 1 */
	output [4:0]	RT_ID_HZ			/* Instruction Decode TISA Operand Register Number 2 */
	);
	
/*********************************************************************/

	assign BRANCH_ID_HZ	=	BRANCH_ID;
	assign MULT_HZ			=	MULT_ID;
	assign RS_ID_HZ		=	RS_ID;
	assign RT_ID_HZ		=	RT_ID;

endmodule

/*****EX -> HZ****************************************************************/

module extohz(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				MEMREAD_EX,		/* Execute Memory Read */
	input				MULTRDY_EX,		/* Instruction Decode Multiplication Unit Operation */
	input [4:0]		RTD_EX,			/* Execute TISA Destination Register Number */
	input				LC_EX,
	/* OUTPUT PORTS */
	output			MEMREAD_EX_HZ,	/* Execute Memory Read */
	output			MULTRDY_HZ,		/* Instruction Decode Multiplication Unit Operation */
	output [4:0]	RTD_EX_HZ,		/* Execute TISA Destination Register Number */
	output			LC_EX_HZ
	);

/*********************************************************************/
	
	assign MEMREAD_EX_HZ	=	MEMREAD_EX;
	assign MULTRDY_HZ		=	MULTRDY_EX;
	assign RTD_EX_HZ		=	RTD_EX;
	assign LC_EX_HZ		=	LC_EX;

endmodule

/*****MA -> HZ****************************************************************/

module matohz(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				MEMREAD_MA,		/* Memory Access Memory Read */
	input [4:0]		RTD_MA,			/* Memory Access TISA Destination Register Number */
	/* OUTPUT PORTS */
	output			MEMREAD_MA_HZ,	/* Memory Access Memory Read */
	output [4:0]	RTD_MA_HZ		/* Memory Access TISA Destination Register Number */
	);

/*********************************************************************/

	assign MEMREAD_MA_HZ	=	MEMREAD_MA;
	assign RTD_MA_HZ		=	RTD_MA;

endmodule

/*****MEM -> HZ****************************************************************/

module memtohz(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		IRQ0_MEM,		/* Interrupt */
	/* OUTPUT PORTS */
	output	IRQ0_HZ			/* Interrupt */
	);
	
/*********************************************************************/

	assign IRQ0_HZ	=	IRQ0_MEM;

endmodule
