/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	toid																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module toid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input 			EXCE,				/* Instruction Fetch Exception Flush */
	input 			FLUSH,			/* Instruction Fetch Flush */
	input 			PCLK,				/* Pipeline Clock */
	input 			RESET,			/* System Reset */
	input 			STALL,			/* Pipeline Stall */
	/* OUTPUT PORTS */
	output 			RESET_ID,		/* Reset Instruction Decode */
	/* PA INPUT PORTS */
	input 			EN_EXT0_PA,		/* Extension 0 Enable */
	input 			EN_EXT1_PA,		/* Extension 1 Enable */
	input 			EN_ID_PA,		/* TISA Enable */
	input 			GR_EXT0_PA,		/* Extension 0 Grant Pipeline Resources */
	input 			GR_EXT1_PA,		/* Extension 1 Grant Pipeline Resources */
	input 			RI_PA,			/* Pipeline Arbiter Reserved/Recognized Instruction */
	/* PA OUTPUT PORTS */
	output 			EN_EXT_ID,		/* Extension Enable */
	output 			EN_ID,			/* TISA Enable */
	output 			RI_PA_ID,		/* Pipeline Arbiter Reserved/Recognized Instruction */
	/* IF INPUT PORTS */
	input [31:0]	INSTR_IF,		/* Instruction Decode Instruction */
	input [31:0]	PC_IF,			/* Instruction Decode Current PC */
	input [31:0]	PC4_IF,			/* Instruction Decode Current PC plus 4 */
	/* IF OUTPUT PORTS */
	output 			BUBBLE_ID,		/* Bubble in Instruction Decode */
	output [31:0]	INSTR_ID,		/* Instruction Decode Instruction */
	output [31:0]	PC_ID,			/* Instruction Decode Current PC */
	output [31:0]	PC4_ID,			/* Instruction Decode Current PC plus 4 */
	/* MA INPUT PORTS */
	input [31:0]	ALURESULT_MA,	/* Data forwarded from Memory Access Phase */
	/* MA OUTPUT PORTS */
	output [31:0]	RDDATA_MA_ID,	/* Data forwarded from Memory Access Phase */
	/* WB INPUT PORTS */
	input [31:0]	WRREGDATA_WB,	/* Data forwarded from Writeback Phase */
	/* WB OUTPUT PORTS */
	output [31:0]	RDDATA_WB_ID,	/* Data forwarded from Writeback Phase */
	/* EXT0 INPUT PORTS */
	input				PCNEXT_EXT0,	/* Extension Conditional PC Update */
	/* EXT1 INPUT PORTS */
	input				PCNEXT_EXT1,	/* Extension Conditional PC Update */
	/* RG INPUT PORTS */
	input [31:0]	REG1DATA_RG,	/* Data from Register Read Port 1 */
	input [31:0]	REG2DATA_RG,	/* Data from Register Read Port 2 */
	/* RG OUTPUT PORTS */
	output [31:0]	REG1DATA_ID,	/* Data from Register Read Port 1 */
	output [31:0]	REG2DATA_ID,	/* Data from Register Read Port 2 */
	/* DF INPUT PORTS */
	input [1:0]		FORWARD_A_DF,	/* Instruction Decode Data Forward Switch Operand A */
	input [1:0]		FORWARD_B_DF,	/* Instruction Decode Data Forward Switch Operand B */
	/* DF OUTPUT PORTS */
	output [1:0]	FORWARD_A_ID,	/* Instruction Decode Data Forward Switch Operand A */
	output [1:0]	FORWARD_B_ID,	/* Instruction Decode Data Forward Switch Operand B */
	/* HZ INPUT PORTS */
	input				STALL_HZ,		/* Pipeline Stall */
	/* HZ OUTPUT PORTS */
	output			STALL_ID,		/* Pipeline Stall */
	/* CP0 INPUT PORTS */
	input [3:0]		CP_CP0,			/* Enabled Coprocessors */
	input				EXC_CP0,			/* Instruction Decode Exception Flush */
	input 			KU_CP0,
	/* CP0 OUTPUT PORTS */
	output [3:0]	CP_ID,			/* Enabled Coprocessors */
	output 			EXCE_ID,			/* Instruction Decode Exception Flush */
	output 			KU_ID
	);

/*****Signals****************************************************************/

	wire EXTFLUSH;	/* Flush for Extension */
	wire GR;			/* Extension Grant Pipeline Resources */
	wire PCNEXT;	/* Extension Conditional PC Update */

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Instruction Decode */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign EXTFLUSH	=	GR	&	PCNEXT;
	assign GR			=	GR_EXT0_PA	|	GR_EXT1_PA;
	assign PCNEXT		=	PCNEXT_EXT0	|	PCNEXT_EXT1;
	assign RESET_ID	=	reset_reg;

/*****IF -> ID****************************************************************/

	iftoid i_f(
		.CLK(CLK),
		.BUBBLE_ID(BUBBLE_ID),
		.EXCE(EXCE),
		.EXTFLUSH(EXTFLUSH),
		.FLUSH(FLUSH),
		.INSTR_ID(INSTR_ID),
		.INSTR_IF(INSTR_IF),
		.PC_ID(PC_ID),
		.PC_IF(PC_IF),
		.PC4_ID(PC4_ID),
		.PC4_IF(PC4_IF),
		.PCLK(PCLK),
		.RESET(RESET_ID),
		.STALL(STALL)
		);

/*****MA -> ID****************************************************************/

	matoid ma(
		.ALURESULT_MA(ALURESULT_MA),
		.RDDATA_MA_ID(RDDATA_MA_ID)
		);

/*****WB -> ID****************************************************************/

	wbtoid wb(
		.WRREGDATA_WB(WRREGDATA_WB),
		.RDDATA_WB_ID(RDDATA_WB_ID)
		);

/*****RG -> ID****************************************************************/

	rgtoid rg(
		.REG1DATA_ID(REG1DATA_ID),
		.REG1DATA_RG(REG1DATA_RG),
		.REG2DATA_ID(REG2DATA_ID),
		.REG2DATA_RG(REG2DATA_RG)
		);

/*****DF -> ID****************************************************************/

	dftoid df(
		.FORWARD_A_DF(FORWARD_A_DF),
		.FORWARD_A_ID(FORWARD_A_ID),
		.FORWARD_B_DF(FORWARD_B_DF),
		.FORWARD_B_ID(FORWARD_B_ID)
		);

/*****HZ -> ID****************************************************************/

	hztoid hz(
		.STALL_HZ(STALL_HZ),
		.STALL_ID(STALL_ID)
		);

/*****CP0 -> ID****************************************************************/

	cp0toid cp0(
		.CP_CP0(CP_CP0),
		.EXC_CP0(EXC_CP0),
		.KU_CP0(KU_CP0),
		.CP_ID(CP_ID),
		.EXCE_ID(EXCE_ID),
		.KU_ID(KU_ID)
		);

/*****PA -> ID****************************************************************/

	patoid pa(
		.EN_EXT_ID(EN_EXT_ID),
		.EN_EXT0_PA(EN_EXT0_PA),
		.EN_EXT1_PA(EN_EXT1_PA),
		.EN_ID(EN_ID),
		.EN_ID_PA(EN_ID_PA),
		.RI_PA(RI_PA),
		.RI_PA_ID(RI_PA_ID)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****IF -> ID****************************************************************/

module iftoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,			/* System Clock 50 - 100 MHZ */
	input 			EXCE,			/* Instruction Fetch Exception Flush */
	input 			EXTFLUSH,	/* Flush for Extension */
	input 			FLUSH,		/* Instruction Fetch Flush */
	input [31:0]	INSTR_IF,	/* Instruction Decode Instruction */
	input [31:0] 	PC_IF,		/* Instruction Decode Current PC */
	input [31:0]	PC4_IF,		/* Instruction Decode Current PC plus 4 */
	input 			PCLK,			/* Pipeline Clock */
	input 			RESET,		/* System Reset */
	input 			STALL,		/* Pipeline Stall */
	/* OUTPUT PORTS */
	output			BUBBLE_ID,	/* Bubble in Instruction Decode */
	output [31:0]	INSTR_ID,	/* Instruction Decode Instruction */
	output [31:0]	PC_ID,		/* Instruction Decode Current PC */
	output [31:0]	PC4_ID		/* Instruction Decode Current PC plus 4 */
	);

/*****Signals****************************************************************/

	wire	IFIDWRITE;

/*****Registers****************************************************************/

	reg [96:0]	if_id;	/* IF -> ID Pipeline Register */
	reg [1:0]	pclkcnt;	/* Pipeline Clock edge detection */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		if_id = 97'b0;
	end

/*********************************************************************/

	assign IFIDWRITE	=	STALL;
	assign BUBBLE_ID	=	if_id[96];		//BUBBLE;
	assign INSTR_ID	=	if_id[95:64];	//INSTR_IF;
	assign PC_ID		=	if_id[63:32];	//PC_IF
	assign PC4_ID		=	if_id[31:0];	//PC4_IF;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		casex({pclkcnt,RESET,IFIDWRITE,FLUSH,EXCE,EXTFLUSH})
			7'bxx0xxxx	:	begin
									/* Reset */
									if_id[95:0] <= 96'b0;
									if_id[96] <= 1'b1;
								end
			7'b011xx1x	:	begin
									/* Exception in Pipeline, Flush */
									if_id[95:0] <= 96'b0;
									if_id[96] <= 1'b1;
								end
			7'b011xx01	:	begin
									/* Extension Executing, Flush */
									if_id[95:0] <= 96'b0;
									if_id[96] <= 1'b1;							
								end
			7'b0111100	:	begin
									/* Control Flush */
									if_id[95:0] <= 96'b0;
									if_id[96] <= 1'b1;
								end
			7'b0111000	:	begin
									/* Clocking the Pipeline */
									if_id[95:0] <= 
									{INSTR_IF,PC_IF,PC4_IF};
									if_id[96] <= 1'b0;
								end
			default		:	begin
								end
		endcase
	end

endmodule

/*****MA -> ID****************************************************************/

module matoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_MA,	/* Data forwarded from Memory Access Phase */
	/* OUTPUT PORTS */
	output [31:0]	RDDATA_MA_ID	/* Data forwarded from Memory Access Phase */
	);

/*********************************************************************/

	assign RDDATA_MA_ID	=	ALURESULT_MA;
	
endmodule

/*****WB -> ID****************************************************************/

module wbtoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	WRREGDATA_WB,	/* Data forwarded from Writeback Phase */
	/* OUTPUT PORTS */
	output [31:0]	RDDATA_WB_ID	/* Data forwarded from Writeback Phase */
	);

/*********************************************************************/

	assign RDDATA_WB_ID	=	WRREGDATA_WB;

endmodule

/*****RG -> ID****************************************************************/

module rgtoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	REG1DATA_RG,	/* Data from Register Read Port 1 */
	input [31:0]	REG2DATA_RG,	/* Data from Register Read Port 2 */
	/* OUTPUT PORTS */
	output [31:0]	REG1DATA_ID,	/* Data from Register Read Port 1 */
	output [31:0]	REG2DATA_ID		/* Data from Register Read Port 2 */
	);

/*********************************************************************/

	assign REG1DATA_ID	=	REG1DATA_RG;
	assign REG2DATA_ID	=	REG2DATA_RG;
	
endmodule

/*****DF -> ID****************************************************************/

module dftoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [1:0]		FORWARD_A_DF,	/* Instruction Decode Data Forward Switch Operand A */
	input [1:0]		FORWARD_B_DF,	/* Instruction Decode Data Forward Switch Operand B */
	/* OUTPUT PORTS */
	output [1:0]	FORWARD_A_ID,	/* Instruction Decode Data Forward Switch Operand A */
	output [1:0]	FORWARD_B_ID	/* Instruction Decode Data Forward Switch Operand B */
	);

/*********************************************************************/

	assign FORWARD_A_ID	=	FORWARD_A_DF;
	assign FORWARD_B_ID	=	FORWARD_B_DF;

endmodule

/*****HZ -> ID****************************************************************/

module hztoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		STALL_HZ,	/* Pipeline Stall */
	/* OUTPUT PORTS */
	output	STALL_ID		/* Pipeline Stall */
	);

/*********************************************************************/

	assign STALL_ID	=	STALL_HZ;

endmodule

/*****CP0 -> ID****************************************************************/

module cp0toid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [3:0]		CP_CP0,		/* Enabled Coprocessors */
	input 			EXC_CP0,		/* Instruction Decode Exception Flush */
	input 			KU_CP0,		/* Kernel/User Mode */
	/* OUTPUT PORTS */
	output [3:0]	CP_ID,		/* Enabled Coprocessors */
	output 			EXCE_ID,		/* Instruction Decode Exception Flush */
	output 			KU_ID			/* Kernel/User Mode */
	);

/*********************************************************************/

	assign CP_ID	=	CP_CP0;
	assign EXCE_ID	=	EXC_CP0;
	assign KU_ID	=	KU_CP0;

endmodule

/*****PA -> ID****************************************************************/

module patoid(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		EN_EXT0_PA,		/* Extension 0 Enable */
	input		EN_EXT1_PA,		/* Extension 1 Enable */
	input		EN_ID_PA,		/* TISA Enable */
	input		RI_PA,			/* Pipeline Arbiter Reserved/Recognized Instruction */
	/* OUTPUT PORTS */
	output	EN_EXT_ID,		/* Extension Enable */
	output	EN_ID,			/* TISA Enable */
	output	RI_PA_ID			/* Pipeline Arbiter Reserved/Recognized Instruction */
	);
	
/*********************************************************************/

	assign EN_EXT_ID	=	EN_EXT0_PA	|	EN_EXT1_PA;
	assign EN_ID		=	EN_ID_PA;
	assign RI_PA_ID	=	RI_PA;
	
endmodule
