/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	toif																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module toif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input 			PCLK,				/* Pipeline Clock */
	input 			RESET,			/* System Reset */
	/* OUTPUT PORTS */
	output 			RESET_IF,		/* Reset Instruction Fetch */
	/* PA INPUT PORTS */
	input 			EN_EXT0_PA,		/* Extension 0 Enable */
	input 			EN_EXT1_PA,		/* Extension 1 Enable */
	input 			EN_ID_PA,		/* TISA Enable */
	input 			GR_EXT0_PA,		/* Extension 0 Grant Pipeline Resources */
	input 			GR_EXT1_PA,		/* Extension 1 Grant Pipeline Resources */
	/* PA OUTPUT PORTS */
	output 			EN_EXT_IF,		/* Extension Enable */
	output 			EN_ID_IF,		/* TISA Enable */
	output 			GR_EXT_IF,		/* Extension Grant Pipeline Resources */
	/* ID INPUT PORTS */
	input 			FLUSH_ID,		/* Instruction Fetch Flush */
	input [31:0]	JMPADD_ID,		/* Branch Jump Address */
	input 			PCSRC_ID,		/* Branch Address Selection Switch */
	/* ID OUTPUT PORTS */
	output 			FLUSH_IF,		/* Instruction Fetch Flush */
	output [31:0]	JMPADD_IF,		/* Branch Jump Address */
	output 			PCSRC_IF,		/* Branch Address Selection Switch */
	/* EXT0 INPUT PORTS */
	input [31:0]	EXTADD_EXT0,	/* Extension 0 Jump Address */
	input 			PCNEXT_EXT0,	/* Extension 0 Conditional PC Update */
	/* EXT1 INPUT PORTS */
	input [31:0]	EXTADD_EXT1,	/* Extension 1 Jump Address */
	input				PCNEXT_EXT1,	/* Extension 1 Conditional PC Update */
	/* EXT OUTPUT PORTS */
	output [31:0]	EXTADD_EXT_IF,	/* Extension Jump Address */
	output			PCNEXT_EXT_IF,	/* Extension Conditional PC Update */
	/* MEM INPUT PORTS */
	input [31:0]	INSTR_MEM,		/* Instruction Fetch Data from Memory */
	/* MEM OUTPUT PORTS */
	output [31:0]	INSTR_IF,		/* Instruction Fetch Data from Memory */
	/* HZ INPUT PORTS */
	input				STALL_HZ,		/* Pipeline Stall */
	/* HZ OUTPUT PORTS */
	output			STALL_IF,		/* Pipeline Stall */
	/* CP0 INPUT PORTS */
	input				EXC_CP0,			/* Instruction Fetch Exception */
	input	[31:0]	EXCADDR_CP0,	/* Exception Jump Address */
	input 			KU_CP0,			/* Kernel/User Mode */
	input 			RE_CP0,			/* Reverse Endian */
	/* CP0 OUTPUT PORTS */
	output 			EXC_IF,			/* Instruction Fetch Exception */
	output [31:0]	EXCADDR_IF,		/* Exception Jump Address */
	output 			KU_IF,			/* Kernel/User Mode */
	output 			RE_IF				/* Reverse Endian */
	);
	
/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Instruction Fetch */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_IF = reset_reg;

/*****PA -> IF****************************************************************/

	patoif pa(
		.EN_EXT_IF(EN_EXT_IF),
		.EN_EXT0_PA(EN_EXT0_PA),
		.EN_EXT1_PA(EN_EXT1_PA),
		.EN_ID_IF(EN_ID_IF),
		.EN_ID_PA(EN_ID_PA),
		.EXTADD_EXT_IF(EXTADD_EXT_IF),
		.GR_EXT_IF(GR_EXT_IF),
		.GR_EXT0_PA(GR_EXT0_PA),
		.GR_EXT1_PA(GR_EXT1_PA),
		.PCNEXT_EXT_IF(PCNEXT_EXT_IF)
		);

/*****ID -> IF****************************************************************/

	idtoif id(
		.FLUSH_ID(FLUSH_ID),
		.FLUSH_IF(FLUSH_IF),
		.JMPADD_ID(JMPADD_ID),
		.JMPADD_IF(JMPADD_IF),
		.PCSRC_ID(PCSRC_ID),
		.PCSRC_IF(PCSRC_IF)
		);

/*****EXT0 -> IF****************************************************************/

	exttoif ext0(
		.EN_EXT_PA(EN_EXT0_PA),
		.EXTADD_EXT(EXTADD_EXT0),
		.EXTADD_EXT_IF(EXTADD_EXT_IF),
		.GR_EXT_PA(GR_EXT0_PA),
		.PCNEXT_EXT(PCNEXT_EXT0),
		.PCNEXT_EXT_IF(PCNEXT_EXT_IF)
		);

/*****EXT1 -> IF****************************************************************/

	exttoif ext1(
		.EN_EXT_PA(EN_EXT1_PA),
		.GR_EXT_PA(GR_EXT1_PA),
		.EXTADD_EXT(EXTADD_EXT1),
		.EXTADD_EXT_IF(EXTADD_EXT_IF),
		.PCNEXT_EXT(PCNEXT_EXT1),
		.PCNEXT_EXT_IF(PCNEXT_EXT_IF)
		);

/*****MEM -> IF****************************************************************/

	memtoif mem(
		.INSTR_IF(INSTR_IF),
		.INSTR_MEM(INSTR_MEM)
		);

/*****HZ -> IF****************************************************************/

	hztoif hz(
		.STALL_HZ(STALL_HZ),
		.STALL_IF(STALL_IF)
		);

/*****CP0 -> IF****************************************************************/

	cp0toif cp0(
		.EXC_CP0(EXC_CP0),
		.EXC_IF(EXC_IF),
		.EXCADDR_CP0(EXCADDR_CP0),
		.EXCADDR_IF(EXCADDR_IF),
		.KU_CP0(KU_CP0),
		.KU_IF(KU_IF),
		.RE_CP0(RE_CP0),
		.RE_IF(RE_IF)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****PA -> IF****************************************************************/

module patoif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			EN_EXT0_PA,		/* Extension 0 Enable */
	input 			EN_EXT1_PA,		/* Extension 1 Enable */
	input 			EN_ID_PA,		/* TISA Enable */
	input 			GR_EXT0_PA,		/* Extension 0 Grant Pipeline Resources */
	input 			GR_EXT1_PA,		/* Extension 1 Grant Pipeline Resources */
	/* OUTPUT PORTS */
	output 			EN_EXT_IF,		/* Extension Enable */
	output 			EN_ID_IF,		/* TISA Enable */
	output [31:0]	EXTADD_EXT_IF,	/* Extension Jump Address */
	output 			GR_EXT_IF,		/* Extension Grant Pipeline Resources */
	output 			PCNEXT_EXT_IF	/* Extension Conditional PC Update */
	);

/*********************************************************************/

	assign EN_ID_IF 		=	EN_ID_PA;
	assign EN_EXT_IF		=	EN_EXT0_PA	|	EN_EXT1_PA;
	assign GR_EXT_IF		=	GR_EXT0_PA	|	GR_EXT1_PA;
	assign PCNEXT_EXT_IF	=	(GR_EXT_IF)?	1'bz:
														1'b0;
	assign EXTADD_EXT_IF =	(EN_EXT_IF | GR_EXT_IF)?	32'bz:
																		32'b0;

endmodule

/*****ID -> IF****************************************************************/

module idtoif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				FLUSH_ID,	/* Instruction Fetch Flush */
	input [31:0]	JMPADD_ID,	/* Branch Jump Address */
	input 			PCSRC_ID, 	/* Branch Address Selection Switch */
	/* OUTPUT PORTS */
	output 			FLUSH_IF,	/* Instruction Fetch Flush */
	output [31:0]	JMPADD_IF,	/* Branch Jump Address */
	output 			PCSRC_IF		/* Branch Address Selection Switch */
	);

/*********************************************************************/

	assign FLUSH_IF 	= 	FLUSH_ID;
	assign JMPADD_IF 	=	JMPADD_ID;
	assign PCSRC_IF 	=	PCSRC_ID;

endmodule

/*****EXT -> IF****************************************************************/

module exttoif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			EN_EXT_PA,		/* Extension Enable */
	input [31:0]	EXTADD_EXT,		/* Extension Jump Address */
	input 			GR_EXT_PA,		/* Extension Grant Pipeline Resources */
	input 			PCNEXT_EXT,		/* Extension Conditional PC Update */
	/* OUTPUT PORTS */
	output [31:0]	EXTADD_EXT_IF,	/* Extension Jump Address */
	output			PCNEXT_EXT_IF	/* Extension Conditional PC Update */
	);

/*********************************************************************/

	assign EXTADD_EXT_IF		=	(~EN_EXT_PA & ~GR_EXT_PA)?	32'bz:
																			EXTADD_EXT;
	assign PCNEXT_EXT_IF		=	(~GR_EXT_PA)?	1'bz:
															PCNEXT_EXT;
															
endmodule

/*****MEM -> IF****************************************************************/

module memtoif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	INSTR_MEM,	/* Instruction Fetch Data from Memory */
	/* OUTPUT PORTS */
	output [31:0]	INSTR_IF		/* Instruction Fetch Data from Memory */
	);

/*********************************************************************/

	assign INSTR_IF	=	INSTR_MEM;

endmodule

/*****HZ -> IF****************************************************************/

module hztoif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		STALL_HZ,	/* Pipeline Stall */
	/* OUTPUT PORTS */
	output	STALL_IF		/* Pipeline Stall */
	);

/*********************************************************************/

	assign STALL_IF	=	STALL_HZ;

endmodule

/*****CP0 -> IF****************************************************************/

module cp0toif(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				EXC_CP0,			/* Instruction Fetch Exception */
	input [31:0]	EXCADDR_CP0,	/* Exception Jump Address */
	input 			KU_CP0,			/* Kernel/User Mode */
	input 			RE_CP0,			/* Reverse Endian */
	/* OUTPUT PORTS */
	output 			EXC_IF,			/* Instruction Fetch Exception */
	output [31:0]	EXCADDR_IF,		/* Exception Jump Address */
	output 			KU_IF,			/* Kernel/User Mode */
	output 			RE_IF				/* Reverse Endian */
	);

/*********************************************************************/

	assign EXC_IF		=	EXC_CP0;
	assign EXCADDR_IF	=	EXCADDR_CP0;
	assign KU_IF		=	KU_CP0;
	assign RE_IF		=	RE_CP0;

endmodule
