/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	toma																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module toma(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				EXCE,					/* Execute Exception Flush */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	/* OUTPUT PORTS */
	output			RESET_MA,			/* Reset Memory Access */
	/* PA INPUT PORTS */
	input				GR_EXT0_PA,			/* Extension 0 Grant Pipeline Resources */
	input				GR_EXT1_PA,			/* Extension 1 Grant Pipeline Resources */
	/* EX INPUT PORTS */
	input [31:0]	ALURESULT_EX,		/* ALU Result */
	input				BLS_EX,				/* Byte Load/Store */
	input				BRANCH_EX,			/* Branch */
	input				BUBBLE_EX,			/* Bubble in Memory Access */
	input				EXTNOP_EX,			/* Extension NOP */
	input				HLS_EX,				/* Halfword Load/Store */
	input				LC_EX,				/* Load/Store Conditional */
	input				MEMREAD_EX,			/* Memory Read */
	input				MEMTOREG_EX,		/* Memory to Regiser */
	input				MEMWRITE_EX,		/* Memory Write */
	input				MEXT_EX,				/* Sign Extend Memory Data */
	input	[31:0]	PC_EX,				/* Currect PC */
	input [31:0]	REG2DATA_EX,		/* Data from Register Read Port 2 */
	input				REGWRITE_EX,		/* Register Write */
	input				RNL_EX,				/* Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EX,			/* TISA Destination Register Number */
	/* EX OUTPUT PORTS */
	output [31:0]	ALURESULT_MA,		/* ALU Result */
	output 			BLS_MA,				/* Byte Load/Store */
	output 			BRANCH_MA,			/* Branch */
	output 			BUBBLE_MA,			/* Bubble in Memory Access */
	output 			EXTNOP_MA,			/* Extension NOP */
	output 			HLS_MA,				/* Halfword Load/Store */
	output 			LC_MA,				/* Load/Store Conditional */
	output 			MEMREAD_MA,			/* Memory Read */
	output 			MEMTOREG_MA,		/* Memory to Regiser */
	output 			MEMWRITE_MA,		/* Memory Write */
	output 			MEXT_MA,				/* Sign Extend Memory Data */
	output [31:0]	PC_MA,				/* Currect PC */
	output [31:0]	REG2DATA_MA,		/* Data from Register Read Port 2 */
	output 			REGWRITE_MA,		/* Memory Access Register Write */
	output 			RNL_MA,				/* Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_MA,			/* TISA Destination Register Number */
	/* EXT0 INPUT PORTS */
	input [31:0]	ALURESULT_EXT0,	/* ALU Result */
	input				BLS_EXT0,			/* Byte Load/Store */
	input				BRANCH_EXT0,		/* Branch */
	input				BUBBLE_EXT0,		/* Bubble in Memory Access */
	input				EXTNOP_EXT0,		/* Extension NOP */
	input				HLS_EXT0,			/* Halfword Load/Store */
	input				LC_EXT0,				/* Load/Store Conditional */
	input				MEMREAD_EXT0,		/* Memory Read */
	input				MEMTOREG_EXT0,		/* Memory to Regiser */
	input				MEMWRITE_EXT0,		/* Memory Write */
	input				MEXT_EXT0,			/* Sign Extend Memory Data */
	input [31:0]	PC_EXT0,				/* Currect PC */
	input [31:0]	REG2DATA_EXT0,		/* Data from Register Read Port 2 */
	input 			REGWRITE_EXT0,		/* Register Write */
	input 			REMA_EXT0,			/* Re-enter at Memory Access */
	input 			RNL_EXT0,			/* Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EXT0,			/* TISA Destination Register Number */
	/* EXT1 INPUT PORTS */
	input [31:0]	ALURESULT_EXT1,	/* ALU Result */
	input 			BLS_EXT1,			/* Byte Load/Store */
	input 			BRANCH_EXT1,		/* Branch */
	input 			BUBBLE_EXT1,		/* Bubble in Memory Access */
	input 			EXTNOP_EXT1,		/* Extension NOP */
	input 			HLS_EXT1,			/* Halfword Load/Store */
	input 			LC_EXT1,				/* Load/Store Conditional */
	input 			MEMREAD_EXT1,		/* Memory Read */
	input 			MEMTOREG_EXT1,		/* Memory to Regiser */
	input 			MEMWRITE_EXT1,		/* Memory Write */
	input 			MEXT_EXT1,			/* Sign Extend Memory Data */
	input [31:0]	PC_EXT1,				/* Currect PC */
	input [31:0]	REG2DATA_EXT1,		/* Data from Register Read Port 2 */
	input 			REGWRITE_EXT1,		/* Register Write */
	input				REMA_EXT1,			/* Re-enter at Memory Access */
	input 			RNL_EXT1,			/* Right/Left Unaligned Load/Store */
	input [4:0] 	WRREG_EXT1,			/* TISA Destination Register Number */
	/* MEM INPUT PORTS */
	input [31:0]	DMDATAOUT_MEM,		/* Memory Data Out from Memory */
	input				ALS_MEM,
	/* MEM OUTPUT PORTS */
	output [31:0]	DMDATAOUT_MA,		/* Memory Data Out from Memory */
	output			ALS_MA,
	/* CP0 INPUT PORTS */
	input				EXC_CP0,				/* Memory Access Exception Flush */
	input				RE_CP0,				/* Reverse Endian */
	/* CP0 OUTPUT PORTS */
	output			EXCE_MA,				/* Memory Access Exception Flush */
	output			RE_MA					/* Reverse Endian */
	);

/*****Signals****************************************************************/

	wire [31:0]	ALURESULT;		/* ALU Result */
	wire			BLS;				/* Byte Load/Store */
	wire			BRANCH;			/* Branch */
	wire			BUBBLE;			/* Bubble in Memory Access */
	wire			EXTNOP;			/* Extension NOP */
	wire			HLS;				/* Halfword Load/Store */
	wire			GR_EX;			/* Extension Grant Pipeline Resources */
	wire			LC;				/* Load/Store Conditional */
	wire			MEMREAD;			/* Memory Read */
	wire			MEMTOREG;		/* Memory to Regiser */
	wire			MEMWRITE;		/* Memory Write */
	wire			MEXT;				/* Sign Extend Memory Data */
	wire [31:0]	PC;				/* Currect PC */
	wire [31:0]	REG2DATA;		/* Data from Register Read Port 2 */
	wire			REGWRITE;		/* Register Write */
	wire			REMA_EX;			/* Re-enter at Memory Access */
	wire			RNL;				/* Right/Left Unaligned Load/Store */
	wire [4:0]	WRREG;			/* TISA Destination Register Number */

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Memory Access */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_MA = reset_reg;

/*********************************************************************/

	or2_1 or0(
		.A(GR_EXT0_PA),
		.B(GR_EXT1_PA),
		.C(GR_EX)
		);

	or2_1 or1(
		.A(REMA_EXT0),
		.B(REMA_EXT1),
		.C(REMA_EX)
		);

/*****EX -> MA_REG****************************************************************/

	extoma ex(
		.ALURESULT_EX(ALURESULT_EX),
		.ALURESULT_MA(ALURESULT),
		.BLS_EX(BLS_EX),
		.BLS_MA(BLS),
		.BRANCH_EX(BRANCH_EX),
		.BRANCH_MA(BRANCH),
		.BUBBLE_EX(BUBBLE_EX),
		.BUBBLE_MA(BUBBLE),
		.EXTNOP_EX(EXTNOP_EX),
		.EXTNOP_MA(EXTNOP),
		.GR(GR_EX),
		.HLS_EX(HLS_EX),
		.HLS_MA(HLS),
		.LC_EX(LC_EX),
		.LC_MA(LC),
		.MEMREAD_EX(MEMREAD_EX),
		.MEMREAD_MA(MEMREAD),
		.MEMTOREG_EX(MEMTOREG_EX),
		.MEMTOREG_MA(MEMTOREG),
		.MEMWRITE_EX(MEMWRITE_EX),
		.MEMWRITE_MA(MEMWRITE),
		.MEXT_EX(MEXT_EX),
		.MEXT_MA(MEXT),
		.PC_EX(PC_EX),
		.PC_MA(PC),
		.REGWRITE_EX(REGWRITE_EX),
		.REGWRITE_MA(REGWRITE),
		.REMA(REMA_EX),
		.REG2DATA_EX(REG2DATA_EX),
		.REG2DATA_MA(REG2DATA),
		.RNL_EX(RNL_EX),
		.RNL_MA(RNL),
		.WRREG_EX(WRREG_EX),
		.WRREG_MA(WRREG)
		);

/*****EXT0 -> MA_REG****************************************************************/

	exttoma ext0(
		.ALURESULT_EXT(ALURESULT_EXT0),
		.ALURESULT_MA(ALURESULT),
		.BLS_EXT(BLS_EXT0),
		.BLS_MA(BLS),
		.BRANCH_EXT(BRANCH_EXT0),
		.BRANCH_MA(BRANCH),
		.BUBBLE_EXT(BUBBLE_EXT0),
		.BUBBLE_MA(BUBBLE),
		.EXTNOP_EXT(EXTNOP_EXT0),
		.EXTNOP_MA(EXTNOP),
		.GR(GR_EXT0_PA),
		.HLS_EXT(HLS_EXT0),
		.HLS_MA(HLS),
		.LC_EXT(LC_EXT0),
		.LC_MA(LC),
		.MEMREAD_EXT(MEMREAD_EXT0),
		.MEMREAD_MA(MEMREAD),
		.MEMTOREG_EXT(MEMTOREG_EXT0),
		.MEMTOREG_MA(MEMTOREG),
		.MEMWRITE_EXT(MEMWRITE_EXT0),
		.MEMWRITE_MA(MEMWRITE),
		.MEXT_EXT(MEXT_EXT0),
		.MEXT_MA(MEXT),
		.PC_EXT(PC_EXT0),
		.PC_MA(PC),
		.REG2DATA_EXT(REG2DATA_EXT0),
		.REG2DATA_MA(REG2DATA),
		.REGWRITE_EXT(REGWRITE_EXT0),
		.REGWRITE_MA(REGWRITE),
		.REMA(REMA_EXT0),
		.RNL_EXT(RNL_EXT0),
		.RNL_MA(RNL),
		.WRREG_EXT(WRREG_EXT0),
		.WRREG_MA(WRREG)
		);

/*****EXT1 -> MA_REG****************************************************************/

	exttoma ext1(
		.ALURESULT_EXT(ALURESULT_EXT1),
		.ALURESULT_MA(ALURESULT),
		.BLS_EXT(BLS_EXT1),
		.BLS_MA(BLS),
		.BRANCH_EXT(BRANCH_EXT1),
		.BRANCH_MA(BRANCH),
		.BUBBLE_EXT(BUBBLE_EXT1),
		.BUBBLE_MA(BUBBLE),
		.EXTNOP_EXT(EXTNOP_EXT1),
		.EXTNOP_MA(EXTNOP),
		.GR(GR_EXT1_PA),
		.HLS_EXT(HLS_EXT1),
		.HLS_MA(HLS),
		.LC_EXT(LC_EXT1),
		.LC_MA(LC),
		.MEMREAD_EXT(MEMREAD_EXT1),
		.MEMREAD_MA(MEMREAD),
		.MEMTOREG_EXT(MEMTOREG_EXT1),
		.MEMTOREG_MA(MEMTOREG),
		.MEMWRITE_EXT(MEMWRITE_EXT1),
		.MEMWRITE_MA(MEMWRITE),
		.MEXT_EXT(MEXT_EXT1),
		.MEXT_MA(MEXT),
		.PC_EXT(PC_EXT1),
		.PC_MA(PC),
		.REGWRITE_MA(REGWRITE),
		.REGWRITE_EXT(REGWRITE_EXT1),
		.REMA(REMA_EXT1),
		.REG2DATA_EXT(REG2DATA_EXT1),
		.REG2DATA_MA(REG2DATA),
		.RNL_EXT(RNL_EXT1),
		.RNL_MA(RNL),
		.WRREG_EXT(WRREG_EXT1),
		.WRREG_MA(WRREG)
		);

/*****MA_REG -> MA****************************************************************/

	extoma_reg exreg(
		.ALURESULT_EX(ALURESULT),
		.ALURESULT_MA(ALURESULT_MA),
		.BLS_EX(BLS),
		.BLS_MA(BLS_MA),
		.BRANCH_EX(BRANCH),
		.BRANCH_MA(BRANCH_MA),
		.BUBBLE_EX(BUBBLE),
		.BUBBLE_MA(BUBBLE_MA),
		.CLK(CLK),
		.EXCE(EXCE),
		.EXTNOP_EX(EXTNOP),
		.EXTNOP_MA(EXTNOP_MA),
		.HLS_EX(HLS),
		.HLS_MA(HLS_MA),
		.LC_EX(LC),
		.LC_MA(LC_MA),
		.MEMREAD_EX(MEMREAD),
		.MEMREAD_MA(MEMREAD_MA),
		.MEMTOREG_EX(MEMTOREG),
		.MEMTOREG_MA(MEMTOREG_MA),
		.MEMWRITE_EX(MEMWRITE),
		.MEMWRITE_MA(MEMWRITE_MA),
		.MEXT_EX(MEXT),
		.MEXT_MA(MEXT_MA),
		.PC_EX(PC),
		.PC_MA(PC_MA),
		.PCLK(PCLK),
		.REG2DATA_EX(REG2DATA),
		.REG2DATA_MA(REG2DATA_MA),
		.REGWRITE_EX(REGWRITE),
		.REGWRITE_MA(REGWRITE_MA),
		.RESET(RESET_MA),
		.RNL_EX(RNL),
		.RNL_MA(RNL_MA),
		.WRREG_EX(WRREG),
		.WRREG_MA(WRREG_MA)
		);

/*****MEM -> MA****************************************************************/

	memtoma mem(
		.DMDATAOUT_MA(DMDATAOUT_MA),
		.DMDATAOUT_MEM(DMDATAOUT_MEM),
		.ALS_MEM(ALS_MEM),
		.ALS_MA(ALS_MA)
		);

/*****CP0 -> MA****************************************************************/

	cp0toma cp0(
		.EXC_CP0(EXC_CP0),
		.EXCE_MA(EXCE_MA),
		.RE_CP0(RE_CP0),
		.RE_MA(RE_MA)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end
	
endmodule

/*****MA_REG -> MA****************************************************************/

module extoma_reg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_EX,	/* ALU Result */
	input 			BLS_EX,			/* Byte Load/Store */
	input 			BRANCH_EX,		/* Branch */
	input 			BUBBLE_EX,		/* Bubble in Memory Access */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input 			EXCE,				/* Execute Exception Flush */
	input 			EXTNOP_EX,		/* Extension NOP */
	input 			HLS_EX,			/* Halfword Load/Store */
	input 			LC_EX,			/* Load/Store Conditional */
	input 			MEMREAD_EX,		/* Memory Read */
	input 			MEMTOREG_EX,	/* Memory to Regiser */
	input 			MEMWRITE_EX,	/* Memory Write */
	input 			MEXT_EX,			/* Sign Extend Memory Data */
	input [31:0]	PC_EX,			/* Currect PC */
	input 			PCLK,				/* Pipeline Clock */
	input [31:0]	REG2DATA_EX,	/* Data from Register Read Port 2 */
	input 			REGWRITE_EX,	/* Register Write */
	input 			RESET,			/* System Reset */
	input 			RNL_EX,			/* Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EX,		/* TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_MA,	/* ALU Result */
	output 			BLS_MA,			/* Byte Load/Store */
	output 			BRANCH_MA,		/* Branch */
	output 			BUBBLE_MA,		/* Bubble in Memory Access */
	output 			EXTNOP_MA,		/* Extension NOP */
	output 			HLS_MA,			/* Halfword Load/Store */
	output 			LC_MA,			/* Load/Store Conditional */
	output 			MEMREAD_MA,		/* Memory Read */
	output 			MEMTOREG_MA,	/* Memory to Regiser */
	output 			MEMWRITE_MA,	/* Memory Write */
	output 			MEXT_MA,			/* Sign Extend Memory Data */
	output [31:0]	PC_MA,			/* Currect PC */
	output [31:0]	REG2DATA_MA,	/* Data from Register Read Port 2 */
	output 			REGWRITE_MA,	/* Register Write */
	output 			RNL_MA,			/* Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_MA			/* TISA Destination Register Number */
	);

/*****Registers****************************************************************/

	reg [112:0]	ex_ma;	/* EX -> MA Pipeline Register */
	reg [1:0]	pclkcnt;	/* Pipeline Clock edge detection */

/*****Initialization****************************************************************/

	initial
	begin
		ex_ma = 113'b0;
		pclkcnt = 2'b0;
	end

/*********************************************************************/

	assign BUBBLE_MA		=	ex_ma[112];		//BUBBLE;
	assign EXTNOP_MA		=	ex_ma[111];		//EXTNOP_EX;
	assign LC_MA			=	ex_ma[110];		//LC_EX;
	assign BRANCH_MA		=	ex_ma[109];		//BRANCH_EX;
	assign BLS_MA			=	ex_ma[108];		//BLS_EX;
	assign HLS_MA			=	ex_ma[107];		//HLS_EX;
	assign MEXT_MA			=	ex_ma[106];		//MEXT_EX;
	assign RNL_MA 			=	ex_ma[105];		//RNL_EX;
	assign REGWRITE_MA	= 	ex_ma[104];		//REGWRITE_EX;
	assign MEMTOREG_MA	= 	ex_ma[103];		//MEMTOREG_EX;
	assign MEMREAD_MA		=	ex_ma[102];		//MEMREAD_EX;
	assign MEMWRITE_MA	=	ex_ma[101];		//MEMWRITE_EX;
	assign WRREG_MA		=	ex_ma[100:96];	//WRREG_EX;
	assign PC_MA			=	ex_ma[95:64];	//PC_EX;
	assign ALURESULT_MA	=	ex_ma[63:32];	//ALURESULT_EX;
	assign REG2DATA_MA	=	ex_ma[31:0];	//REG2DATA_EX;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		casex({pclkcnt,RESET,EXCE})
			4'bxx0x	:	begin
								/* Reset */
								ex_ma[112] <= 1'b1;
								ex_ma[111:0] <= 112'b0;
							end
			4'b0111	:	begin
								/* Exception in Pipeline, Flush */
								ex_ma[111:0] <= 112'b0;
								ex_ma[112] <= 1'b1;
							end
			4'b0110	:	begin
								/* Clocking the Pipeline */
								ex_ma <= {BUBBLE_EX,EXTNOP_EX,LC_EX,
									BRANCH_EX,BLS_EX,HLS_EX,
									MEXT_EX,RNL_EX,
									REGWRITE_EX,MEMTOREG_EX,
									MEMREAD_EX,MEMWRITE_EX,
									WRREG_EX,PC_EX,
									ALURESULT_EX,
									REG2DATA_EX};
							end
			default	:	begin
								/* NOP */
							end
		endcase
	end
	
endmodule

/*****EX -> MA_REG****************************************************************/

module extoma(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_EX,	/* ALU Result */
	input 			BLS_EX,			/* Byte Load/Store */
	input 			BRANCH_EX,		/* Branch */
	input 			BUBBLE_EX,		/* Bubble in Memory Access */
	input 			EXTNOP_EX,		/* Extension NOP */
	input 			HLS_EX,			/* Halfword Load/Store */
	input 			GR,				/* Extension 0 Grant Pipeline Resources */
	input 			LC_EX,			/* Load/Store Conditional */
	input 			MEMREAD_EX,		/* Memory Read */
	input 			MEMTOREG_EX,	/* Memory to Regiser */
	input 			MEMWRITE_EX,	/* Memory Write */
	input 			MEXT_EX,			/* Sign Extend Memory Data */
	input [31:0]	PC_EX,			/* Currect PC */
	input [31:0]	REG2DATA_EX,	/* Data from Register Read Port 2 */
	input 			REGWRITE_EX,	/* Register Write */
	input 			REMA,				/* Re-enter at Memory Access */
	input 			RNL_EX,			/* Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EX,		/* TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_MA,	/* ALU Result */
	output 			BLS_MA,			/* Byte Load/Store */
	output 			BRANCH_MA,		/* Branch */
	output 			BUBBLE_MA,		/* Bubble in Memory Access */
	output 			EXTNOP_MA,		/* Extension NOP */
	output 			HLS_MA,			/* Halfword Load/Store */
	output 			LC_MA,			/* Load/Store Conditional */
	output 			MEMREAD_MA,		/* Memory Read */
	output 			MEMTOREG_MA,	/* Memory to Regiser */
	output 			MEMWRITE_MA,	/* Memory Write */
	output 			MEXT_MA,			/* Sign Extend Memory Data */
	output [31:0]	PC_MA,			/* Currect PC */
	output [31:0]	REG2DATA_MA,	/* Data from Register Read Port 2 */
	output 			REGWRITE_MA,	/* Register Write */
	output 			RNL_MA,			/* Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_MA			/* TISA Destination Register Number */
	);

/*********************************************************************/
	
	assign BUBBLE_MA		=	(REMA & GR)?	1'bz:		BUBBLE_EX;
	assign EXTNOP_MA		=	(REMA & GR)?	1'bz:		EXTNOP_EX;
	assign LC_MA			=	(REMA & GR)?	1'bz:		LC_EX;
	assign BRANCH_MA		=	(REMA & GR)?	1'bz:		BRANCH_EX;
	assign BLS_MA			=	(REMA & GR)?	1'bz:		BLS_EX;
	assign HLS_MA			=	(REMA & GR)?	1'bz:		HLS_EX;
	assign MEXT_MA			=	(REMA & GR)?	1'bz:		MEXT_EX;
	assign RNL_MA			=	(REMA & GR)?	1'bz:		RNL_EX;
	assign REGWRITE_MA	=	(REMA & GR)?	1'bz:		REGWRITE_EX;
	assign MEMTOREG_MA	=	(REMA & GR)?	1'bz:		MEMTOREG_EX;
	assign MEMREAD_MA		=	(REMA & GR)?	1'bz:		MEMREAD_EX;
	assign MEMWRITE_MA	=	(REMA & GR)?	1'bz:		MEMWRITE_EX;
	assign WRREG_MA		=	(REMA & GR)?	5'bz:		WRREG_EX;
	assign PC_MA			=	(REMA & GR)?	32'bz:	PC_EX;
	assign ALURESULT_MA	=	(REMA & GR)?	32'bz:	ALURESULT_EX;
	assign REG2DATA_MA	=	(REMA & GR)?	32'bz:	REG2DATA_EX;
endmodule

/*****EXT0 -> MA_REG****************************************************************/

module exttoma(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_EXT,		/* ALU Result */
	input				BLS_EXT,				/* Byte Load/Store */
	input				BRANCH_EXT,			/* Branch */
	input				BUBBLE_EXT,			/* Bubble in Memory Access */
	input				EXTNOP_EXT,			/* Extension NOP */
	input				GR,					/* Extension Grant Pipeline Resources */
	input				HLS_EXT,				/* Halfword Load/Store */
	input				LC_EXT,				/* Load/Store Conditional */
	input				MEMREAD_EXT,		/* Memory Read */
	input				MEMTOREG_EXT,		/* Memory to Regiser */
	input				MEMWRITE_EXT,		/* Memory Write */
	input				MEXT_EXT,			/* Sign Extend Memory Data */
	input [31:0]	PC_EXT,				/* Currect PC */
	input [31:0]	REG2DATA_EXT,		/* Data from Register Read Port 2 */
	input				REGWRITE_EXT,		/* Register Write */
	input				REMA,					/* Re-enter at Memory Access */
	input				RNL_EXT,				/* Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EXT,			/* TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_MA,		/* ALU Result */
	output			BLS_MA,				/* Byte Load/Store */
	output			BRANCH_MA,			/* Branch */
	output			BUBBLE_MA,			/* Bubble in Memory Access */
	output			EXTNOP_MA,			/* Extension NOP */
	output			HLS_MA,				/* Halfword Load/Store */
	output			LC_MA,				/* Load/Store Conditional */
	output			MEMREAD_MA,			/* Memory Read */
	output			MEMTOREG_MA,		/* Memory to Regiser */
	output			MEMWRITE_MA,		/* Memory Write */
	output			MEXT_MA,				/* Sign Extend Memory Data */
	output [31:0]	PC_MA,				/* Currect PC */
	output [31:0]	REG2DATA_MA,		/* Data from Register Read Port 2 */
	output			REGWRITE_MA,		/* Register Write */
	output			RNL_MA,				/* Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_MA				/* TISA Destination Register Number */
	);

/*********************************************************************/

	assign BUBBLE_MA		=	(~REMA | ~GR)?		1'bz:		BUBBLE_EXT;
	assign EXTNOP_MA		=	(~REMA | ~GR)?		1'bz:		EXTNOP_EXT;
	assign LC_MA			=	(~REMA | ~GR)?		1'bz:		LC_EXT;
	assign BRANCH_MA		=	(~REMA | ~GR)?		1'bz:		BRANCH_EXT;
	assign BLS_MA			=	(~REMA | ~GR)?		1'bz:		BLS_EXT;
	assign HLS_MA			=	(~REMA | ~GR)?		1'bz:		HLS_EXT;
	assign MEXT_MA			=	(~REMA | ~GR)?		1'bz:		MEXT_EXT;
	assign RNL_MA			=	(~REMA | ~GR)?		1'bz:		RNL_EXT;
	assign REGWRITE_MA	=	(~REMA | ~GR)?		1'bz:		REGWRITE_EXT;
	assign MEMTOREG_MA	=	(~REMA | ~GR)?		1'bz:		MEMTOREG_EXT;
	assign MEMREAD_MA		=	(~REMA | ~GR)?		1'bz:		MEMREAD_EXT;
	assign MEMWRITE_MA	=	(~REMA | ~GR)?		1'bz:		MEMWRITE_EXT;
	assign WRREG_MA		=	(~REMA | ~GR)?		5'bz:		WRREG_EXT;
	assign PC_MA			=	(~REMA | ~GR)?		32'bz:	PC_EXT;
	assign ALURESULT_MA	=	(~REMA | ~GR)?		32'bz:	ALURESULT_EXT;
	assign REG2DATA_MA	=	(~REMA | ~GR)?		32'bz:	REG2DATA_EXT;

endmodule

/*****MEM -> MA****************************************************************/

module memtoma(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	DMDATAOUT_MEM,	/* Memory Data Out from Memory */
	input		ALS_MEM,
	/* OUTPUT PORTS */
	output [31:0]	DMDATAOUT_MA,	/* Memory Data Out from Memory */
	output	ALS_MA
	);

/*********************************************************************/

	assign DMDATAOUT_MA	=	DMDATAOUT_MEM;
	assign ALS_MA		=	ALS_MEM;
	
endmodule

/*****CP0 -> MA****************************************************************/

module cp0toma(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 	EXC_CP0,		/* Memory Access Exception Flush */
	input		RE_CP0,		/* Reverse Endian */
	/* OUTPUT PORTS */
	output	EXCE_MA,		/* Memory Access Exception Flush */
	output	RE_MA			/* Reverse Endian */
	);

/*********************************************************************/

	assign EXCE_MA	=	EXC_CP0;
	assign RE_MA	=	RE_CP0;

endmodule
