/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*					Bharat Sukhwani													*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	tomem																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/*	The TOMEM module gathers all signals required by the memory			*/
/*	interface and routes them to that region of the design.  This		*/
/*	module also multiplexes signals coming from the extension region.	*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Extension Peripherals, Bug Fixes, see Manual		*/
/*																							*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module tomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,						/* System Clock 50 - 100 MHZ */
	input				PCLK,						/* Pipeline Clock */
	input				RESET,					/* System Reset */
	/* OUTPUT PORTS */
	output			RESET_MEM,				/* Reset Memory */
	/* IF INPUT PORTS */
	input [31:0]	PC_IF,					/* Instruction Fetch Current PC to Memory */
	/* IF OUTPUT PORTS */
	output [31:0]	PC_MEM,					/* Instruction Fetch Current PC to Memory */
	/* ID INPUT PORTS */
	input				FLUSH_ID,				/* Instruction Decode Flush */
	/* ID OUTPUT PORTS */
	output			FLUSH_MEM,				/* Instruction Decode Flush */
	/* MA INPUT PORTS */
	input				BLS_MA,					/* Memory Access Byte Load/Store */
	input [31:0]	DMADD_MA,				/* Memory Access Data Memory Address */
	input [31:0]	DMDATAIN_MA,			/* Memory Access Memory Data In to Memory */
	input				HLS_MA,					/* Memory Access Halfword Load/Store */
	input				MEMREAD_MA,				/* Memory Access Memory Read */
	input				MEMWRITE_MA,			/* Memory Access Memory Write */
	input				RNL_MA,					/* Memory Access Right/Left Unaligned Load/Store */
	input				LC_MA,
	/* MA OUTPUT PORTS */
	output			BLS_MEM,					/* Memory Access Byte Load/Store */
	output [31:0]	DMADD_MEM,				/* Memory Access Data Memory Address */
	output [31:0]	DMDATAIN_MEM,			/* Memory Access Memory Data In to Memory */
	output			HLS_MEM,					/* Memory Access Halfword Load/Store */
	output			MEMREAD_MEM,			/* Memory Access Memory Read */
	output			MEMWRITE_MEM,			/* Memory Access Memory Write */
	output			RNL_MEM,					/* Memory Access Right/Left Unaligned Load/Store */
	output			LC_MEM,
	/* WB INPUT PORTS */
	input 			BUBBLE_WB,
	input [31:0]	PC_WB,
	/* WB OUTPUT PORTS */
	output			BUBBLE_WB_MEM,
	output [31:0]	PC_WB_MEM,
	/* HZ INPUT PORTS */
	input				STALL_HZ,				/* Pipeline Stall */
	/* HZ OUTPUT PORTS */
	output			STALL_MEM,				/* Pipeline Stall */
	/* CP0 INPUT PORTS */
	input				EXC_IF_CP0,				/* Instruction Fetch Exception Flush */
	input				EXC_MA_CP0,				/* Memory Access Exception Flush */
	input				EXT0_PER_IN,			/* */
	input				EXT1_PER_IN,			/* */
	input				RE_CP0,				/* */
	/* CP0 OUTPUT PORTS */
	output			EXC_IF_MEM,				/* Instruction Fetch Exception Flush */
	output			EXC_MA_MEM,				/* Memory Access Exception Flush */
	output			RE_MEM,
	output			PER_EXT0_MEM,
	output			PER_EXT1_MEM,
	/* MEM INPUT PORTS */
	input				BAT_EN_EXT0_IN,		/* */
	input				BAT_EN_EXT1_IN,		/* */
	/* EXT0 INPUT PORTS */
	input				BLS_EXT0_IN,			/* Extension 0 Byte Load/Store */
	input				EXC_EXT0_IN,			/* Extension 0 Exception */
	/*input				EXTPER0_DNE_IN,		/* */
	/*input				EXTPER0_MY_ADDR_IN,	/* */
	input				GR_EXT0_IN,				/* Extension 0 Grant */
	input				HLS_EXT0_IN,			/* Extension 0 Halfword Load/Store */
	input [31:0]	MADDR_EXT0_IN,			/* Extension 0 Memory Address */
	input [31:0]	MDATA_EXT0_IN,			/* Extension 0 Memory Data */
	input				MLOCK_EXT0_IN,			/* Extension 0 Memory Lock */
	input				MOE_EXT0_IN,			/* Extension 0 Memory Read */
	input				MWE_EXT0_IN,			/* Extension 0 Memory Write */
	input				RNL_EXT0_IN,			/* Extension 0 Right/Left Load/Store */
	input 			PER0_DNE_IN,		/* */
	input				PER0_MY_ADDR_IN,		/* */
	input				PER0_PRESENT_IN,		/* */
	input 			PER0_PRIVILEGE_IN,	/* */
	input				PER0_WANTS_INTR_IN,	/* */
	input				INT_EXT0_IN,	/* */
	/* EXT0 OUTPUT PORTS */
	output 			PER0_DNE_OUT,		/* */
	output			PER0_MY_ADDR_OUT,		/* */
	output			PER0_PRESENT_OUT,		/* */
	output 			PER0_PRIVILEGE_OUT,	/* */
	output			PER0_WANTS_INTR_OUT,	/* */
	output			INT_EXT0_OUT, 	/* */
	/* EXT1 INPUT PORTS */
	input				BLS_EXT1_IN,			/* Extension 1 Byte Load/Store */
	input				EXC_EXT1_IN,			/* Extension 1 Exception */
	/*input				EXTPER1_DNE_IN,		/* */
	/*input				EXTPER1_MY_ADDR_IN,	/* */
	input				GR_EXT1_IN,				/* Extension 1 Grant */
	input				HLS_EXT1_IN,			/* Extension 1 Halfword Load/Store */
	input [31:0]	MADDR_EXT1_IN,			/* Extension 1 Memory Address */
	input [31:0]	MDATA_EXT1_IN,			/* Extension 1 Memory Data */
	input				MLOCK_EXT1_IN,			/* Extension 1 Memory Lock */
	input				MOE_EXT1_IN,			/* Extension 1 Memory Read */
	input				MWE_EXT1_IN,			/* Extension 1 Memory Write */
	input				RNL_EXT1_IN,			/* Extension 1 Right/Left Load/Store */
	input 			PER1_DNE_IN,		/* */
	input				PER1_MY_ADDR_IN,		/* */
	input				PER1_PRESENT_IN,		/* */
	input 			PER1_PRIVILEGE_IN,	/* */
	input				PER1_WANTS_INTR_IN,	/* */
	input				INT_EXT1_IN,	/* */
	/* EXT1 OUTPUT PORTS */
	output 			PER1_DNE_OUT,		/* */
	output			PER1_MY_ADDR_OUT,		/* */
	output			PER1_PRESENT_OUT,		/* */
	output 			PER1_PRIVILEGE_OUT,	/* */
	output			PER1_WANTS_INTR_OUT,	/* */
	output			INT_EXT1_OUT,	/* */
	/* EXT OUTPUT PORTS */
	output			BLS_EXT_OUT,			/* Extension Byte Load/Store */
	output			EXC_EXT_OUT,			/* Extension Exception */
	output			HLS_EXT_OUT,			/* Extension Halfword Load/Store */
	output [31:0]	MADDR_EXT_OUT,			/* Extension Memory Address */
	output [31:0]	MDATA_EXT_OUT,			/* Extension Memory Data */
	output			MLOCK_EXT_OUT,			/* Extension Memory Lock */
	output			MOE_EXT_OUT,			/* Extension Memory Read */
	output			MWE_EXT_OUT,			/* Extension Memory Write */
	output			RNL_EXT_OUT,				/* Extension Right/Left Load/Store */
	output [31:0]	MDATA_OUT_PER_EXT			/* */
	);

/*****Signals****************************************************************/

	/*wire			W_BLS_EXT;		/* Extension Byte Load/Store */
	/*wire			W_EXC_EXT;		/* Extension Exception */
	/*wire			W_EXTPER_DNE;	/* */
	/*wire			W_HLS_EXT;		/* Extension Halfword Load/Store */
	/*wire [31:0]	W_MADDR_EXT;	/* Extension Memory Address */
	/*wire [31:0]	W_MDATA_EXT;	/* Extension Memory Data */
	/*wire			W_MLOCK_EXT;	/* Extension Memory Lock */
	/*wire			W_MOE_EXT;		/* Extension Memory Read */
	/*wire			W_MWE_EXT;		/* Extension Memory Write */
	/*wire			W_RNL_EXT;		/* Extension Right/Left Load/Store */

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Memory */

/*****Initialization****************************************************************/

	initial
	begin
		reset_reg = 1'b0;
		pclkcnt = 2'b0;
	end

/*********************************************************************/

	assign RESET_MEM		=	reset_reg;
	assign MADDR_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 32'bz:	32'b0;
	//assign MDATA_EXT_OUT	= W_MDATA_EXT;
	assign MOE_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:		1'b0;
	assign MWE_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:		1'b0;
	assign MLOCK_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:	1'b0;
	assign RNL_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:		1'b0;
	assign BLS_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:		1'b0;
	assign HLS_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:		1'b0;
	assign EXC_EXT_OUT	= (GR_EXT0_IN || GR_EXT1_IN) ? 1'bz:		1'b0;

/*****IF -> MEM****************************************************************/

	iftomem i_f(
		.PC_IF(PC_IF),
		.PC_MEM(PC_MEM)
		);

/*****ID -> MEM****************************************************************/

	idtomem id(
		.FLUSH_ID(FLUSH_ID),
		.FLUSH_MEM(FLUSH_MEM)
		);

/*****MA -> MEM****************************************************************/

	matomem ma(
		.BLS_MA(BLS_MA),
		.BLS_MEM(BLS_MEM),
		.HLS_MA(HLS_MA),
		.HLS_MEM(HLS_MEM),
		.DMADD_MA(DMADD_MA),
		.DMADD_MEM(DMADD_MEM),
		.DMDATAIN_MA(DMDATAIN_MA),
		.DMDATAIN_MEM(DMDATAIN_MEM),
		.MEMREAD_MA(MEMREAD_MA),
		.MEMREAD_MEM(MEMREAD_MEM),
		.MEMWRITE_MA(MEMWRITE_MA),
		.MEMWRITE_MEM(MEMWRITE_MEM),
		.RNL_MA(RNL_MA),
		.RNL_MEM(RNL_MEM),
		.LC_MA(LC_MA),
		.LC_MEM(LC_MEM)
		);

/*****WB -> MEM****************************************************************/

	wbtomem wb(
		.BUBBLE_WB(BUBBLE_WB),
		.PC_WB(PC_WB),
		.BUBBLE_WB_MEM(BUBBLE_WB_MEM),
		.PC_WB_MEM(PC_WB_MEM)
		);

/*****HZ -> MEM****************************************************************/

	hztomem hz(
		.STALL_HZ(STALL_HZ),
		.STALL_MEM(STALL_MEM)
		);

/*****CP0 -> MEM****************************************************************/

	cp0tomem cp0(
		.EXC_IF_CP0(EXC_IF_CP0),
		.EXC_IF_MEM(EXC_IF_MEM),
		.EXC_MA_CP0(EXC_MA_CP0),
		.EXC_MA_MEM(EXC_MA_MEM),
		.RE_CP0(RE_CP0),
		.RE_MEM(RE_MEM),
		.EXT0_PER_IN(EXT0_PER_IN),
		.EXT1_PER_IN(EXT1_PER_IN),
		.PER_EXT0_MEM(PER_EXT0_MEM),
		.PER_EXT1_MEM(PER_EXT1_MEM)
		);

/*****EXT0 -> MEM****************************************************************/

	ext_to_mem	ext_mem0(
		.BAT_EN_EXT_IN(BAT_EN_EXT0_IN),
		.BLS_IN(BLS_EXT0_IN),
		.BLS_OUT(BLS_EXT_OUT),
		.EXC_EXT_IN(EXC_EXT0_IN),
		.EXC_EXT_OUT(EXC_EXT_OUT),
		.EXT_PER_IN(EXT0_PER_IN),
		/*.EXTPER_MY_ADDR_IN(EXTPER0_MY_ADDR_IN),*/
		/*.EXTPER_DNE_IN(EXTPER0_DNE_IN),*/
		.GR_IN(GR_EXT0_IN),
		.HLS_IN(HLS_EXT0_IN),
		.HLS_OUT(HLS_EXT_OUT),
		.MADDR_IN(MADDR_EXT0_IN),
		.MADDR_OUT(MADDR_EXT_OUT),
		.MDATA_IN(MDATA_EXT0_IN),
		.MDATA_OUT(MDATA_EXT_OUT),
		.MLOCK_IN(MLOCK_EXT0_IN),
		.MLOCK_OUT(MLOCK_EXT_OUT),
		.MOE_IN(MOE_EXT0_IN),
		.MOE_OUT(MOE_EXT_OUT), 
		.MWE_IN(MWE_EXT0_IN),
		.MWE_OUT(MWE_EXT_OUT),
		/*.RESET_IN(RESET_MEM),*/
		.RNL_IN(RNL_EXT0_IN),
		.RNL_OUT(RNL_EXT_OUT),
		.PER_DNE_IN(PER0_DNE_IN),
		.PER_MY_ADDR_IN(PER0_MY_ADDR_IN),
		.PER_PRESENT_IN(PER0_PRESENT_IN),
		.PER_PRIVILEGE_IN(PER0_PRIVILEGE_IN),
		.PER_WANTS_INTR_IN(PER0_WANTS_INTR_IN),
		.PER_DNE_OUT(PER0_DNE_OUT),
		.PER_MY_ADDR_OUT(PER0_MY_ADDR_OUT),
		.PER_PRESENT_OUT(PER0_PRESENT_OUT),
		.PER_PRIVILEGE_OUT(PER0_PRIVILEGE_OUT),
		.PER_WANTS_INTR_OUT(PER0_WANTS_INTR_OUT),
		.INT_EXT_IN(INT_EXT0_IN),
		.INT_EXT_OUT(INT_EXT0_OUT),
		.MDATA_OUT_PER(MDATA_OUT_PER_EXT)
		);

/*****EXT1 -> MEM****************************************************************/
	
	ext_to_mem	ext_mem1(
		.BAT_EN_EXT_IN(BAT_EN_EXT1_IN),
		.BLS_IN(BLS_EXT1_IN),
		.BLS_OUT(BLS_EXT_OUT),
		.EXC_EXT_IN(EXC_EXT1_IN),
		.EXC_EXT_OUT(EXC_EXT_OUT),
		.EXT_PER_IN(EXT1_PER_IN),
		/*.EXTPER_MY_ADDR_IN(EXTPER1_MY_ADDR_IN),*/
		/*.EXTPER_DNE_IN(EXTPER1_DNE_IN),*/
		.GR_IN(GR_EXT1_IN),
		.HLS_IN(HLS_EXT1_IN),
		.HLS_OUT(HLS_EXT_OUT),
		.MADDR_IN(MADDR_EXT1_IN),
		.MADDR_OUT(MADDR_EXT_OUT),
		.MDATA_IN(MDATA_EXT1_IN),
		.MDATA_OUT(MDATA_EXT_OUT),
		.MLOCK_IN(MLOCK_EXT1_IN),
		.MLOCK_OUT(MLOCK_EXT_OUT),
		.MOE_IN(MOE_EXT1_IN),
		.MOE_OUT(MOE_EXT_OUT), 
		.MWE_IN(MWE_EXT1_IN),
		.MWE_OUT(MWE_EXT_OUT),
		/*.RESET_IN(RESET_MEM),*/
		.RNL_IN(RNL_EXT1_IN),
		.RNL_OUT(RNL_EXT_OUT),
		.PER_DNE_IN(PER1_DNE_IN),
		.PER_MY_ADDR_IN(PER1_MY_ADDR_IN),
		.PER_PRESENT_IN(PER1_PRESENT_IN),
		.PER_PRIVILEGE_IN(PER1_PRIVILEGE_IN),
		.PER_WANTS_INTR_IN(PER1_WANTS_INTR_IN),
		.PER_DNE_OUT(PER1_DNE_OUT),
		.PER_MY_ADDR_OUT(PER1_MY_ADDR_OUT),
		.PER_PRESENT_OUT(PER1_PRESENT_OUT),
		.PER_PRIVILEGE_OUT(PER1_PRIVILEGE_OUT),
		.PER_WANTS_INTR_OUT(PER1_WANTS_INTR_OUT),
		.INT_EXT_IN(INT_EXT1_IN),
		.INT_EXT_OUT(INT_EXT1_OUT),
		.MDATA_OUT_PER(MDATA_OUT_PER_EXT)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end
	
endmodule

/*****IF -> MEM****************************************************************/

module iftomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	PC_IF,	/* Instruction Fetch Current PC to Memory */
	/* OUTPUT PORTS */
	output [31:0]	PC_MEM	/* Instruction Fetch Current PC to Memory */
	);

/*********************************************************************/

	assign PC_MEM	=	PC_IF;

endmodule

/*****ID -> MEM****************************************************************/

module idtomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		FLUSH_ID,		/* Instruction Decode Flush */
	/* OUTPUT PORTS */
	output	FLUSH_MEM		/* Instruction Decode Flush */
	);

/*********************************************************************/
	
	assign FLUSH_MEM	=	FLUSH_ID;

endmodule

/*****MA -> MEM****************************************************************/

module matomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BLS_MA,			/* Memory Access Byte Load/Store */
	input [31:0]	DMADD_MA,		/* Memory Access Data Memory Address */
	input [31:0]	DMDATAIN_MA,	/* Memory Access Memory Data In to Memory */
	input				HLS_MA,			/* Memory Access Halfword Load/Store */
	input				MEMREAD_MA,		/* Memory Access Memory Read */
	input				MEMWRITE_MA,	/* Memory Access Memory Write */
	input				RNL_MA,			/* Memory Access Right/Left Unaligned Load/Store */
	input				LC_MA,
	/* OUTPUT PORTS */
	output			BLS_MEM,			/* Memory Access Byte Load/Store */
	output [31:0]	DMADD_MEM,		/* Memory Access Data Memory Address */
	output [31:0]	DMDATAIN_MEM,	/* Memory Access Memory Data In to Memory */
	output			HLS_MEM,			/* Memory Access Halfword Load/Store */
	output			MEMREAD_MEM,	/* Memory Access Memory Read */
	output			MEMWRITE_MEM,	/* Memory Access Memory Write */
	output			RNL_MEM,			/* Memory Access Right/Left Unaligned Load/Store */
	output			LC_MEM
	);

/*********************************************************************/

	assign BLS_MEM			=	BLS_MA;
	assign DMADD_MEM		=	DMADD_MA;
	assign DMDATAIN_MEM	=	DMDATAIN_MA;
	assign HLS_MEM			=	HLS_MA;
	assign MEMREAD_MEM	=	MEMREAD_MA;
	assign MEMWRITE_MEM	=	MEMWRITE_MA;
	assign RNL_MEM			=	RNL_MA;
	assign LC_MEM =		LC_MA;

endmodule

/*****HZ -> MEM****************************************************************/

module hztomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		STALL_HZ,		/* Pipeline Stall */
	/* OUTPUT PORTS */
	output	STALL_MEM		/* Pipeline Stall */
	);

/*********************************************************************/
		
	assign STALL_MEM	=	STALL_HZ;

endmodule

/*****CP0 -> MEM****************************************************************/

module cp0tomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		EXC_IF_CP0,		/* Instruction Fetch Exception Flush */
	input		EXC_MA_CP0,		/* Memory Access Exception Flush */
	input		RE_CP0,
	input 	EXT0_PER_IN,
	input		EXT1_PER_IN,
	/* OUTPUT PORTS */
	output	EXC_IF_MEM,		/* Instruction Fetch Exception Flush */
	output	EXC_MA_MEM,		/* Memory Access Exception Flush */
	output	RE_MEM,
	output	PER_EXT0_MEM,
	output	PER_EXT1_MEM
	);

/*********************************************************************/
	
	assign EXC_IF_MEM	=	EXC_IF_CP0;
	assign EXC_MA_MEM	=	EXC_MA_CP0;
	assign RE_MEM = 		RE_CP0;
	assign PER_EXT0_MEM = 	EXT0_PER_IN;
	assign PER_EXT1_MEM = 	EXT1_PER_IN;

endmodule

/*****EXT1 -> MEM****************************************************************/

module ext_to_mem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BAT_EN_EXT_IN,			/* */
	input				BLS_IN,					/* Extension Byte Load/Store */
	input				EXC_EXT_IN,				/* Extension Exception */
	input				EXT_PER_IN,				/* */
	/*input				EXTPER_DNE_IN,			/* */
	/*input				EXTPER_MY_ADDR_IN,	/* */
	input				GR_IN,					/* Extension Grant */
	input				HLS_IN,					/* Extension Halfword Load/Store */
	input [31:0]	MADDR_IN,				/* Extension Memory Address */
	input [31:0]	MDATA_IN,				/* Extension Memory Data */
	input				MLOCK_IN,				/* Extension Memory Lock */
	input				MOE_IN,					/* Extension Memory Read */
	input				MWE_IN,					/* Extension Memory Write */
	/*input				RESET_IN,				/* System Reset */
	input				RNL_IN,					/* Extension Right/Left Load/Store */
	input 			PER_DNE_IN,		/* */
	input				PER_MY_ADDR_IN,		/* */
	input				PER_PRESENT_IN,		/* */
	input 			PER_PRIVILEGE_IN,	/* */
	input				PER_WANTS_INTR_IN,	/* */
	input				INT_EXT_IN,		/* */
	/* OUTPUT PORTS */	
	output			BLS_OUT,					/* Extension Byte Load/Store */
	output			EXC_EXT_OUT,			/* Extension Exception */
	output			HLS_OUT,					/* Extension Halfword Load/Store */
	output [31:0]	MADDR_OUT,				/* Extension Memory Address */
	output [31:0]	MDATA_OUT,				/* Extension Memory Data */
	output [31:0]	MDATA_OUT_PER,			/* */
	output			MLOCK_OUT,				/* Extension Memory Lock */
	output			MOE_OUT,					/* Extension Memory Read */
	output			MWE_OUT,					/* Extension Memory Write */
	output			RNL_OUT,					/* Extension Right/Left Load/Store */
	output 			PER_DNE_OUT,		/* */
	output			PER_MY_ADDR_OUT,		/* */
	output			PER_PRESENT_OUT,		/* */
	output 			PER_PRIVILEGE_OUT,	/* */
	output			PER_WANTS_INTR_OUT,	/* */
	output			INT_EXT_OUT		/* */
	);

/*********************************************************************/

	assign INT_EXT_OUT			=	(EXT_PER_IN)?												INT_EXT_IN:			1'b0;
	assign PER_DNE_OUT			=	(EXT_PER_IN && PER_MY_ADDR_IN)?						PER_DNE_IN:	
											(BAT_EN_EXT_IN && PER_MY_ADDR_IN)?					PER_DNE_IN:
																											1'b1;
	assign PER_MY_ADDR_OUT		=	(EXT_PER_IN)?												PER_MY_ADDR_IN:
											(BAT_EN_EXT_IN)?											PER_MY_ADDR_IN:
																											1'b0;
	assign PER_PRESENT_OUT		=	PER_PRESENT_IN;
	assign PER_PRIVILEGE_OUT	=	PER_PRIVILEGE_IN;
	assign PER_WANTS_INTR_OUT	=	PER_WANTS_INTR_IN;
	assign BLS_OUT					=	(~GR_IN)? 													1'bz:					BLS_IN;
	assign EXC_EXT_OUT			=	(~GR_IN)? 													1'bz:					EXC_EXT_IN;
	assign HLS_OUT					=	(~GR_IN)?													1'bz:					HLS_IN;
	assign MADDR_OUT				=	(~GR_IN)?													32'bz:				MADDR_IN;
	assign MDATA_OUT				=	(GR_IN)?					MDATA_IN:	32'bz;
	assign MDATA_OUT_PER			=	(EXT_PER_IN && PER_MY_ADDR_IN)?						MDATA_IN:
								(BAT_EN_EXT_IN && PER_MY_ADDR_IN)?					MDATA_IN:
																						32'bz;
	assign MLOCK_OUT				=	(~GR_IN)? 													1'bz:					MLOCK_IN;
	assign MOE_OUT					=	(~GR_IN)? 													1'bz:					MOE_IN;
	assign MWE_OUT					=	(~GR_IN)?	 												1'bz:					MWE_IN;
	assign RNL_OUT					=	(~GR_IN)? 													1'bz:					RNL_IN;
	
endmodule

/*****WB -> MEM****************************************************************/

module wbtomem(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		BUBBLE_WB,
	input [31:0] PC_WB,
	/* OUTPUT PORTS */
	output	BUBBLE_WB_MEM,
	output [31:0]	PC_WB_MEM
	);

/*********************************************************************/

	assign BUBBLE_WB_MEM =	BUBBLE_WB;
	assign PC_WB_MEM = PC_WB;

endmodule
