/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	topa																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module topa(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input 			CLK,				/* System Clock 50 - 100 MHZ */
	input 			PCLK,				/* Pipeline Clock */
	input 			RESET,			/* System Reset */
	/* OUTPUT PORTS */
	output 			RESET_PA,		/* Reset Pipeline Arbiter */
	/* ID INPUT PORTS */
	input 			RI_ID,			/* TISA Reserved/Recognized Instruction */
	/* ID OUTPUT PORTS */
	output 			RI_ID_PA,		/* TISA Reserved/Recognized Instruction */
	/* EXT0 INPUT PORTS */
	input 			ACK_EXT0,		/* Extension 0 Enable Acknowledged */
	input 			PI_EXT0,			/* Extension 0 Passive/Parrallel Instruction */
	input 			RI_EXT0,			/* Extension 0 Reserved/Recognized Instruction */
	/* EXT0 OUTPUT PORTS */
	output 			ACK_EXT0_PA,	/* Extension 0 Enable Acknowledged */
	output 			PI_EXT0_PA,		/* Extension 0 Passive/Parrallel Instruction */
	output 			RI_EXT0_PA,		/* Extension 0 Reserved/Recognized Instruction */
	/* EXT1 INPUT PORTS */
	input 			ACK_EXT1,		/* Extension 1 Enable Acknowledged */
	input 			PI_EXT1,			/* Extension 1 Passive/Parrallel Instruction */
	input 			RI_EXT1,			/* Extension 1 Reserved/Recognized Instruction */
	/* EXT0 OUTPUT PORTS */
	output 			ACK_EXT1_PA,	/* Extension 1 Enable Acknowledged */
	output 			PI_EXT1_PA,		/* Extension 1 Passive/Parrallel Instruction */
	output 			RI_EXT1_PA,  	/* Extension 1 Reserved/Recognized Instruction */
	/* HZ INPUT PORTS */
	input 			STALL_HZ,		/* Pipeline Stall */
	/* HZ OUTPUT PORTS */
	output	 		STALL_PA,		/* Pipeline Stall */
	/* CP0 INPUT PORTS */
	input [31:0]	EXTCTR_CP0,		/* Extension Control Register */
	/* CP0 OUTPUT PORTS */
	output [31:0]	EXTCTR_PA		/* Extension Control Register */
	);

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Pipeline Arbiter */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_PA	=	reset_reg;

/*****ID -> PA****************************************************************/

	idtopa id(
		.RI_ID(RI_ID),
		.RI_ID_PA(RI_ID_PA)
		);

/*****EXT0 -> PA****************************************************************/

	exttopa ext0(
		.ACK_EXT(ACK_EXT0),
		.ACK_EXT_PA(ACK_EXT0_PA),
		.PI_EXT(PI_EXT0),
		.PI_EXT_PA(PI_EXT0_PA),
		.RI_EXT(RI_EXT0),
		.RI_EXT_PA(RI_EXT0_PA)
		);

/*****EXT1 -> PA****************************************************************/

	exttopa ext1(
		.ACK_EXT(ACK_EXT1),
		.ACK_EXT_PA(ACK_EXT1_PA),
		.PI_EXT(PI_EXT1),
		.PI_EXT_PA(PI_EXT1_PA),
		.RI_EXT(RI_EXT1),
		.RI_EXT_PA(RI_EXT1_PA)
		);

/*****HZ -> PA****************************************************************/

	hztopa hz(
		.STALL_HZ(STALL_HZ),
		.STALL_PA(STALL_PA)
		);

/*****CP0 -> PA****************************************************************/

	cp0topa cp0(
		.EXTCTR_CP0(EXTCTR_CP0),
		.EXTCTR_PA(EXTCTR_PA)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****ID -> PA****************************************************************/

module idtopa(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		RI_ID,	/* TISA Reserved/Recognized Instruction */
	/* OUTPUT PORTS */
	output	RI_ID_PA	/* TISA Reserved/Recognized Instruction */
	);

/*********************************************************************/

	assign RI_ID_PA	=	RI_ID;
	
endmodule

/*****EXT -> PA****************************************************************/

module exttopa(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		ACK_EXT,			/* Extension Enable Acknowledged */
	input		PI_EXT,			/* Extension Passive/Parrallel Instruction */
	input		RI_EXT,			/* Extension Reserved/Recognized Instruction */
	/* OUTPUT PORTS */
	output	ACK_EXT_PA,		/* Extension Enable Acknowledged */
	output	PI_EXT_PA,		/* Extension Passive/Parrallel Instruction */
	output	RI_EXT_PA		/* Extension Reserved/Recognized Instruction */
	);

/*********************************************************************/

	assign PI_EXT_PA	=	PI_EXT;
	assign ACK_EXT_PA	=	ACK_EXT;
	assign RI_EXT_PA	=	RI_EXT;

endmodule

/*****HZ -> PA****************************************************************/

module hztopa(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		STALL_HZ,	/* Pipeline Stall */
	/* OUTPUT PORTS */
	output	STALL_PA		/* Pipeline Stall */
	);

/*********************************************************************/
	
	assign STALL_PA	=	STALL_HZ;
	
endmodule

/*****CP0 -> PA****************************************************************/

module cp0topa(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	EXTCTR_CP0,	/* Extension Control Register */
	/* OUTPUT PORTS */
	output [31:0]	EXTCTR_PA	/* Extension Control Register */
	);

/*********************************************************************/
	
	assign EXTCTR_PA	=	EXTCTR_CP0;
	
endmodule
