/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	torg																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	eBug Extension, Bug Fixes, see Manual				*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module torg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* System Reset */
	/* OUTPUT PORTS */
	output			RESET_RG,			/* Reset Registers */
	/* PA INPUT PORTS */
	input				GR_EXT0_PA,			/* Extension 0 Grant */
	input				GR_EXT1_PA,			/* Extension 1 Grant */
	/* PA OUTPUT PORTS */
	output			GR_RG,				/* Extension Grant */
	/* ID INPUT PORTS */
	input [4:0]		RS_ID,				/* Register Read Port 1 Register Number */
	input [4:0]		RT_ID,				/* Register Read Port 2 Register Number */
	/* EX INPUT PORTS */
	input [31:0]	HI_EX,				/* */
	input [31:0]	LO_EX,				/* */
	input [31:0]	FPSTATUS_EX,		/* */
	input [31:0]	FPIMP_EX,			/* */
	/* EX OUTPUT PORTS */
	output [31:0]	HI_RG,				/* */
	output [31:0]	LO_RG,				/* */
	output [31:0]	FPSTATUS_RG,		/* */
	output [31:0]	FPIMP_RG,			/* */
	/* MA INPUT PORTS */
	input [4:0]		RD_MA,				/* Memory Access TISA Destination Register Number */
	input [31:0]	RDDATA_MA,			/* Data forwarded from Memory Access Phase */
	input				REGWRITE_MA,		/* Memory Access Register Write */
	/* MA OUTPUT PORTS */
	output [4:0]	RD_MA_RG,			/* Memory Access TISA Destination Register Number */
	output [31:0]	RDDATA_MA_RG,		/* Data forwarded from Memory Access Phase */
	output			REGWRITE_MA_RG,	/* Memory Access Register Write */
	/* WB INPUT PORTS */
	input				BHLS_WB,				/* Byte/Halfword Load/Store */
	input [1:0]		DMADD_WB,			/* Least Significant bits of the Data Memory Address */
	input				REGWRITE_WB,		/* Writeback Register Write */
	input				RNL_WB,				/* Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_WB,			/* Writeback TISA Destination Register Number */
	input [31:0]	WRREGDATA_WB,		/* Data forwarded from Writeback Phase */
	/* WB OUTPUT PORTS */
	output			BHLS_RG,				/* Byte/Halfword Load/Store */
	output [1:0]	DMADD_RG,			/* Least Significant bits of the Data Memory Address */
	output			REGWRITE_WB_RG,	/* Writeback Register Write */
	output			RNL_RG,				/* Right/Left Unaligned Load/Store */
	output [4:0]	RD_WB_RG,			/* Writeback TISA Destination Register Number */
	output [31:0]	RDDATA_WB_RG,		/* Data forwarded from Writeback Phase */
	/* EXT0 INPUT PORTS */
	input [4:0]		RDREG1_EXT0,		/* Register Read Port 1 Register Number */
	input [4:0]		RDREG2_EXT0,		/* Register Read Port 2 Register Number */
	input [4:0]		RDREG3_EXT0,		/* Register Read Port 3 Register Number */
	input [4:0]		RDREG4_EXT0,		/* Register Read Port 4 Register Number */
	input				REGWRITE1_EXT0,	/* Register Write Port 1 Write Enable */
	input				REGWRITE2_EXT0,	/* Register Write Port 2 Write Enable */
	input [31:0]	WRDATA1_EXT0,		/* Register Write Port 1 Data */
	input [31:0]	WRDATA2_EXT0,		/* Register Write Port 2 Data */
	input [4:0]		WRREG1_EXT0,		/* Register Write Port 1 Register Number */
	input [4:0]		WRREG2_EXT0,		/* Register Write Port 2 Register Number */
	input				FSPECIAL_EXT0,		/* */
	/* EXT1 INPUT PORTS */
	input [4:0]		RDREG1_EXT1,		/* Register Read Port 1 Register Number */
	input [4:0]		RDREG2_EXT1,		/* Register Read Port 2 Register Number */
	input [4:0]		RDREG3_EXT1,		/* Register Read Port 3 Register Number */
	input [4:0]		RDREG4_EXT1,		/* Register Read Port 4 Register Number */
	input				REGWRITE1_EXT1,	/* Register Write Port 1 Write Enable */
	input				REGWRITE2_EXT1,	/* Register Write Port 2 Write Enable */
	input [31:0]	WRDATA1_EXT1,		/* Register Write Port 1 Data */
	input [31:0]	WRDATA2_EXT1,		/* Register Write Port 2 Data */
	input [4:0]		WRREG1_EXT1,		/* Register Write Port 1 Register Number */
	input [4:0]		WRREG2_EXT1,		/* Register Write Port 2 Register Number */
	input 			FSPECIAL_EXT1,		/* */
	/* EXT OUTPUT PORTS */
	output [4:0]	RDREG1_RG,			/* Register Read Port 1 Register Number */
	output [4:0]	RDREG2_RG,			/* Register Read Port 2 Register Number */
	output [4:0]	RDREG3_RG,			/* Register Read Port 3 Register Number */
	output [4:0]	RDREG4_RG,			/* Register Read Port 4 Register Number */
	output			REGWRITE1_RG,		/* Register Write Port 1 Write Enable */
	output			REGWRITE2_RG,		/* Register Write Port 2 Write Enable */
	output [31:0]	WRDATA1_RG,			/* Register Write Port 1 Data */
	output [31:0]	WRDATA2_RG,			/* Register Write Port 2 Data */
	output [4:0]	WRREG1_RG,			/* Register Write Port 1 Register Number */
	output [4:0]	WRREG2_RG,			/* Register Write Port 2 Register Number */
	output			FSPECIAL_RG,		/* */
	/* CP0 INPUT PORTS */
	input				RE_CP0,				/* Reverse Endian */
	input [31:0]	STATUS_CP0,			/* */
	input [31:0]	BADVADDR_CP0,		/* */
	input [31:0]	CAUSE_CP0,			/* */
	/* CP0 OUTPUT PORTS */
	output			RE_RG,				/* Reverse Endian */
	output [31:0]	STATUS_RG,			/* */
	output [31:0]	BADVADDR_RG,		/* */
	output [31:0]	CAUSE_RG				/* */
	);

/*****Registers****************************************************************/
	
	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Registers */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_RG = reset_reg;

/*********************************************************************/

	or2_1 or0(
		.A(GR_EXT0_PA),
		.B(GR_EXT1_PA),
		.C(GR_RG)
		);

/*****ID -> RG****************************************************************/

	idtorg id(
		.GR(GR_RG),
		.RDREG1_RG(RDREG1_RG),
		.RDREG2_RG(RDREG2_RG),
		.RDREG3_RG(RDREG3_RG),
		.RDREG4_RG(RDREG4_RG),
		.RS_ID(RS_ID),
		.RT_ID(RT_ID),
		.FSPECIAL_RG(FSPECIAL_RG)
		);

/*****EX -> RG****************************************************************/

	extorg ex(
		.HI_EX(HI_EX),
		.LO_EX(LO_EX),
		.FPSTATUS_EX(FPSTATUS_EX),
		.FPIMP_EX(FPIMP_EX),
		.HI_RG(HI_RG),
		.LO_RG(LO_RG),
		.FPSTATUS_RG(FPSTATUS_RG),
		.FPIMP_RG(FPIMP_RG)
		);

/*****MA -> RG****************************************************************/

	matorg ma(
		.RD_MA(RD_MA),
		.RD_MA_RG(RD_MA_RG),
		.RDDATA_MA(RDDATA_MA),
		.RDDATA_MA_RG(RDDATA_MA_RG),
		.REGWRITE_MA(REGWRITE_MA),
		.REGWRITE_MA_RG(REGWRITE_MA_RG)
		);

/*****WB -> RG****************************************************************/

	wbtorg wb(
		.DMADD_RG(DMADD_RG),
		.DMADD_WB(DMADD_WB),
		.BHLS_RG(BHLS_RG),
		.BHLS_WB(BHLS_WB),
		.GR(GR_RG),
		.RD_WB_RG(RD_WB_RG),
		.RDDATA_WB_RG(RDDATA_WB_RG),
		.REGWRITE_WB(REGWRITE_WB),
		.REGWRITE_WB_RG(REGWRITE_WB_RG),
		.REGWRITE1_RG(REGWRITE1_RG),
		.REGWRITE2_RG(REGWRITE2_RG),
		.RNL_RG(RNL_RG),
		.RNL_WB(RNL_WB),
		.WRDATA1_RG(WRDATA1_RG),
		.WRDATA2_RG(WRDATA2_RG),
		.WRREG_WB(WRREG_WB),
		.WRREG1_RG(WRREG1_RG),
		.WRREG2_RG(WRREG2_RG),
		.WRREGDATA_WB(WRREGDATA_WB)
		);

/*****EXT0 -> RG****************************************************************/

	exttorg ext0(
		.GR(GR_EXT0_PA),
		.RDREG1_EXT(RDREG1_EXT0),
		.RDREG1_RG(RDREG1_RG),
		.RDREG2_EXT(RDREG2_EXT0),
		.RDREG2_RG(RDREG2_RG),
		.RDREG3_EXT(RDREG3_EXT0),
		.RDREG3_RG(RDREG3_RG),
		.RDREG4_EXT(RDREG4_EXT0),
		.RDREG4_RG(RDREG4_RG),
		.REGWRITE_WB(REGWRITE_WB),
		.REGWRITE1_EXT(REGWRITE1_EXT0),
		.REGWRITE1_RG(REGWRITE1_RG),
		.REGWRITE2_EXT(REGWRITE2_EXT0),
		.REGWRITE2_RG(REGWRITE2_RG),
		.WRDATA1_EXT(WRDATA1_EXT0),
		.WRDATA1_RG(WRDATA1_RG),
		.WRDATA2_EXT(WRDATA2_EXT0),
		.WRDATA2_RG(WRDATA2_RG),
		.WRREG1_EXT(WRREG1_EXT0),
		.WRREG1_RG(WRREG1_RG),
		.WRREG2_EXT(WRREG2_EXT0),
		.WRREG2_RG(WRREG2_RG),
		.FSPECIAL_EXT(FSPECIAL_EXT0),
		.FSPECIAL_RG(FSPECIAL_RG)
		);

/*****EXT1 -> RG****************************************************************/

	exttorg ext1(
		.GR(GR_EXT1_PA),
		.RDREG1_EXT(RDREG1_EXT1),
		.RDREG1_RG(RDREG1_RG),
		.RDREG2_EXT(RDREG2_EXT1),
		.RDREG2_RG(RDREG2_RG),
		.RDREG3_EXT(RDREG3_EXT1),
		.RDREG3_RG(RDREG3_RG),
		.RDREG4_EXT(RDREG4_EXT1),
		.RDREG4_RG(RDREG4_RG),
		.REGWRITE_WB(REGWRITE_WB),
		.REGWRITE1_EXT(REGWRITE1_EXT1),
		.REGWRITE1_RG(REGWRITE1_RG),
		.REGWRITE2_EXT(REGWRITE2_EXT1),
		.REGWRITE2_RG(REGWRITE2_RG),
		.WRDATA1_EXT(WRDATA1_EXT1),
		.WRDATA1_RG(WRDATA1_RG),
		.WRDATA2_EXT(WRDATA2_EXT1),
		.WRDATA2_RG(WRDATA2_RG),
		.WRREG1_EXT(WRREG1_EXT1),
		.WRREG1_RG(WRREG1_RG),
		.WRREG2_EXT(WRREG2_EXT1),
		.WRREG2_RG(WRREG2_RG),
		.FSPECIAL_EXT(FSPECIAL_EXT1),
		.FSPECIAL_RG(FSPECIAL_RG)
		);

/*****CP0 -> RG****************************************************************/

	cp0torg cp0(
		.RE_CP0(RE_CP0),
		.RE_RG(RE_RG),
		.STATUS_CP0(STATUS_CP0),
		.BADVADDR_CP0(BADVADDR_CP0),
		.CAUSE_CP0(CAUSE_CP0),
		.STATUS_RG(STATUS_RG),
		.BADVADDR_RG(BADVADDR_RG),
		.CAUSE_RG(CAUSE_RG)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end
endmodule

/*****ID -> RG****************************************************************/

module idtorg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				GR,			/* Extension Grant */
	input [4:0]		RS_ID,		/* Register Read Port 1 Register Number */
	input [4:0]		RT_ID,		/* Register Read Port 2 Register Number */
	/* OUTPUT PORTS */
	output [4:0]	RDREG1_RG,	/* Register Read Port 1 Register Number */
	output [4:0]	RDREG2_RG,	/* Register Read Port 2 Register Number */
	output [4:0]	RDREG3_RG,	/* Register Read Port 3 Register Number */
	output [4:0]	RDREG4_RG,	/* Register Read Port 4 Register Number */
	output			FSPECIAL_RG	/* */
	);
	
/*********************************************************************/

	assign RDREG1_RG		=	(GR == 1'b1)?	5'bz:	RS_ID;
	assign RDREG2_RG		=	(GR == 1'b1)?	5'bz:	RT_ID;
	assign RDREG3_RG		=	(GR == 1'b1)?	5'bz:	5'b0;
	assign RDREG4_RG		=	(GR == 1'b1)?	5'bz:	5'b0;
	assign FSPECIAL_RG	=	(GR == 1'b1)?	1'bz:	1'b0;

endmodule

/*****MA -> RG****************************************************************/

module matorg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [4:0]		RD_MA,				/* Memory Access TISA Destination Register Number */
	input [31:0]	RDDATA_MA,			/* Data forwarded from Memory Access Phase */
	input				REGWRITE_MA,		/* Memory Access Register Write */
	/* OUTPUT PORTS */
	output [4:0]	RD_MA_RG,			/* Memory Access TISA Destination Register Number */
	output [31:0]	RDDATA_MA_RG,		/* Data forwarded from Memory Access Phase */
	output			REGWRITE_MA_RG		/* Memory Access Register Write */
	);

/*********************************************************************/
	
	assign RD_MA_RG			=	RD_MA;
	assign RDDATA_MA_RG		=	RDDATA_MA;
	assign REGWRITE_MA_RG	=	REGWRITE_MA;

endmodule

/*****EX -> RG****************************************************************/

module extorg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	HI_EX,				/* */
	input [31:0]	LO_EX,				/* */
	input [31:0]	FPSTATUS_EX,		/* */
	input [31:0]	FPIMP_EX,			/* */
	/* OUTPUT PORTS */
	output [31:0]	HI_RG,				/* */
	output [31:0]	LO_RG,				/* */
	output [31:0]	FPSTATUS_RG,		/* */
	output [31:0]	FPIMP_RG				/* */
	);

/*********************************************************************/

	assign HI_RG			=	HI_EX;
	assign LO_RG			=	LO_EX;
	assign FPSTATUS_RG	=	FPSTATUS_EX;
	assign FPIMP_RG		=	FPIMP_EX;

endmodule

/*****WB -> RG****************************************************************/

module wbtorg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BHLS_WB,				/* Byte/Halfword Load/Store */
	input [1:0]		DMADD_WB,			/* Least Significant bits of the Data Memory Address */
	input				GR,					/* Extension Grant */
	input				REGWRITE_WB,		/* Writeback Register Write */
	input				RNL_WB,				/* Right/Left Unaligned Load/Store */
	input [31:0]	WRREGDATA_WB,		/* Data forwarded from Writeback Phase */
	input [4:0]		WRREG_WB,			/* Writeback TISA Destination Register Number */
	/* OUTPUT PORTS */
	output 			BHLS_RG,				/* Byte/Halfword Load/Store */
	output [1:0]	DMADD_RG,			/* Least Significant bits of the Data Memory Address */
	output [4:0]	RD_WB_RG,			/* Writeback TISA Destination Register Number */
	output [31:0]	RDDATA_WB_RG,		/* Data forwarded from Writeback Phase */
	output			REGWRITE_WB_RG,	/* Writeback Register Write */
	output			REGWRITE1_RG,		/* Register Write Port 1 Write Enable */
	output			REGWRITE2_RG,		/* Register Write Port 2 Write Enable */
	output			RNL_RG,				/* Right/Left Unaligned Load/Store */
	output [31:0]	WRDATA1_RG,			/* Register Write Port 1 Data */
	output [31:0]	WRDATA2_RG,			/* Register Write Port 2 Data */
	output [4:0]	WRREG1_RG,			/* Register Write Port 1 Register Number */
	output [4:0]	WRREG2_RG			/* Register Write Port 2 Register Number */
	);

/*********************************************************************/
	
	assign BHLS_RG				=	(GR & ~REGWRITE_WB)?	1'b0:		BHLS_WB;
	assign DMADD_RG			= 	(GR & ~REGWRITE_WB)?	2'b0:		DMADD_WB;
	assign RD_WB_RG			=	WRREG_WB;
	assign RDDATA_WB_RG		=	WRREGDATA_WB;
	assign REGWRITE_WB_RG	=	REGWRITE_WB;
	assign REGWRITE1_RG		= 	(GR & ~REGWRITE_WB)?	1'bz:		REGWRITE_WB;
	assign REGWRITE2_RG		= 	(GR & ~REGWRITE_WB)?	1'bz:		1'b0;
	assign RNL_RG				=	(GR & ~REGWRITE_WB)?	1'b0:		RNL_WB;
	assign WRDATA1_RG			= 	(GR & ~REGWRITE_WB)?	32'bz:	WRREGDATA_WB;
	assign WRDATA2_RG			= 	(GR & ~REGWRITE_WB)?	32'bz:	32'b0;
	assign WRREG1_RG			= 	(GR & ~REGWRITE_WB)?	5'bz:		WRREG_WB;
	assign WRREG2_RG			= 	(GR & ~REGWRITE_WB)?	5'bz:		5'b0;

endmodule

/*****EXT -> RG****************************************************************/

module exttorg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				GR,					/* Extension Grant */
	input [4:0]		RDREG1_EXT,			/* Register Read Port 1 Register Number */
	input [4:0]		RDREG2_EXT,			/* Register Read Port 2 Register Number */
	input [4:0]		RDREG3_EXT,			/* Register Read Port 3 Register Number */
	input [4:0]		RDREG4_EXT,			/* Register Read Port 4 Register Number */
	input				REGWRITE_WB,		/* Writeback Register Write */
	input				REGWRITE1_EXT,		/* Register Write Port 1 Write Enable */
	input				REGWRITE2_EXT,		/* Register Write Port 2 Write Enable */
	input [31:0]	WRDATA1_EXT,		/* Register Write Port 1 Data */
	input [31:0]	WRDATA2_EXT,		/* Register Write Port 2 Data */
	input [4:0]		WRREG1_EXT,			/* Register Write Port 1 Register Number */
	input [4:0]		WRREG2_EXT,			/* Register Write Port 2 Register Number */
	input				FSPECIAL_EXT,		/* */
	/* OUTPUT PORTS */
	output [4:0]	RDREG1_RG,			/* Register Read Port 1 Register Number */
	output [4:0]	RDREG2_RG,			/* Register Read Port 2 Register Number */
	output [4:0]	RDREG3_RG,			/* Register Read Port 3 Register Number */
	output [4:0]	RDREG4_RG,			/* Register Read Port 4 Register Number */
	output			REGWRITE1_RG,		/* Register Write Port 1 Write Enable */
	output			REGWRITE2_RG,		/* Register Write Port 2 Write Enable */
	output [31:0]	WRDATA1_RG,			/* Register Write Port 1 Data */
	output [31:0]	WRDATA2_RG,			/* Register Write Port 2 Data */
	output [4:0]	WRREG1_RG,			/* Register Write Port 1 Register Number */
	output [4:0]	WRREG2_RG,			/* Register Write Port 2 Register Number */
	output			FSPECIAL_RG			/* */
	);
	
/*********************************************************************/

	assign RDREG1_RG		=	(~GR)?					5'bz:		RDREG1_EXT;
	assign RDREG2_RG		=	(~GR)?					5'bz:		RDREG2_EXT;
	assign RDREG3_RG		=	(~GR)?					5'bz:		RDREG3_EXT;
	assign RDREG4_RG		=	(~GR)?					5'bz:		RDREG4_EXT;
	assign REGWRITE1_RG	=	(~GR | REGWRITE_WB)?	1'bz:		REGWRITE1_EXT;
	assign REGWRITE2_RG	= 	(~GR | REGWRITE_WB)?	1'bz:		REGWRITE2_EXT;
	assign WRDATA1_RG		=	(~GR | REGWRITE_WB)?	32'bz:	WRDATA1_EXT;
	assign WRDATA2_RG		=	(~GR | REGWRITE_WB)?	32'bz:	WRDATA2_EXT;
	assign WRREG1_RG		=	(~GR | REGWRITE_WB)?	5'bz:		WRREG1_EXT;
	assign WRREG2_RG		=	(~GR | REGWRITE_WB)?	5'bz:		WRREG2_EXT;
	assign FSPECIAL_RG	=	(~GR)?					1'bz:		FSPECIAL_EXT;

endmodule

/*****CP0 -> RG****************************************************************/

module cp0torg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				RE_CP0,				/* Reverse Endian */
	input [31:0]	STATUS_CP0,			/* */
	input [31:0]	BADVADDR_CP0,		/* */
	input [31:0]	CAUSE_CP0,			/* */
	/* OUTPUT PORTS */
	output			RE_RG,				/* Reverse Endian */
	output [31:0]	STATUS_RG,			/* */
	output [31:0]	BADVADDR_RG,		/* */
	output [31:0]	CAUSE_RG				/* */
	);

	assign RE_RG			=	RE_CP0;
	assign STATUS_RG		=	STATUS_CP0;
	assign BADVADDR_RG	=	BADVADDR_CP0;
	assign CAUSE_RG		=	CAUSE_CP0;

endmodule
