/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	towb																*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module towb(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input				EXCE,					/* Memory Access Exception Flush */
	input				PCLK,					/* Pipeline Clock */
	input				RESET,				/* Reset Memory Access */
	/* OUTPUT PORTS */
	output			RESET_WB,			/* Reset Writeback */
	/* PA INPUT PORTS */
	input				GR_EXT0_PA,			/* Extension 0 Grant Pipeline Resources */
	input				GR_EXT1_PA,			/* Extension 1 Grant Pipeline Resources */
	/* MA INPUT PORTS */
	input [31:0]	ALURESULT_MA,		/* Memory Access ALU Result */
	input				BHLS_MA,				/* Memory Access Byte/Halfword Load/Store */
	input				BRANCH_MA,			/* Memory Access Branch */
	input				BUBBLE_MA,			/* Bubble in Memory Access  */
	input [1:0]		DMADD_MA,			/* Memory Access  Least Significant bits of the Data Memory Address */
	input [31:0]	DMDATAOUT_MA,		/* Memory Access Memory Data Out */
	input				EXTNOP_MA,			/* Memory Access  Extension NOP */
	input				MEMTOREG_MA,		/* Memory Access Memory to Regiser */
	input [31:0]	PC_MA,				/* Memory Access  Currect PC */
	input				REGWRITE_MA,		/* Memory Access  Register Write */
	input				RNL_MA,				/* Memory Access  Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_MA,			/* Memory Access  TISA Destination Register Number */
	/* MA OUTPUT PORTS */
	output [31:0]	ALURESULT_WB,		/* Writeback ALU Result */
	output			BHLS_WB,				/* Writeback Byte/Halfword Load/Store */
	output			BRANCH_WB,			/* Writeback Branch */
	output			BUBBLE_WB,			/* Bubble in Writeback */
	output [1:0]	DMADD_WB,			/* Writeback Least Significant bits of the Data Memory Address */
	output [31:0]	DMDATAOUT_WB,		/* Writeback Memory Data Out */
	output			EXTNOP_WB,			/* Writeback Extension NOP */
	output			MEMTOREG_WB,		/* Writeback Memory to Regiser */
	output [31:0]	PC_WB,				/* Writeback Currect PC */
	output			REGWRITE_WB,		/* Writeback Register Write */
	output			RNL_WB,				/* Writeback Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_WB,			/* Writeback TISA Destination Register Number */
	/* EXT0 INPUT PORTS */
	input [31:0]	ALURESULT_EXT0,	/* Extension 0 ALU Result */
	input				BHLS_EXT0,			/* Extension 0 Byte/Halfword Load/Store */
	input				BRANCH_EXT0,		/* Extension 0 Branch */
	input				BUBBLE_EXT0,		/* Extension 0 Bubble in Memory Access */
	input [1:0]		DMADD_EXT0,			/* Extension 0 Least Significant bits of the Data Memory Address */
	input [31:0]	DMDATAOUT_EXT0,	/* Extension 0 Memory Data Out */
	input				EXTNOP_EXT0,		/* Extension 0 Extension NOP */
	input				MEMTOREG_EXT0,		/* Extension 0 Memory to Regiser */
	input [31:0]	PC_EXT0,				/* Extension 0 Currect PC */
	input				REGWRITE_EXT0,		/* Extension 0 Register Write */
	input				REWB_EXT0,			/* Extension 0 Re-enter at Writeback */ 
	input				RNL_EXT0,			/* Extension 0 Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EXT0,			/* Extension 0 TISA Destination Register Number */
	/* EXT1 INPUT PORTS */
	input [31:0]	ALURESULT_EXT1,	/* Extension 1 ALU Result */
	input				BHLS_EXT1,			/* Extension 1 Byte/Halfword Load/Store */
	input				BRANCH_EXT1,		/* Extension 1 Branch */
	input				BUBBLE_EXT1,		/* Extension 1 Bubble in Memory Access */
	input [1:0]		DMADD_EXT1,			/* Extension 1 Least Significant bits of the Data Memory Address */
	input [31:0]	DMDATAOUT_EXT1,	/* Extension 1 Memory Data Out */
	input				EXTNOP_EXT1,		/* Extension 1 Extension NOP */
	input				MEMTOREG_EXT1,		/* Extension 1 Memory to Regiser */
	input [31:0]	PC_EXT1,				/* Extension 1 Currect PC */
	input				REGWRITE_EXT1,		/* Extension 1 Writeback Register Write */
	input				REWB_EXT1,			/* Extension 1 Re-enter at Writeback */
	input				RNL_EXT1,			/* Extension 1 Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EXT1			/* Extension 1 TISA Destination Register Number */
	);

/*****Signals****************************************************************/

	wire [31:0]	ALURESULT;	/* ALU Result */
	wire			BHLS;			/* Byte/Halfword Load/Store */
	wire			BRANCH;		/* Branch */
	wire			BUBBLE;		/* Bubble */
	wire [1:0]	DMADD;		/* Least Significant bits of the Data Memory Address */
	wire [31:0]	DMDATAOUT;	/* Memory Data Out */
	wire			EXTNOP;		/* Extension NOP */
	wire			GR_MA;		/* Extension Grant Pipeline Resources */
	wire			MEMTOREG;	/* Memory to Regiser */
	wire [31:0]	PC;			/* Currect PC */
	wire			REGWRITE;	/* Register Write */
	wire			REWB_MA;		/* Extension Re-enter at Writeback */
	wire			RNL;			/* Right/Left Unaligned Load/Store */
	wire [4:0]	WRREG;		/* TISA Destination Register Number */

/*****Registers****************************************************************/

	reg [1:0]	pclkcnt;		/* Pipeline Clock edge detection */
	reg			reset_reg;	/* Reset Writeback */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		reset_reg = 1'b0;
	end

/*********************************************************************/

	assign RESET_WB = reset_reg;

/*********************************************************************/

	or2_1 or0(
		.A(GR_EXT0_PA),
		.B(GR_EXT1_PA),
		.C(GR_MA)
		);

	or2_1 or1(
		.A(REWB_EXT0),
		.B(REWB_EXT1),
		.C(REWB_MA)
		);

/*****MA -> WB_REG****************************************************************/

	matowb ma(
		.ALURESULT_MA(ALURESULT_MA),
		.ALURESULT_WB(ALURESULT),
		.BHLS_MA(BHLS_MA),
		.BHLS_WB(BHLS),
		.BRANCH_MA(BRANCH_MA),
		.BUBBLE_MA(BUBBLE_MA),
		.BUBBLE_WB(BUBBLE),
		.BRANCH_WB(BRANCH),
		.DMADD_MA(DMADD_MA),
		.DMADD_WB(DMADD),
		.DMDATAOUT_MA(DMDATAOUT_MA),
		.DMDATAOUT_WB(DMDATAOUT),
		.EXTNOP_MA(EXTNOP_MA),
		.EXTNOP_WB(EXTNOP),
		.GR(GR_MA),
		.MEMTOREG_MA(MEMTOREG_MA),
		.MEMTOREG_WB(MEMTOREG),
		.PC_MA(PC_MA),
		.PC_WB(PC),
		.REGWRITE_MA(REGWRITE_MA),
		.REGWRITE_WB(REGWRITE),
		.REWB(REWB_MA),
		.RNL_MA(RNL_MA),
		.RNL_WB(RNL),
		.WRREG_MA(WRREG_MA),
		.WRREG_WB(WRREG)
		);

/*****EXT0 -> WB_REG****************************************************************/

	exttowb ext0(
		.ALURESULT_EXT(ALURESULT_EXT0),
		.ALURESULT_WB(ALURESULT),
		.BHLS_EXT(BHLS_EXT0),
		.BHLS_WB(BHLS),
		.BRANCH_EXT(BRANCH_EXT0),
		.BRANCH_WB(BRANCH),
		.BUBBLE_EXT(BUBBLE_EXT0),
		.BUBBLE_WB(BUBBLE),
		.DMADD_EXT(DMADD_EXT0),
		.DMADD_WB(DMADD),
		.DMDATAOUT_EXT(DMDATAOUT_EXT0),
		.DMDATAOUT_WB(DMDATAOUT),
		.EXTNOP_EXT(EXTNOP_EXT0),
		.EXTNOP_WB(EXTNOP),
		.GR(GR_EXT0_PA),
		.MEMTOREG_EXT(MEMTOREG_EXT0),
		.MEMTOREG_WB(MEMTOREG),
		.PC_EXT(PC_EXT0),
		.PC_WB(PC),
		.REGWRITE_EXT(REGWRITE_EXT0),
		.REGWRITE_WB(REGWRITE),
		.REWB(REWB_EXT0),
		.RNL_EXT(RNL_EXT0),
		.RNL_WB(RNL),
		.WRREG_EXT(WRREG_EXT0),
		.WRREG_WB(WRREG)
		);

/*****EXT1 -> WB_REG****************************************************************/

	exttowb ext1(
		.ALURESULT_EXT(ALURESULT_EXT1),
		.ALURESULT_WB(ALURESULT),
		.BHLS_EXT(BHLS_EXT1),
		.BHLS_WB(BHLS),
		.BRANCH_EXT(BRANCH_EXT1),
		.BRANCH_WB(BRANCH),
		.BUBBLE_EXT(BUBBLE_EXT1),
		.BUBBLE_WB(BUBBLE),
		.DMADD_EXT(DMADD_EXT1),
		.DMADD_WB(DMADD),
		.DMDATAOUT_EXT(DMDATAOUT_EXT1),
		.DMDATAOUT_WB(DMDATAOUT),
		.EXTNOP_EXT(EXTNOP_EXT1),
		.EXTNOP_WB(EXTNOP),
		.GR(GR_EXT1_PA),
		.MEMTOREG_EXT(MEMTOREG_EXT1),
		.MEMTOREG_WB(MEMTOREG),
		.PC_EXT(PC_EXT1),
		.PC_WB(PC),
		.REGWRITE_EXT(REGWRITE_EXT1),
		.REGWRITE_WB(REGWRITE),
		.REWB(REWB_EXT0),
		.RNL_EXT(RNL_EXT1),
		.RNL_WB(RNL),
		.WRREG_EXT(WRREG_EXT1),
		.WRREG_WB(WRREG)
		);

/*****WB_REG -> WB****************************************************************/

	matowb_reg mareg(
		.ALURESULT_MA(ALURESULT),
		.ALURESULT_WB(ALURESULT_WB),
		.BHLS_MA(BHLS),
		.BHLS_WB(BHLS_WB),
		.BRANCH_MA(BRANCH),
		.BRANCH_WB(BRANCH_WB),
		.BUBBLE_MA(BUBBLE),
		.BUBBLE_WB(BUBBLE_WB),
		.CLK(CLK),
		.DMADD_MA(DMADD),
		.DMADD_WB(DMADD_WB),
		.DMDATAOUT_MA(DMDATAOUT),
		.DMDATAOUT_WB(DMDATAOUT_WB),
		.EXCE(EXCE),
		.EXTNOP_MA(EXTNOP),
		.EXTNOP_WB(EXTNOP_WB),
		.MEMTOREG_MA(MEMTOREG),
		.MEMTOREG_WB(MEMTOREG_WB),
		.PC_MA(PC),
		.PC_WB(PC_WB),
		.PCLK(PCLK),
		.REGWRITE_MA(REGWRITE),
		.REGWRITE_WB(REGWRITE_WB),
		.RESET(RESET_WB),
		.RNL_MA(RNL),
		.RNL_WB(RNL_WB),
		.WRREG_MA(WRREG),
		.WRREG_WB(WRREG_WB)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		case(pclkcnt)
			2'b01		:	begin
								/* Synchronize Reset to Pipeline Clock */
								reset_reg <= RESET;
							end
			default	:	begin
							end
		endcase
	end

endmodule

/*****WB_REG -> WB****************************************************************/

module matowb_reg(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_MA,		/* Memory Access ALU Result */
	input				BHLS_MA,				/* Memory Access Byte/Halfword Load/Store */
	input				BRANCH_MA,			/* Memory Access Branch */
	input				BUBBLE_MA,			/* Bubble in Memory Access */
	input				CLK,					/* System Clock 50 - 100 MHZ */
	input [1:0]		DMADD_MA,			/* Memory Access Least Significant bits of the Data Memory Address */
	input [31:0]	DMDATAOUT_MA,		/* Memory Access Memory Data Out */
	input				EXCE,					/* Memory Access Exception Flush */
	input				EXTNOP_MA,			/* Memory Access Extension NOP */
	input				MEMTOREG_MA,		/* Memory Access Memory to Regiser */
	input [31:0]	PC_MA,				/* Memory Access Currect PC */
	input				PCLK,					/* Pipeline Clock */
	input				REGWRITE_MA,		/* Memory Access Register Write */
	input				RESET,				/* System Reset */
	input				RNL_MA,				/* Memory Access Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_MA,			/* Memory Access TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_WB,		/* Writeback ALU Result */
	output			BHLS_WB,				/* Writeback Byte/Halfword Load/Store */
	output			BRANCH_WB,			/* Writeback Branch */
	output			BUBBLE_WB,			/* Bubble in Writeback */
	output [1:0]	DMADD_WB,			/* Writeback Least Significant bits of the Data Memory Address */
	output [31:0]	DMDATAOUT_WB,		/* Writeback Memory Data Out */
	output			EXTNOP_WB,			/* Writeback Extension NOP */
	output			MEMTOREG_WB,		/* Writeback Memory to Regiser */
	output [31:0]	PC_WB,				/* Writeback Currect PC */
	output			REGWRITE_WB,		/* Writeback Register Write */
	output			RNL_WB,				/* Writeback Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_WB				/* Writeback TISA Destination Register Number */
	);
	
/*****Registers****************************************************************/

	reg [109:0] ma_wb;	/* MA -> WB Pipeline Register */
	reg [1:0] pclkcnt;	/* Pipeline Clock edge detection */

/*****Initialization****************************************************************/

	initial
	begin
		pclkcnt = 2'b0;
		ma_wb = 110'b0;
	end

/*********************************************************************/

	assign BUBBLE_WB		=	ma_wb[109];			//BUBBLE;
	assign EXTNOP_WB		=	ma_wb[108];			//EXTNOP_MA;
	assign BRANCH_WB		=	ma_wb[107];			//BRANCH_MA;
	assign BHLS_WB			=	ma_wb[106];			//BHLS_MA;
	assign RNL_WB			=	ma_wb[105];			//RNL_MA
	assign REGWRITE_WB	=	ma_wb[104];			//REGWRITE_MA;
	assign MEMTOREG_WB	=	ma_wb[103];			//MEMTOREG_MA;
	assign DMADD_WB		=	ma_wb[102:101];	//DMADD_MA;
	assign WRREG_WB		=	ma_wb[100:96];		//WRREG_MA;
	assign PC_WB			=	ma_wb[95:64];		//PC_MA;
	assign ALURESULT_WB	=	ma_wb[63:32];		//ALURESULT_MA;
	assign DMDATAOUT_WB	=	ma_wb[31:0];		//DMDATAOUT_MA;

/*********************************************************************/

	always@(posedge CLK)
	begin
		/* Pipeline Clock edge detection */
		pclkcnt <= {pclkcnt[0],PCLK};
	end

	always@(posedge CLK)
	begin
		casex({pclkcnt,RESET,EXCE})
			4'bxx0x	:	begin
								/* Reset */ 
								ma_wb[109] <= 1'b1;
								ma_wb[108:0] <= 109'b0;
							end
			4'b0111	:	begin
								/* Exception in Pipeline, Flush */
								ma_wb[108:0] <= 109'b0;
								ma_wb[109] <= 1'b1;
							end
			4'b0110	:	begin
								/* Clocking the Pipeline */
								ma_wb <= {BUBBLE_MA,EXTNOP_MA,BRANCH_MA,
									BHLS_MA,RNL_MA,
									REGWRITE_MA,MEMTOREG_MA,
									DMADD_MA,WRREG_MA,PC_MA,
									ALURESULT_MA,
									DMDATAOUT_MA};	
							end
			default	:	begin
							end
		endcase
	end
endmodule

/*****MA -> WB_REG****************************************************************/

module matowb(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_MA,		/* Memory Access ALU Result */
	input				BHLS_MA,				/* Memory Access Byte/Halfword Load/Store */
	input				BRANCH_MA,			/* Memory Access Branch */
	input				BUBBLE_MA,			/* Bubble in Memory Access */
	input [1:0]		DMADD_MA,			/* Memory Access Least Significant bits of the Data Memory Address */
	input [31:0]	DMDATAOUT_MA,		/* Memory Access Memory Data Out */
	input				EXTNOP_MA,			/* Memory Access Extension NOP */
	input				GR,					/* Extension 0 Grant Pipeline Resources */
	input				MEMTOREG_MA,		/* Memory Access Memory to Regiser */
	input [31:0]	PC_MA,				/* Memory Access Currect PC */
	input				REGWRITE_MA,		/* Memory Access Register Write */
	input				REWB,					/* Extension Re-enter at Writeback */
	input				RNL_MA,				/* Memory Access Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_MA,			/* Memory Access TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_WB,		/* Writeback ALU Result */
	output			BHLS_WB,				/* Writeback Byte/Halfword Load/Store */
	output			BRANCH_WB,			/* Writeback Branch */
	output			BUBBLE_WB,			/* Bubble in Writeback */
	output [1:0]	DMADD_WB,			/* Writeback Least Significant bits of the Data Memory Address */
	output [31:0]	DMDATAOUT_WB,		/* Writeback Memory Data Out */
	output			EXTNOP_WB,			/* Writeback Extension NOP */
	output			MEMTOREG_WB,		/* Writeback Memory to Regiser */
	output [31:0]	PC_WB,				/* Writeback Currect PC */
	output			REGWRITE_WB,		/* Writeback Register Write */
	output			RNL_WB,				/* Writeback Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_WB				/* Writeback TISA Destination Register Number */
	);

/*********************************************************************/

	assign ALURESULT_WB	= 	(REWB & GR)?	32'bz:	ALURESULT_MA;
	assign BHLS_WB			=	(REWB & GR)?	1'bz:		BHLS_MA;
	assign BUBBLE_WB		= 	(REWB & GR)?	1'bz:		BUBBLE_MA;
	assign BRANCH_WB		=	(REWB & GR)?	1'bz:		BRANCH_MA;
	assign DMADD_WB		= 	(REWB & GR)?	2'bz:		DMADD_MA;
	assign DMDATAOUT_WB	= 	(REWB & GR)?	32'bz:	DMDATAOUT_MA;
	assign EXTNOP_WB		=	(REWB & GR)?	1'bz:		EXTNOP_MA;
	assign MEMTOREG_WB	= 	(REWB & GR)?	1'bz:		MEMTOREG_MA;
	assign PC_WB			=	(REWB & GR)?	32'bz:	PC_MA;
	assign REGWRITE_WB	= 	(REWB & GR)?	1'bz:		REGWRITE_MA;
	assign RNL_WB			=	(REWB & GR)?	1'bz:		RNL_MA;
	assign WRREG_WB		= 	(REWB & GR)?	5'bz:		WRREG_MA;

endmodule

/*****EXT -> WB_REG****************************************************************/

module exttowb(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ALURESULT_EXT,		/* Extension ALU Result */
	input				BHLS_EXT,			/* Extension Byte/Halfword Load/Store */
	input				BRANCH_EXT,			/* Extension Branch */
	input				BUBBLE_EXT,			/* Extension Bubble in Memory Access */
	input [1:0]		DMADD_EXT,			/* Extension Least Significant bits of the Data Memory Address */
	input [31:0]	DMDATAOUT_EXT,		/* Extension Memory Data Out */
	input				EXTNOP_EXT,			/* Extension Extension NOP */
	input				GR,					/* Extension Grant Pipeline Resources */
	input				MEMTOREG_EXT,		/* Extension Memory to Regiser */
	input [31:0]	PC_EXT,				/* Extension Currect PC */
	input				REGWRITE_EXT,		/* Extension Register Write */
	input				REWB,					/* Extension Re-enter at Writeback */
	input				RNL_EXT,				/* Extension Right/Left Unaligned Load/Store */
	input [4:0]		WRREG_EXT,			/* Extension TISA Destination Register Number */
	/* OUTPUT PORTS */
	output [31:0]	ALURESULT_WB,		/* Writeback ALU Result */
	output			BHLS_WB,				/* Writeback Byte/Halfword Load/Store */
	output			BRANCH_WB,			/* Writeback Branch */
	output			BUBBLE_WB,			/* Bubble in Writeback */
	output [1:0]	DMADD_WB,			/* Writeback Least Significant bits of the Data Memory Address */
	output [31:0]	DMDATAOUT_WB,		/* Writeback Memory Data Out */
	output			EXTNOP_WB,			/* Writeback Extension NOP */
	output			MEMTOREG_WB,		/* Writeback Memory to Regiser */
	output [31:0]	PC_WB,				/* Writeback Currect PC */
	output			REGWRITE_WB,		/* Writeback Register Write */
	output			RNL_WB,				/* Writeback Right/Left Unaligned Load/Store */
	output [4:0]	WRREG_WB				/* Writeback TISA Destination Register Number */
	);

/*********************************************************************/
	
	assign ALURESULT_WB	= 	(~REWB | ~GR)?		32'bz:	ALURESULT_EXT;
	assign BHLS_WB			=	(~REWB | ~GR)?		1'bz:		BHLS_EXT;
	assign BRANCH_WB		=	(~REWB | ~GR)?		1'bz:		BRANCH_EXT;
	assign BUBBLE_WB		= 	(~REWB | ~GR)?		1'bz:		BUBBLE_EXT;
	assign DMADD_WB		= 	(~REWB | ~GR)?		2'bz:		DMADD_EXT;
	assign DMDATAOUT_WB	= 	(~REWB | ~GR)?		32'bz:	DMDATAOUT_EXT;
	assign EXTNOP_WB		=	(~REWB | ~GR)?		1'bz:		EXTNOP_EXT;
	assign MEMTOREG_WB	= 	(~REWB | ~GR)?		1'bz:		MEMTOREG_EXT;
	assign PC_WB			=	(~REWB | ~GR)?		32'bz:	PC_EXT;
	assign REGWRITE_WB	= 	(~REWB | ~GR)?		1'bz:		REGWRITE_EXT;
	assign RNL_WB			=	(~REWB | ~GR)?		1'bz:		RNL_EXT;
	assign WRREG_WB		= 	(~REWB | ~GR)?		5'bz:		WRREG_EXT;

endmodule
