/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	usart_controller												*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module usart_controller(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input [31:0]	ADDR_IN,		/* Address Bus from Data Path Memory Interface */
	input [1:0]		BYTES,		/* Number of Bytes to Read/Write */
	input [31:0]	CADDR,		/* Configuration Base Address */
	input				CLK,			/* System Clock 50 - 100 MHZ */
	input [31:0]	DATA_IN,		/* Data Bus from Data Path Memory Interface */
	input				OE,			/* Output Enable */
	input				RESET,		/* System Reset */
	input				RX,			/* Serial Receive */
	input				SRT,			/* Start Transaction */
	input				USARTCLK,	/* USART Clock 100 MHZ */
	input				WE,			/* Write Enable */
	/* OUTPUT PORTS */
	output [31:0]	BASETAG,		/* BASETAG Register */
	output [31:0]	DATA_OUT,	/* Data Bus to Data Path Memory Interface */
	output			DNE,			/* Transaction Done */
	output			SCLK,			/* Baud Clock */
	output [31:0]	STATUS,		/* Control Register */
	output			TX,			/* Serial Transmit */
	output			USART_IRQ	/* USART Interrupt */
	);

/*****Signals****************************************************************/

	wire			BAUD_CLK;	/* Baud Clock */
	wire [3:0]	BM;			/* Byte Selects */
	wire			BREAK;		/* Received Break */
	wire [31:0]	DATA_IN_C;	/* Endian Flipped Incoming Data */
	wire [31:0]	DATA_IN_M;	/* Masked Incoming Data */
	wire [31:0]	DATA_OUT_C;	/* Endian Flipped Outgoing Data */
	wire			FRAME;		/* Frame Error */
	wire			PARE;			/* Parity Error */
	wire [31:0]	REG_M;		/* Masked Configuration Register Data */
	wire [31:0]	REG_M_0;		/* Selected Configuration Register Data */
	wire			REOT;			/* End of Receive */
	wire [8:0]	RXDATAW;		/* Received Data */
	wire			RXRDY;		/* Receiver Ready */
	wire			TEOT;			/* End of Transmit */
	wire			TIDLE;		/* Transmitter Idle */
	wire			TXRDY;		/* Transmitter Ready */
	wire [31:0]	WMASK_IN;	/* Incoming Data Write Mask */
	wire [31:0]	WMASK_REG;	/* Configuration Register Data Write Mask */
	wire [31:0] BASE_ADDR;	/* */
	wire [31:0] MAX_ADDR;	/* */

/*****Registers****************************************************************/

	reg [31:0]	basetagr;			/* BASETAG Register */
	reg [31:0]	baud;					/* Baud Register */
	reg [31:0]	channelstatus;		/* Channel Status Register */
	reg [31:0]	control;				/* Control Register */
	reg [31:0]	data_in_m_reg;		/* Masked Incoming Data */
	reg [3:0]	dnecnt;				/* Transaction Progress Counter */
	reg			dner;					/* Configuration Space Transaction Done */
	reg [31:0]	intrdisable;		/* Interrupt Disable Register */
	reg [31:0]	intrenable;			/* Interrupt Enable Register */
	reg [31:0]	intrmask;			/* Interrupt Mask Register */
	reg			men_reg;				/* Transaction Enabled for USART Configuration Space */
	reg [31:0]	reg_m_0_reg;		/* Selected Configuration Register Data */
	reg [31:0]	reg_m_reg;			/* Masked Configuration Register Data */
	reg			rxd;					/* Received Data flag */
	reg [31:0]	rxdata;				/* Receive Register */
	reg			rxen;					/* Receiver Enabled */
	reg [32:0]	timeout;				/* Time-out Counter */
	reg			timer;				/* Time-out Enabled */
	reg			tsrt;					/* Transmit Start */
	reg [31:0]	txdata;				/* Transmit Register */
	reg			txen;					/* Transmiter Enabled */
	reg [1:0]	txrdycnt;			/* Transmiter Ready Edge Detection */

/*****Parameters****************************************************************/

	/* Register Addresses */
	parameter	BASETAGREG =			16'h0000;	/* BASETAG Register Relative Address */
	parameter	CONTROLREG =			16'h0004;	/* Control Register Relative Address */
	parameter	INTRENABLEREG =		16'h0008;	/* Interrupt Enable Register Relative Address */
	parameter	INTRDISABLEREG =		16'h000c;	/* Interrupt Disable Register Relative Address */
	parameter	INTRMASK =				16'h0010;	/* Interrupt Mask Register Relative Address */
	parameter	CHANNELSTATUSREG =	16'h0014;	/* Channel Status Register Relative Address */
	parameter	RXDATAREG = 			16'h0018;	/* Recieve Register Relative Address */
	parameter	TXDATAREG =				16'h001c;	/* Transmit Register Relative Address */
	parameter	BAUDREG =				16'h0020;	/* Baud Register Relative Address */
	parameter	TIMEOUTREG =			16'h0024;	/* Timeout Register Relative Address */
	parameter	RESERVED0REG =			16'h0028;
	parameter	RESERVED1REG =			16'h002c;
	parameter	RESERVED2REG =			16'h0030;
	parameter	RESERVED3REG =			16'h0034;
	parameter	RESERVED4REG =			16'h0038;
	parameter	RESERVED5REG =			16'h003c;

	/* BASETAG Register */
	parameter	BASEl =					31;			/* BASE Bit Range Left */
	parameter	BASEr =					16;			/* BASE Bit Range Right */
	parameter	TAGl = 					15;			/* TAG Bit Range Left */
	parameter	TAGr = 					0;				/* TAG Bit Range Right */
	parameter	TAG =						6;				/* TAG Value */

	/* Control Register */
	parameter	CHMODEl =				31;			/* Channel Mode Bit Range Left */
	parameter	CHMODEr =				30;			/* Channel Mode Bit Range Right */
	parameter	NBSTOPl =				29;			/* Stop Bits Bit Range Left */
	parameter	NBSTOPr =				28;			/* Stop Bits Bit Range Right */
	parameter	USCLKSl =				27;			/* Use Clock Bit Range Left */
	parameter	USCLKSr =				24;			/* Use Clock Bit Range Right */
	parameter	CHRLl =					23;			/* Character Length Bit Range Left */
	parameter	CHRLr =					22;			/* Character Length Bit Range Right */
	parameter	PARl = 					21;			/* Parity Type Bit Range Left */
	parameter	PARr = 					19;			/* Parity Type Bit Range Right */
	parameter	CLKO =					18;			/* Output Clock */
	parameter	MODE9 =					17;			/* 9 bit Mode */
	parameter	SENDA = 					16;			/* Send Address */
	parameter	STTTO = 					11;			/* Start Time-Out */
	parameter	STPBRK =					10;			/* Stop Break */
	parameter	STTBRK = 				9;				/* Start Break */
	parameter	RSTSTA = 				8;				/* Reset Status Bits */
	parameter	TXDIS = 					7;				/* Disable Transmitter */
	parameter	TXEN = 					6;				/* Enable Transmitter */
	parameter	RXDIS =					5;				/* Disable Receiver */
	parameter	RXEN	=					4;				/* Enable Receiver */
	parameter	RSTTX = 					3;				/* Reset Transmitter */
	parameter	RSTRX = 					2;				/* Reset Receiver */
	parameter	RST = 					0;				/* Reset Usart */

	parameter	SIZE = 64;

	/* Interrupt, Channel Status Register */
	parameter	TXEMPTY =				9;				/* Transmitter Empty */
	parameter	TIMEOUTINT =			8;				/* Time-out */
	parameter	PAREINT =				7;				/* Parity Error */
	parameter	FRAMEINT =				6;				/* Frame Error */
	parameter	OVRE =					5;				/* Overrun Error */
	parameter	ENDTX =					4;				/* End of Transmit */
	parameter	ENDRX =					3;				/* End of Receive */
	parameter	RXBRK =					2;				/* Recieve Break */
	parameter	TXRDYINT =				1;				/* Transmitter Ready */
	parameter	RXRDYINT = 				0;				/* Receiver Ready */

/*****Variables****************************************************************/

	integer i;

/*****Initialization****************************************************************/

	initial
	begin
		channelstatus = 32'b0;
		control = 32'b0;
		control[CHRLl:CHRLr] = 2'b11;
		control[PARl:PARr] = 3'b100;
		basetagr = 32'b0;
		basetagr[TAGl:TAGr] = TAG;
		baud = 32'b0;
		dnecnt = 4'b1111;
		dner = 1'b1;
		intrdisable = 32'b0;
		intrenable = 32'b0;
		intrmask = 32'b0;
		men_reg = 1'b0;
		reg_m_0_reg = 32'b0;
		reg_m_reg = 32'b0;
		rxd = 1'b0;
		rxdata = 32'b0;
		rxen = 1'b0;
		timeout = {1'b0,32'hffffffff};
		timer = 1'b0;
		tsrt = 1'b0;
		txen = 1'b0;
		txdata = 32'b0;
		txrdycnt = 2'b0;
	end

/*********************************************************************/

	assign BASETAG		=	basetagr;
	assign DNE			=	dner;
	assign SCLK			=	BAUD_CLK;
	assign STATUS		=	channelstatus;
	assign USART_IRQ	=	(| (channelstatus & intrmask));

/*****Address Decoding****************************************************************/

	assign BASE_ADDR = CADDR;
	assign MAX_ADDR = CADDR + SIZE;
	assign MEN = (ADDR_IN >= BASE_ADDR) & (ADDR_IN < MAX_ADDR);

/*****Incoming Data Handling****************************************************************/

	assign WMASK_REG	=	{{8{BM[0]}},{8{BM[1]}},{8{BM[2]}},{8{BM[3]}}};
	assign WMASK_IN	=	{{8{~BM[0]}},{8{~BM[1]}},{8{~BM[2]}},{8{~BM[3]}}};
	assign DATA_IN_M	=	DATA_IN_C & WMASK_IN;
	assign REG_M		=	reg_m_0_reg & WMASK_REG;

/*****Outgoing Data Handling****************************************************************/
	
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BASETAGREG)?basetagr:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CONTROLREG)?control:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == INTRENABLEREG))?intrmask:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == INTRENABLEREG))?intrenable:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == INTRDISABLEREG))?intrmask:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == INTRDISABLEREG))?intrdisable:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == CHANNELSTATUSREG)?channelstatus:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == RXDATAREG)?rxdata:32'bz;
	assign REG_M_0 = (OE & (ADDR_IN[BASEr-1:0] == TXDATAREG))?32'hffffffff:32'bz;
	assign REG_M_0 = (WE & (ADDR_IN[BASEr-1:0] == TXDATAREG))?txdata:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == INTRMASK)?intrmask:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == BAUDREG)?baud:32'bz;
	assign REG_M_0 = (ADDR_IN[BASEr-1:0] == TIMEOUTREG)?timeout[31:0]:32'bz;
	assign REG_M_0 = (men_reg & (ADDR_IN[BASEr-1:0] == RESERVED0REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (ADDR_IN[BASEr-1:0] == RESERVED1REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (ADDR_IN[BASEr-1:0] == RESERVED2REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (ADDR_IN[BASEr-1:0] == RESERVED3REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (ADDR_IN[BASEr-1:0] == RESERVED4REG))?32'b0:32'bz;
	assign REG_M_0 = (men_reg & (ADDR_IN[BASEr-1:0] == RESERVED5REG))?32'b0:32'bz;

	assign DATA_OUT = (OE & men_reg)?DATA_OUT_C:32'bz;

/*****Byte Selects****************************************************************/

	bytesel2 bs(
		.ADDR(ADDR_IN[1:0]),
		.BYTES(BYTES),
		.BW(BM)
		);

/*****Endian Flips****************************************************************/

	/* Endian Flip incoming data */
	endianflip32 ed1(
		.IN(DATA_IN),
		.EN(1'b1),
		.OUT(DATA_IN_C)
		);

	/* Endian Flip outgoing data */
	endianflip32 ed2(
		.IN(reg_m_0_reg),
		.EN(1'b1),
		.OUT(DATA_OUT_C)
		);

/*****USART****************************************************************/

	usart u0(
		.BAUD_CLK(BAUD_CLK),
		.BREAK(BREAK),
		.BREAKSRT(control[STTBRK]),
		.BREAKSTP(control[STPBRK]),
		.CHMODE(control[CHMODEl:CHMODEr]),
		.CHRL(control[CHRLl:CHRLr]),
		.CLK(USARTCLK),
		.CLK_DIV(baud),
		.CLK_EXT(SCLK),
		.CLK_SEL(control[USCLKSl:USCLKSr]),
		.FRAME(FRAME),
		.MODE9(control[MODE9]),
		.NBSTOP(control[NBSTOPl:NBSTOPr]),
		.PAR(control[PARl:PARr]),
		.PARE(PARE),
		.REOT(REOT),
		.RESET(RESET & ~control[RST]),
		.RSTRX(~control[RSTRX] & rxen),
		.RSTTX(~control[RSTTX] & txen),
		.RX(RX),
		.RX_DATA(RXDATAW),
		.RXRDY(RXRDY),
		.SYNC(1'b0),
		.TEOT(TEOT),
		.TIDLE(TIDLE),
		.TX(TX),
		.TX_DATA(txdata[8:0]),
		.TX_SRT(tsrt),
		.TXRDY(TXRDY)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			data_in_m_reg = 32'b0;
			men_reg = 1'b0;
			reg_m_0_reg = 32'b0;
			reg_m_reg = 32'b0;
		end
		else
		begin
			/* Latch Signals */
			data_in_m_reg = DATA_IN_M;
			dnecnt = {dnecnt[2:0],dner};
			men_reg = MEN;
			reg_m_0_reg = REG_M_0;
			reg_m_reg = REG_M;
		end
	end

	always@(posedge CLK)
	begin
		if (RXRDY)
		begin
			/* Received Byte */
			rxd = 1'b1;
		end
		if (timer & rxen)
		begin
			/* Decrement Time-out */
			timeout = timeout - 1;
		end
		if (tsrt & (txrdycnt == 2'b10))
		begin
			/* Transmitter Started */
			tsrt = 1'b0;
		end
		txrdycnt = {txrdycnt[0],TXRDY};
		casex({RESET,control[RST],dnecnt[3],dner})
			4'b0xxx	:	begin
								/* Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								baud = 32'b0;
								channelstatus = 32'b0;
								control = 32'b0;
								control[CHRLl:CHRLr] = 2'b11;
								control[PARl:PARr] = 3'b100;
								dner = 1'b1;
								intrdisable = 32'b0;
								intrenable = 32'b0;
								intrmask = 32'b0;
								rxd = 1'b0;
								rxdata = 32'b0;
								rxen = 1'b0;
								timeout = {1'b0,32'hffffffff};
								timer = 1'b0;
								tsrt = 1'b0;
								txen = 1'b0;
								txrdycnt = 2'b0;
								txdata = 32'b0;
							end
			4'b11xx	:	begin
								/* Soft Reset */
								basetagr = 32'b0;
								basetagr[TAGl:TAGr] = TAG;
								baud = 32'b0;
								channelstatus = 32'b0;
								control = 32'b0;
								control[CHRLl:CHRLr] = 2'b11;
								control[PARl:PARr] = 3'b100;
								dner = 1'b1;
								intrdisable = 32'b0;
								intrenable = 32'b0;
								intrmask = 32'b0;
								rxd = 1'b0;
								rxdata = 32'b0;
								rxen = 1'b0;
								timeout = {1'b0,32'hffffffff};
								timer = 1'b0;
								tsrt = 1'b0;
								txen = 1'b0;
								txrdycnt = 2'b0;
								txdata = 32'b0;
							end
			4'b1000	:	begin	
								/* Update Registers */
								dner = 1'b1;
								if (men_reg & OE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										RXDATAREG		:	begin
																	/* Clear Receiver Ready */
																	channelstatus[RXRDYINT] = 1'b0;
																end
										TIMEOUTREG		:	begin
																	/* Stop Time-out */
																	timer = 1'b0;
																	control[STTTO] = 1'b0;
																	channelstatus[TIMEOUTINT] = 1'b0;
																end
										default			:	begin
																	/* Do Nothing */
																	/* Address not Recognized */
																end
									endcase
								end
								if (men_reg & WE)
								begin
									casex(ADDR_IN[BASEr-1:0])
										BASETAGREG			:	begin
																		/* Update Basetag Register */
																		basetagr = {(data_in_m_reg[31:TAGl+1] | reg_m_reg[31:TAGl+1]),basetagr[TAGl:TAGr]};
																	end
										CONTROLREG			:	begin
																		/* Update Control Register */
																		control = data_in_m_reg | reg_m_reg;
																	end
										INTRENABLEREG		:	begin
																		/* Update Interrupt Enable Register */
																		intrenable = data_in_m_reg | reg_m_reg;
																	end
										INTRDISABLEREG		:	begin
																		/* Update Interrupt Disable Register */
																		intrdisable = data_in_m_reg | reg_m_reg;
																	end
										TXDATAREG			:	begin
																		/* Update Transmit Register */
																		txdata = data_in_m_reg | reg_m_reg;
																		/* Start Transmitter */
																		tsrt = 1'b1;
																	end
										TIMEOUTREG			:	begin
																		/* Update Timeout Register */
																		timeout = {1'b0,data_in_m_reg | reg_m_reg};
																	end
										BAUDREG				:	begin
																		/* Update Baud Register */
																		baud = data_in_m_reg | reg_m_reg;
																	end
										default				:	begin
																		/* Do Nothing */
																		/* Address not Recognized */
																	end
									endcase
								end
							end
			4'b10x1	:	begin
								/* Wait for incoming request */
								if (SRT & men_reg)
								begin
									/* Request Recognized */
									dner = 1'b0;
								end
								if (timeout[32])
								begin
									/* Time-out */
									channelstatus[TIMEOUTINT] = 1'b1;
									timeout[32] = 1'b0;
								end
								if (BREAK)
								begin
									/* Recieve Break */
									channelstatus[RXBRK] = 1'b1;
								end
								if (FRAME)
								begin
									/* Frame Error */
									channelstatus[FRAMEINT] = 1'b1;
								end
								if (PARE)
								begin
									/* Parity Error */
									channelstatus[PAREINT] = 1'b1;
								end
								if (TIDLE)
								begin
									/* Transmitter Empty */
									channelstatus[TXEMPTY] = 1'b1;
								end
								if (TEOT)
								begin
									/* End of Transmit */
									channelstatus[ENDTX] = 1'b1;
								end
								if (REOT)
								begin
									/* End of Receive */
									channelstatus[ENDRX] = 1'b1;
								end
								if (RXRDY & rxen & channelstatus[RXRDYINT])
								begin
									/* Overrun Error */
									channelstatus[OVRE] = 1'b1;
								end
								if (~RXRDY & rxd)
								begin
									/* Recieved Byte */
									rxd = 1'b0;
									channelstatus[RXRDYINT] = rxen;
									if (rxen)
									begin
										/* Update Receive Register */
										timer = 1'b0;
										rxdata[8:0] = RXDATAW;
									end
								end
								else
								begin
									channelstatus[ENDRX] = 1'b0;
								end
								if (TXRDY)
								begin
									/* Transmitter Ready */
									channelstatus[TXRDYINT] = txen;
								end
								else
								begin
									channelstatus[TXEMPTY] = 1'b0;
									channelstatus[ENDTX] = 1'b0;
									channelstatus[TXRDYINT] = 1'b0;
								end
								if (control[RSTRX])
								begin
									/* Reset Receiver */
									control[RSTRX] = 1'b0;
								end
								if (control[RSTTX])
								begin
									/* Reset Transmitter */
									control[RSTTX] = 1'b0;
								end
								if ((control[RXEN]) && (~rxen))
								begin
									/* Enable Receiver */
									rxen = 1'b1;
									control[RXDIS] = 1'b0;
								end
								if ((control[RXDIS]) && (rxen))
								begin
									/* Disable Receiver */
									channelstatus[RXRDYINT] = 1'b0;
									rxen = 1'b0;
									control[RXEN] = 1'b0;
								end
								if ((control[TXEN]) && (~txen))
								begin
									/* Enable Transmitter */
									channelstatus[TXRDYINT] = 1'b1;
									txen = 1'b1;
									control[TXDIS] = 1'b0;
								end
								if ((control[TXDIS]) && (txen))
								begin
									/* Disable Transmitter */
									channelstatus[TXRDYINT] = 1'b0;
									txen = 1'b0;
									control[TXEN] = 1'b0;
								end
								if (control[RSTSTA])
								begin
									/* Reset Status Bits */
									channelstatus[PAREINT] = 1'b0;
									channelstatus[FRAMEINT] = 1'b0;
									channelstatus[OVRE] = 1'b0;
									channelstatus[RXBRK] = 1'b0;
									control[RSTSTA] = 1'b0;
								end
								if (control[STTBRK])
								begin
									/* Start Break */
									if (txen)
									begin
										if (TXRDY)
										begin
											control[STTBRK] = 1'b0;
										end
									end
									else
									begin
										control[STTBRK] = 1'b0;
									end
								end
								if (control[STPBRK])
								begin
									/* Stop Break */
									control[STTBRK] = 1'b0;
									if (txen)
									begin
										if (TXRDY)
										begin
											control[STPBRK] = 1'b0;
										end
									end
									else
									begin
										control[STPBRK] = 1'b0;
									end
								end
								if (control[STTTO])
								begin
									/* Start Time-out */
									timer = 1'b1;
								end
								else
								begin
									timer = 1'b0;
								end
								/* Update Interrupt Mask */
								for (i=0; i<32; i=i+1)
								begin
									if (intrenable[i]) intrmask[i] = 1'b1;
								end
								intrenable = 32'b0;
								for (i=0; i<32; i=i+1)
								begin
									if (intrdisable[i]) intrmask[i] = 1'b0;
								end
								intrdisable = 32'b0;
							end
			default	:	begin
								/* Do Nothing */
							end
		endcase
	end
endmodule

/*****USART****************************************************************/

module usart(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BREAKSRT,	/* Start Break */
	input				BREAKSTP,	/* Stop Break */
	input [1:0]		CHMODE,		/* Channel Mode */
	input [1:0]		CHRL,			/* Character Length */
	input				CLK,			/* USART Clock 100 MHZ */
	input [31:0]	CLK_DIV,		/* Clock Divider */
	input				CLK_EXT,		/* External Clock */
	input [3:0]		CLK_SEL,		/* Clock Selection */
	input				MODE9,		/* 9 bit Mode */
	input [1:0]		NBSTOP,		/* Number of Stop Bits */
	input [2:0]		PAR,			/* Parity Type */
	input				RESET,		/* System Reset */
	input				RSTRX,		/* Reset Receiver */
	input				RSTTX,		/* Reset Transmitter */
	input				RX,			/* Serial Receive */
	input				SYNC,			/* Synchronous Mode */
	input [8:0]		TX_DATA,		/* Transmit Data */
	input				TX_SRT,		/* Transmit Start */
	/* OUTPUT PORTS */
	output			BAUD_CLK,	/* Baud Clock */
	output			BREAK,		/* Received Break */
	output			FRAME,		/* Frame Error */
	output			PARE,			/* Parity Error */
	output			REOT,			/* End of Receive */
	output			RIDLE,		/* Receiver Idle */
	output [8:0]	RX_DATA,		/* Received Data */
	output			RXRDY,		/* Receiver Ready */
	output			TEOT,			/* End of Transmit */
	output			TIDLE,		/* Transmitter Idle */
	output			TX,			/* Serial Transmit */
	output			TXRDY			/* Transmitter Ready */
	);
	
/*****Signals****************************************************************/

	wire BAUD_CLK16_EN;	/* Baud 16x Clock Enable */
	wire RX_0;				/* Serial Receive Internal */
	wire TX_0;				/* Serial Transmit Interal */

/*****Channel Mode****************************************************************/

	assign RX_0	= 	(CHMODE == 2'b00)?	RX:
						(CHMODE == 2'b01)?	RX:
						(CHMODE == 2'b10)?	TX_0:
						(CHMODE == 2'b11)?	1'b1:	
													RX;	
													
	assign TX = 	(CHMODE == 2'b00)?	TX_0:
						(CHMODE == 2'b01)?	RX:
						(CHMODE == 2'b10)?	1'b1:
						(CHMODE == 2'b11)?	RX:
													TX_0;

/*****Baud Clock Generator****************************************************************/

	baudclk_generator bg(
		.BAUD_CLK(BAUD_CLK),
		.BAUD_CLK16_EN(BAUD_CLK16_EN),
		.CLK(CLK),
		.CLK_DIV(CLK_DIV),
		.CLK_EXT(CLK_EXT),
		.CLK_SEL(CLK_SEL),
		.RESET(RESET),
		.SYNC(SYNC)
		);

/*****Serial Transmitter****************************************************************/
	
	transmitter ut(
		.BAUD_CLK16(BAUD_CLK16_EN),
		.BREAKSRT(BREAKSRT),
		.BREAKSTP(BREAKSTP),
		.CHRL(CHRL),
		.CLK(CLK),
		.IDLE(TIDLE),
		.MODE9(MODE9),
		.NBSTOP(NBSTOP),
		.PAR(PAR),
		.RESET(RESET & RSTTX),
		.SYNC(SYNC),
		.TEOT(TEOT),
		.TX(TX_0),
		.TX_DATA(TX_DATA),
		.TX_SRT(TX_SRT),
		.TXRDY(TXRDY)
		);

/*****Serial Receiver****************************************************************/

	receiver ur(
		.BAUD_CLK16(BAUD_CLK16_EN),
		.BREAK(BREAK),
		.CHRL(CHRL),
		.CLK(CLK),
		.FRAME(FRAME),
		.IDLE(RIDLE),
		.MODE9(MODE9),
		.NBSTOP(NBSTOP),
		.PAR(PAR),
		.PARE(PARE),
		.REOT(REOT),
		.RESET(RESET & RSTRX),
		.RX(RX_0),
		.RX_DATA(RX_DATA),
		.RXRDY(RXRDY),
		.SYNC(SYNC)
		);

endmodule

/*****Serial Transmitter****************************************************************/

module transmitter(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input			BAUD_CLK16,		/* Baud 16x Clock Enable */
	input			BREAKSRT,		/* Start Break */
	input			BREAKSTP,		/* Stop Break */
	input [1:0]	CHRL,				/* Character Length Selection */
	input			CLK,				/* USART Clock 100 MHZ */
	input			MODE9,			/* 9 bit Mode */
	input [1:0]	NBSTOP,			/* Number of Stop Bits */
	input [2:0]	PAR,				/* Parity Type */
	input			RESET,			/* System Reset */
	input			SYNC,				/* Synchronous Mode */
	input [8:0]	TX_DATA,			/* Transmit Data */
	input			TX_SRT,			/* Start Transmitter */
	/* OUTPUT PORTS */
	output		IDLE,				/* Idle */
	output		TEOT,				/* End of Transmit */
	output		TX,				/* Serial Transmit */
	output		TXRDY				/* Transmitter Ready */
	);

/*****Signals****************************************************************/

	wire			BAUD16TICK;		/* 16 Baud 16x Tick */ 
	wire			BAUD8TICK;		/* 8 Baud 16x Tick */
	wire [3:0]	CHARLEN;			/* Character Length */
	wire [2:0]	STOPLEN;			/* Stop Length */
	wire			TXDATA_XOR;		/* Data XORed */
	wire			TXPAR_CALC;		/* Parity */

/*****Registers****************************************************************/

	reg [3:0]	bitcnt;			/* Bit Counter */
	reg [4:0]	idle_count;		/* Idle Counter */
	reg			rsttick;			/* Reset Tick */
	reg [2:0]	stopcnt;			/* Stop Counter */
	reg			txbreak;			/* Transmit Break */
	reg			txpar;			/* Transmit Parity */
	reg [8:0]	txdata;			/* Transmit Data */
	reg			txrdyr;			/* Transmit Ready */
	reg			txreg;			/* Transmit Bit */
	reg			txsrt;			/* Transmit Start */
	reg			txstop;			/* Transmit Stop */

/*****Initialization****************************************************************/

	initial
	begin
		bitcnt = 4'b0;
		idle_count = 5'b0;
		rsttick = 1'b0;
		stopcnt = 3'b0;
		txbreak = 1'b0;
		txpar = 1'b0;
		txrdyr = 1'b1;
		txreg = 1'b1;
		txsrt = 1'b0;
		txstop = 1'b0;
		txdata = 9'b0;
	end

/*********************************************************************/

	assign IDLE 	=	idle_count[4];
	assign TEOT 	=	(BAUD8TICK & (idle_count == 5'h0F));
	assign TX 		=	txreg;
	assign TXRDY 	=	txrdyr;

/*****Character Length****************************************************************/

	assign CHARLEN =	(MODE9 == 1'b1)?	4'b1001:
							(CHRL == 2'b00)?	4'b0101:
							(CHRL == 2'b01)?	4'b0110:
							(CHRL == 2'b10)?	4'b0111:
							(CHRL == 2'b11)?	4'b1000:
													4'b1000;

/*****Stop Length****************************************************************/

	assign STOPLEN = 	(NBSTOP == 2'b00)?							3'b100:
							((NBSTOP == 2'b01) && (SYNC == 1'b0))?	3'b101:
							((NBSTOP == 2'b01) && (SYNC == 1'b1))?	3'b100:
							(NBSTOP == 2'b10)?							3'b110:
																				3'b110;

/*****Calculate Parity****************************************************************/

	assign TXDATA_XOR = txdata[0] ^ txdata[1] ^ txdata[2] ^ txdata[3] ^ txdata[4] ^ txdata[5] ^ txdata[6] ^ txdata[7] ^ txdata[8];
	assign TXPAR_CALC = 	(PAR == 3'b000)?	(TXDATA_XOR == 1'b0):	
								(PAR == 3'b001)?	(TXDATA_XOR == 1'b1):
								(PAR == 3'b010)?	1'b0:
								(PAR == 3'b011)?	1'b1:
														1'b0;

/*****Baud Tick****************************************************************/

	baudtick bt(
		.BAUD_CLK16(BAUD_CLK16),
		.BAUD16TICK(BAUD16TICK),
		.BAUD8TICK(BAUD8TICK),
		.CLK(CLK),
		.RESET(RESET),
		.RESET_INT(rsttick)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET)
		begin
			if (txrdyr == 1'b0)
			begin
				/* Transmitter Active */
				idle_count = 5'b0;
			end
			else
			begin
				/* Transmitter Stopped */
				if ((BAUD8TICK) && (~idle_count[4]))
				begin
					idle_count = idle_count + 1;
				end
			end
		end
		else
		begin
			/* Reset */
			idle_count = 5'b0;
		end
	end

	always@(posedge CLK)
	begin
		if ((BREAKSTP) && (txrdyr))
		begin
			/* Transmit Break */
			txbreak = 1'b0;
		end
		casex({RESET,txrdyr,(TX_SRT | BREAKSRT),rsttick,txsrt,BAUD8TICK,BAUD16TICK,txpar,txstop})
			9'b0xxxxxxxx	:		begin
											/* Reset */
											txreg = 1'b1;
											rsttick = 1'b0;
											txrdyr = 1'b1;
											txsrt = 1'b0;
											txstop = 1'b0;
											txbreak = 1'b0;
											txpar = 1'b0;
											bitcnt = 4'b0;
											stopcnt = 3'b0;
											txdata = 9'b0;
										end
			9'b11100xxxx	:		begin
											/* Start Transmitter */
											txrdyr = 1'b0;
											rsttick = 1'b1;
											txsrt = 1'b1;
											txreg = 1'b0;
											txstop = 1'b0;
											txpar = 1'b0;
											bitcnt = 4'b0;
											stopcnt = 3'b0;
											if (BREAKSRT)
											begin
												txbreak = 1'b1;
												txdata = 9'b0;
											end
											else 
											begin
												txbreak = 1'b0;
												txdata = TX_DATA;
											end
										end
			9'bxxx1xxxxx		:	begin
											/* Reset Tick */
											rsttick = 1'b0;
										end
			9'b10x011x00		:	begin
											/* Transmitter Started */
											txsrt = 1'b0;
										end
			9'b10x00x100		:	begin
											/* Send Bits */
											if (txbreak) txreg = 1'b0;
											else txreg = txdata[bitcnt];
											bitcnt = bitcnt + 1;
											if (bitcnt == CHARLEN)
											begin
												if (PAR[2:1] == 2'b10)
												begin
													txstop = 1'b1;
												end
												else txpar = 1'b1;
											end
										end
			9'b10x00x110		:	begin
											/* Send Parity */
											if (txbreak) txreg = 1'b0;
											else txreg = TXPAR_CALC;
											txpar = 1'b0;
											txstop = 1'b1;
										end
			9'b10x001x01		:	begin
											/* Send Stop */
											stopcnt = stopcnt + 1;
											if (stopcnt > 3'b001) 
											begin
												if (txbreak) txreg = 1'b0;
												else txreg = 1'b1;
											end
											if (stopcnt == STOPLEN)
											begin
												txrdyr = 1'b1;
												txreg = 1'b1;
												txstop = 1'b0;
											end
										end
			default				:	begin
											/* Do Nothing */
										end
		endcase
	end
	
endmodule

/*****Serial Receiver****************************************************************/

module receiver(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				BAUD_CLK16,	/* Baud 16x Clock Enable */
	input [1:0]		CHRL,			/* Character Length Selection */
	input				CLK,			/* USART Clock 100 MHZ */
	input				MODE9,		/* 9 bit Mode */
	input [1:0]		NBSTOP,		/* Number of Stop Bits */
	input [2:0]		PAR,			/* Parity Type */
	input				RESET,		/* System Reset */
	input				RX,			/* Serial Receive */
	input				SYNC,			/* Synchronous Mode */
	/* OUTPUT PORTS */
	output			BREAK,		/* Received Break */
	output			FRAME,		/* Frame Error */
	output			IDLE,			/* Idle */
	output			PARE,			/* Parity Error */
	output			REOT,			/* End of Receive */
	output [8:0]	RX_DATA,		/* Received Data */
	output			RXRDY			/* Receiver Ready */
	);

/*****Signals****************************************************************/
	
	wire			BAUD16TICK;		/* 16 Baud 16x Ticks */
	wire			BAUD8TICK;		/* 8 Baud 16x Ticks */
	wire [3:0]	CHARLEN;			/* Character Length */
	wire			RXDATA_OR;		/* ORed Data */
	wire			RXDATA_XOR;		/* XORed Data */
	wire			RXPAR_CALC;		/* Parity */
	wire [2:0]	STOPLEN;			/* Stop Length */

/*****Registers****************************************************************/

	reg [3:0]	bitcnt;		/* Bit Counter */
	reg [4:0]	idle_count;	/* Idle Counter */
	reg			rsttick;		/* Reset Tick */
	reg			rxbreak;		/* Received Break */
	reg [1:0]	rxcnt;		/* Start Detection */
	reg [8:0]	rxdata;		/* Received Data */
	reg			rxframe;		/* Frame Error */
	reg			rxlow;		/* Start Bit Flag */
	reg			rxneg;		/* Negedge Flag */
	reg			rxpar;		/* Parity Flag */
	reg			rxpare;		/* Parity Error */
	reg			rxrdyr;		/* Receiver Ready */
	reg			rxstart;		/* Receiver Started */
	reg			rxstop;		/* Receiver Stopped */
	reg [2:0]	stopcnt;		/* Stop Counter */

/*****Initialization****************************************************************/

	initial
	begin
		bitcnt = 4'b0;
		idle_count = 5'b0;
		rsttick = 1'b0;
		rxbreak = 1'b0;
		rxcnt = 2'b11;
		rxdata = 9'b0;
		rxframe = 1'b0;
		rxlow = 1'b0;
		rxneg = 1'b0;
		rxpar = 1'b0;
		rxrdyr = 1'b0;
		rxstart = 1'b0;
		rxstop = 1'b0;
		rxpare = 1'b0;
		stopcnt = 3'b0;
	end

/*********************************************************************/

	assign BREAK	=	rxbreak;
	assign FRAME	=	rxframe;
	assign IDLE		=	idle_count[4];
	assign PARE		=	rxpare;
	assign REOT		=	(BAUD8TICK & (idle_count == 5'h0F));
	assign RX_DATA	=	rxdata;
	assign RXRDY	=	rxrdyr;

/*****Character Length****************************************************************/

	assign CHARLEN =	(MODE9 == 1'b1)?	4'b1001:
							(CHRL == 2'b00)?	4'b0101:
							(CHRL == 2'b01)?	4'b0110:
							(CHRL == 2'b10)?	4'b0111:
							(CHRL == 2'b11)?	4'b1000:
													4'b1000;

/*****Stop Length****************************************************************/

	assign STOPLEN = 	(NBSTOP == 2'b00)?							3'b010:
							((NBSTOP == 2'b01) && (SYNC == 1'b0))?	3'b011:
							((NBSTOP == 2'b01) && (SYNC == 1'b1))?	3'b010:
							(NBSTOP == 2'b10)?							3'b100:
																				3'b100;

/*****Calculate Parity****************************************************************/

	assign RXDATA_OR	=	rxdata[0] | rxdata[1] | rxdata[2] | rxdata[3] | rxdata[4] | rxdata[5] | rxdata[6] | rxdata[7] | rxdata[8];
	assign RXDATA_XOR	=	rxdata[0] ^ rxdata[1] ^ rxdata[2] ^ rxdata[3] ^ rxdata[4] ^ rxdata[5] ^ rxdata[6] ^ rxdata[7] ^ rxdata[8];
	assign RXPAR_CALC	=	(PAR == 3'b000)?	(RXDATA_XOR == 1'b0):
								(PAR == 3'b001)?	(RXDATA_XOR == 1'b1):
								(PAR == 3'b010)?	1'b0:
								(PAR == 3'b011)?	1'b1:
														1'b0;

/*****Baud Tick****************************************************************/

	baudtick bt(
		.CLK(CLK),
		.BAUD_CLK16(BAUD_CLK16),
		.RESET(RESET),
		.RESET_INT(rsttick),
		.BAUD8TICK(BAUD8TICK),
		.BAUD16TICK(BAUD16TICK)
		);

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET)
		begin
			if (rxstart)
			begin
				/* Receiver Active */
				idle_count = 5'b0;
			end
			else
			begin
				/* Receiver Stoped */
				if ((BAUD8TICK) && (~idle_count[4]))
				begin
					idle_count = idle_count + 1;
				end
			end
		end
		else
		begin
				/* Reset */
				idle_count = 5'b0;
		end
	end

	always@(posedge CLK)
	begin
		casex({RESET,RX,rxneg,rsttick,rxlow,BAUD8TICK,rxstart,BAUD16TICK,rxpar,rxstop,rxrdyr})
			11'b0xxxxxxxxxx	:	begin	
											/* Reset */
											rxstart = 1'b0;
											rsttick = 1'b0;
											rxneg = 1'b0;
											rxlow = 1'b0;
											rxpar = 1'b0;
											rxstop = 1'b0;
											rxrdyr = 1'b0;
											rxframe = 1'b0;
											rxpare = 1'b0;
											rxbreak = 1'b0;
											rxcnt = 2'b11;
											stopcnt = 3'b0;
											bitcnt = 4'b0;
											rxdata = 9'b0;
										end
			11'b10000x0x00x	:	begin
											/* Start Detected */
											if (rxcnt == 2'b10)
											begin
												rxrdyr = 1'b0;
												rxneg = 1'b1;
											end
										end
			11'b10100x0x000	:	begin
											/* Reset Tick */
											rxneg = 1'b0;
											rxlow = 1'b1;
											rsttick = 1'b1;
										end
			11'bxxx1xxxxxxx	:	begin
											/* Reset Tick */
											rsttick = 1'b0;
										end	
			11'b1000110x000	:	begin
											/* Start Receiver */
											rxstart = 1'b1;
											rsttick = 1'b1;
											rxlow = 1'b0;
											rxrdyr = 1'b0;
											bitcnt = 4'b0;
											stopcnt = 3'b0;
											rxdata = 9'b0;
											rxframe = 1'b0;
											rxbreak = 1'b0;
											rxpare = 1'b0;
										end
			11'b1x000x11000	:	begin
											/* Capture Bits */
											rxdata = {RX,rxdata[8:1]};
											bitcnt = bitcnt + 1;
											if (bitcnt == CHARLEN)
											begin
												if (PAR[2:1] == 2'b10)
												begin
													rxstop = 1'b1;
												end
												else rxpar = 1'b1;
											end
										end
			11'b1x000x11100	:	begin
											/* Capture Parity */
											if (PAR[2] == 1'b0)
											begin
												if (RX == RXPAR_CALC)
												begin
												end
												else
												begin
													rxpare = 1'b1; 
												end
											end
											rxpar = 1'b0;
											rxstop = 1'b1;
										end
			11'bxxxxxxxxxx1	:	begin
											/* Toggle Receiver Ready */
											rxrdyr = 1'b0;
										end
			11'b1x00011x010	:	begin
											/* Capture Stop */
											stopcnt = stopcnt + 1;
											if ((~RX) & (RXDATA_OR) & (stopcnt > 3'b001)) rxframe = 1'b1;
											if ((~RX) & (~RXDATA_OR) & (stopcnt > 3'b001)) rxbreak = 1'b1;
											/* Finish Shifting Data */
											if (stopcnt == 3'b001)
											begin
												if (CHARLEN == 4'b1000) rxdata = {1'b0,rxdata[8:1]};
												if (CHARLEN == 4'b0111) rxdata = {2'b0,rxdata[8:2]};
												if (CHARLEN == 4'b0110) rxdata = {3'b0,rxdata[8:3]};
												if (CHARLEN == 4'b0101) rxdata = {4'b0,rxdata[8:4]};
											end
											if (stopcnt == 3'b010)
											begin
												/* Toggle Receiver Ready */
												rxrdyr = 1'b1;
											end
											if (stopcnt == STOPLEN)
											begin
												rxstart = 1'b0;
												rxstop = 1'b0;
											end
										end
			default				:	begin
											/* Do Nothing */
										end
		endcase
		rxcnt = {rxcnt[0],RX};
	end
	
endmodule

/*****Baud Tick****************************************************************/

module baudtick(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input		BAUD_CLK16,		/* Baud 16x Clock Enable */
	input		CLK,				/* USART Clock 100 MHZ */
	input		RESET,			/* System Reset */
	input		RESET_INT,		/* Internal Reset */
	/* OUTPUT PORTS */
	output	BAUD16TICK,		/* 16 Baud 16x Ticks Flag */
	output	BAUD8TICK		/* 8 Baud 16x Ticks Flag */
	);

/*****Signals****************************************************************/

	wire BAUD16TICK_0;	/* 16 Baud 16x Ticks */
	wire BAUD8TICK_0;		/* 8 Baud 16x Ticks */

/*****Registers****************************************************************/

	reg			rstflag;	/* Internal Reset Flag */
	reg [3:0]	counter;	/* Tick Counter */

/*****Initialization****************************************************************/

	initial 
	begin
		rstflag = 1'b1;
		counter = 4'b0;
	end

/*********************************************************************/

	assign BAUD16TICK		=	BAUD16TICK_0 & BAUD_CLK16;
	assign BAUD16TICK_0	=	counter[3] & BAUD8TICK_0;
	assign BAUD8TICK		=	BAUD8TICK_0 & BAUD_CLK16;
	assign BAUD8TICK_0	=	counter[2] & counter[1] & counter[0];

/*********************************************************************/

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			rstflag = 1'b1;
		end
		else
		begin
			if (counter == 4'b0)
			begin
				/* Reset Acknowledged */
				rstflag = 1'b1;
			end
			if (RESET_INT)
			begin
				/* Internal Reset */
				rstflag = 1'b0;
			end
		end
	end

	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			counter = 4'b0;
		end
		else
		begin
			if (BAUD_CLK16)
			begin
				if (rstflag == 1'b0)
				begin
					/* Internal Reset */
					counter = 4'b0;
				end
				else 
				begin
					/* Increment Counter */
					counter = counter + 1;
				end
			end
		end
	end

endmodule

/*****Baud Clock Generator****************************************************************/

module baudclk_generator(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input				CLK,				/* USART Clock 100 MHZ */
	input [31:0]	CLK_DIV,			/* Clock Divider */
	input				CLK_EXT,			/* External Clock */
	input [3:0]		CLK_SEL,			/* Clock Selection */
	input				RESET,			/* System Reset */
	input				SYNC,				/* Synchronous Mode */
	/* OUTPUT PORTS */	
	output			BAUD_CLK,		/* Baud Clock */
	output			BAUD_CLK_EN,	/* Baud Clock Enable */
	output			BAUD_CLK16,		/* Baud 16x Clock */
	output			BAUD_CLK16_EN	/* Baud 16x Clock Enable */
	);

/*****Signals****************************************************************/
	
	wire CLK_EN;			/* Divide by Power of 2 Clock Enable */
	wire CLK_ENBD;			/* Frequency Divieder Clock Enable */
	wire CLK_OUT_0;		/* Divide by Power of 2 Clock */
	wire CLK_OUT_1;		/* Frequency Divider Clock */
	wire CLK_OUT_D16;		/* Output 16x Clock */
	wire CLK_OUT_D16_EN;	/* Output 16x Clock Enable */
	wire CLK_OUT_EN_1;	/* Outptu Clock Enable */

/*****Registers****************************************************************/

	reg [3:0]	bdcounter;		/* Divide 16 Counter */
	reg			clk_outr;		/* Output Clock */
	reg [6:0]	clkcounter;		/* Power of 2 Divider */
	reg [31:0]	clkdiv_reg;		/* Clock Divider */
	reg [31:0]	freqcounter;	/* Frequency Divder Counter */

/*****Initialization****************************************************************/

	initial
	begin
		clk_outr = 1'b0;
		bdcounter = 4'b0;
		clkcounter = 7'b0;
		clkdiv_reg = 32'b0;
		freqcounter = 32'b0;
	end

/*********************************************************************/
														
	assign CLK_ENBD			=	~| freqcounter;
	assign CLK_OUT_D16		=	bdcounter[3];
	assign CLK_OUT_D16_EN	=	bdcounter[3] & bdcounter[2] & bdcounter[1] & bdcounter[0] & CLK_EN & CLK_ENBD & clk_outr;

/*****Source Clock Selection (Power of 2 Divide)****************************************************************/
	
	MUX1_8to1 mux0(
		.a0(CLK),
		.a1(clkcounter[0]),
		.a2(clkcounter[1]),
		.a3(clkcounter[2]),
		.a4(clkcounter[3]),
		.a5(clkcounter[4]),
		.a6(clkcounter[5]),
		.a7(clkcounter[6]),
		.sel(CLK_SEL[2:0]),
		.out(CLK_OUT_0)
		);
		
	MUX1_8to1 mux1(
		.a0(1'b1),
		.a1(clkcounter[0]),
		.a2(clkcounter[1] & clkcounter[0]),
		.a3(clkcounter[2] & clkcounter[1] & clkcounter[0]),
		.a4(clkcounter[3] & clkcounter[2] & clkcounter[1] & clkcounter[0]),
		.a5(clkcounter[4] & clkcounter[3] & clkcounter[2] & clkcounter[1] & clkcounter[0]),
		.a6(clkcounter[5] & clkcounter[4] & clkcounter[3] & clkcounter[2] & clkcounter[1] & clkcounter[0]),
		.a7(clkcounter[6] & clkcounter[5] & clkcounter[4] & clkcounter[3] & clkcounter[2] & clkcounter[1] & clkcounter[0]),
		.sel(CLK_SEL[2:0]),
		.out(CLK_EN)
		);

/*****Source Clock Selection (Frequency Divider)****************************************************************/

	MUX1_4to1 mux6(
		.a0(1'b0),
		.a1(CLK_OUT_0),
		.a2(clk_outr),
		.a3(clk_outr),
		.sel({(| clkdiv_reg),clkdiv_reg[0]}),
		.out(CLK_OUT_1)
		);
		
	MUX1_4to1 mux7(
		.a0(1'b0),
		.a1(CLK_EN),
		.a2(CLK_EN & CLK_ENBD & clk_outr),
		.a3(CLK_EN & CLK_ENBD & clk_outr),
		.sel({(| clkdiv_reg),clkdiv_reg[0]}),
		.out(CLK_OUT_EN_1)
		);

/*****Baud Clock 16x****************************************************************/

	MUX1_2to1 mux2(
		.a0(CLK_OUT_1),
		.a1(CLK_EXT),
		.sel(SYNC & CLK_SEL[3]),
		.out(BAUD_CLK16)
		);
		
	MUX1_2to1 mux3(
		.a0(CLK_OUT_EN_1),
		.a1(CLK_EXT),
		.sel(SYNC & CLK_SEL[3]),
		.out(BAUD_CLK16_EN)
		);

/*****Baud Clock****************************************************************/
		
	MUX1_2to1 mux4(
		.a0(CLK_OUT_D16),
		.a1(BAUD_CLK16),
		.sel(SYNC),
		.out(BAUD_CLK)
		);
		
	MUX1_2to1 mux5(
		.a0(CLK_OUT_D16_EN),
		.a1(BAUD_CLK16_EN),
		.sel(SYNC),
		.out(BAUD_CLK_EN)
		);
		
/*********************************************************************/

	/* Divide Source Clock by Power of 2 */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			clkdiv_reg = 32'b0;
			clkcounter = 7'b0;
		end
		else
		begin
			/* Latch Divider */
			clkdiv_reg = CLK_DIV;
			/* Increment Counter */ 
			clkcounter = clkcounter + 1;
		end
	end

	/* Divide Source Clock by Frequency Divider */
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			freqcounter = 32'b0;
			clk_outr = 1'b0;
		end
		else
		begin
			if (CLK_EN)
			begin
				/* Source Clock Enable */
				if (CLK_ENBD)
				begin
					/* Reset Counter */
					freqcounter[31:1] = clkdiv_reg[31:1];
					freqcounter[0] = 1'b0;
					clk_outr = ~clk_outr;
				end
				else
				begin
					/* Decrement Counter */
					freqcounter = freqcounter - 2;
				end
			end
		end
	end
	
	always@(posedge CLK)
	begin
		if (RESET == 1'b0)
		begin
			/* Reset */
			bdcounter = 4'b0;
		end
		else
		begin
			if (SYNC & CLK_SEL[3])
			begin
				if (clkdiv_reg == 32'b0)
				begin
					/* no divide by 16*/
					bdcounter = 4'b0;
				end
				else if (clkdiv_reg == 32'b1)
				begin
					if (CLK_EN)
					begin
						/* power of 2 divde, no freqency divide, divide by 16 */
						bdcounter = bdcounter + 1;
					end
				end
				else
				begin
					if ((CLK_EN) && (CLK_ENBD) && (clk_outr))
					begin
						/* power of 2 divde, freqency divide, divide by 16 */
						bdcounter = bdcounter + 1;
					end
				end
			end
			else
			begin
				if (CLK_EXT)
				begin
					/* Use External Clock */
					bdcounter = bdcounter + 1;
				end
			end
		end
	end
	
endmodule
