/* Copyright (c) Microsoft Corporation. All rights reserved.			*/

/*********************************************************************/
/* Company:		Microsoft Research (MSR)										*/
/*					Microsoft Corporation											*/
/* Group:		Embedded Systems Group											*/
/* Engineer: 	Richard Neil Pittman												*/
/*																							*/
/* Project Name:	eMIPS Dynamically Extensible Processor					*/
/* Design Name:	eMIPSv1															*/
/* Module Name:	top_dcm															*/
/* Target Devices:	Xilinx Virtex 4 FPGA (xc4vlx25-10ff668)			*/
/* Tool versions:		8.2i sp 3 and 8.2i sp1 PR								*/
/* Description:																		*/
/*																							*/
/* Dependencies:																		*/
/*																							*/
/* Revision:																			*/
/* Revision	0.0	-	Pre Release													*/
/* Revision	1.0	-	First General Release									*/
/* Revision	1.1	-	Bug Fixes, see Manual									*/
/* Additional Comments:																*/
/*																							*/
/*********************************************************************/

`timescale 1ns / 1ps

module top_dcm(
/*****Ports****************************************************************/
	/* INPUT PORTS */
	input CLKIN_IN,				/* Input Source Clock, Board Clock 100 MHZ */
	input CLKPER_IN,				/* Input System Ace Clock 33 MHZ */
	input RESET,					/* Hardware Reset */
	input SLOW_FASTN,				/* Select System Clock Speed 100 or 50 MHZ */
	input SLOW_FASTN2,			/* Select SRAM Clock Speed 100 MHZ or 50 MHZ */
	input SRAMCLK_FB_IN, 		/* SRAM Clock Feedback 50 - 100 MHZ */
	/* OUTPUT PORTS */
	output CLKIN_IBUFG,			/* Buffered Source Clock 100 MHZ */
	output CLKPER,					/* Buffered System Ace Clock 33 MHZ */
	output LOCKED_OUT,			/* Digital Clock Manager Lock */
	output MEMCLK,					/* Buffered Memory Interface Clock 50 - 100 MHZ */
	output SRAMCLK,				/* Buffered & Deskewed Output SRAM Clock 50 - 100 MHZ */
	output SRAMCLK_INT,			/* Buffered & Deskewed Internal SRAM Clock 50 - 100 MHZ */
	output SYSCLK,					/* Buffered & Deskewed System Clock 50 - 100 MHZ */
	output TMRCLK,					/* Buffered & Deskewed Timer Clock 10 MHZ */
	output USARTCLK				/* Buffered & Deskewed USART Clock 100 MHZ */
	);
	
/*****Signals****************************************************************/

   wire 			CLK0_BUF_DCM;		/* Source Clock Buffered by System DCM, 100 MHZ */
	wire 			CLK0_BUF_PMCD;		/* Source Clock Buffered by System PMCD, 100 MHZ */
   wire 			CLK0_D2;				/* Source Clock Divided by 2, 50 MHZ */
	wire 			CLK0_D4;				/* Source Clock Divided by 4, 25 MHZ */
	wire 			CLK0_D8;				/* Source Clock Divided by 8, 12.5 MHZ */
   wire 			CLKDV_BUF_DCM;		/* Source Clock Divided by 10 Buffered by System DCM, 10 MHZ*/
	wire 			CLKDV_BUF_PMCD;	/* Source Clock Divided by 10 Buffered by System PMCD, 10 MHZ*/
   wire 			CLKFB_IN;			/* Source Clock Internal Feedback 100 MHZ */
	wire 			CLKPER_IBUFG;		/* System Ace Clock Buffered 33 MHZ */
   wire [6:0]	GND1;					/* 7-bit GND */
   wire [15:0]	GND2;					/* 16-bit GND */
   wire			GND3;					/* 1-bit GND */
	wire 			LOCKED_SYS;			/* System DCM Lock */
	wire 			LOCKED_SRAM;		/* SRAM DCM Lock */
	wire			SRAMCLK_FB_IBUFG;	/* SRAM Clock Feedback Buffered 50 - 100 MHZ */
	
/*********************************************************************/

   assign GND1 = 7'b0000000;
   assign GND2 = 16'b0000000000000000;
   assign GND3 = 0;
	assign LOCKED_OUT = LOCKED_SYS & LOCKED_SRAM;

/*****Clock Input Buffers****************************************************************/

	IBUFG IBUFG_CLKIN_IN(
		.I(CLKIN_IN), 
		.O(CLKIN_IBUFG)
		);
		
	IBUFG IBUFG_SRAMCLK_FB(
		.I(SRAMCLK_FB_IN), 
		.O(SRAMCLK_FB_IBUFG)
		);
		
	IBUFG IBUFG_CLKPER(
		.I(CLKPER_IN),
		.O(CLKPER_IBUFG)
		);
		
/*****Global Clock Buffers****************************************************************/
		
	BUFG BUFG_CLKPER(
		.I(CLKPER_IBUFG),
		.O(CLKPER)
		);	

   BUFG BUFG_CLKFB(
		.I(CLK0_BUF_PMCD), 
		.O(CLKFB_IN)
		);

   BUFG BUFG_TMR(
		.I(CLKDV_BUF_PMCD), 
		.O(TMRCLK)
		);
		
	BUFG BUFG_USART(
		.I(CLK0_BUF_PMCD),
		.O(USARTCLK)
		);

   BUFGMUX_VIRTEX4 BUFG_SYS(	// 100 or 50MHz
		.O(SYSCLK),					// Clock MUX output
		.I0(CLK0_BUF_PMCD),		// Clock0 input
		.I1(CLK0_D2),				// Clock1 input
		.S(SLOW_FASTN)				// Clock select input
		);

   BUFGMUX_VIRTEX4 BUFG_SRAM(	// 100 or 50MHz
		.O(SRAMCLK_INT),			// Clock MUX output
		.I0(CLK0_BUF_PMCD),		// Clock0 input
		.I1(CLK0_D2),				// Clock1 input
		.S(SLOW_FASTN2)			// Clock select input
		);
	
	BUFGMUX_VIRTEX4 BUFG_MEM (	// 100 or 50MHz
		.O(MEMCLK),					// Clock MUX output
		.I0(CLK0_BUF_PMCD),		// Clock0 input
		.I1(CLK0_D2),				// Clock1 input
		.S(SLOW_FASTN)				// Clock select input
		);
		
/*****System DCM****************************************************************/
   
   DCM_ADV DCM_ADV_SYS (
		.CLKFB(CLKFB_IN), 
		.CLKIN(CLKIN_IBUFG), 
		.DADDR(GND1[6:0]), 
		.DCLK(GND3), 
		.DEN(GND3), 
		.DI(GND2[15:0]), 
		.DWE(GND3), 
		.PSCLK(GND3), 
		.PSEN(GND3), 
		.PSINCDEC(GND3), 
		.RST(~RESET), 
		.CLKDV(CLKDV_BUF_DCM), 
		.CLKFX(), 
		.CLKFX180(), 
		.CLK0(CLK0_BUF_DCM), 
		.CLK2X(), 
		.CLK2X180(), 
		.CLK90(), 
		.CLK180(), 
		.CLK270(), 
		.DO(), 
		.DRDY(), 
		.LOCKED(LOCKED_SYS), 
		.PSDONE()
		);
	defparam DCM_ADV_SYS.CLK_FEEDBACK = "1X";
	defparam DCM_ADV_SYS.CLKDV_DIVIDE = 10.0;
	defparam DCM_ADV_SYS.CLKFX_DIVIDE = 1;
	defparam DCM_ADV_SYS.CLKFX_MULTIPLY = 4;
	defparam DCM_ADV_SYS.CLKIN_DIVIDE_BY_2 = "FALSE";
	defparam DCM_ADV_SYS.CLKIN_PERIOD = 10.0;
	defparam DCM_ADV_SYS.CLKOUT_PHASE_SHIFT = "NONE";
	defparam DCM_ADV_SYS.DCM_AUTOCALIBRATION = "TRUE";
	defparam DCM_ADV_SYS.DCM_PERFORMANCE_MODE = "MAX_SPEED";
	defparam DCM_ADV_SYS.DESKEW_ADJUST = "SYSTEM_SYNCHRONOUS";
	defparam DCM_ADV_SYS.DFS_FREQUENCY_MODE = "LOW";
	defparam DCM_ADV_SYS.DLL_FREQUENCY_MODE = "LOW";
	defparam DCM_ADV_SYS.DUTY_CYCLE_CORRECTION = "TRUE";
	defparam DCM_ADV_SYS.FACTORY_JF = 16'hF0F0;
	defparam DCM_ADV_SYS.PHASE_SHIFT = 0;
	defparam DCM_ADV_SYS.STARTUP_WAIT = "FALSE";
	
/*****System PMCD****************************************************************/
   
   PMCD #(
		.EN_REL("FALSE"),				// TRUE/FALSE to allow synchronous deassertion of RST
		.RST_DEASSERT_CLK("CLKA")	// Reset syncronization to which clock: CLKA, CLKB, CLKC or CLKD
		)
	PMCD_SYS (
		.CLKA1(CLK0_BUF_PMCD),		// Output CLKA divided by 1
		.CLKA1D2(CLK0_D2),			// Output CLKA divided by 2
		.CLKA1D4(CLK0_D4),			// Output CLKA divided by 4
		.CLKA1D8(CLK0_D8),			// Output CLKA divided by 8
		.CLKB1(CLKDV_BUF_PMCD),		// Output phase matched CLKB
		.CLKC1(),						// Output phase matched CLKC
		.CLKD1(),						// Output phase matched CLKD
		.CLKA(CLK0_BUF_DCM),       // Input CLKA
		.CLKB(CLKDV_BUF_DCM),		// Input CLKB
		.CLKC(1'b0),					// Input CLKC
		.CLKD(1'b0),					// Input CLKD
		.REL(1'b1),						// PCMD release input
		.RST(~RESET)					// Active high reset input
		);
		
/*****SRAM DCM****************************************************************/
		
	DCM_ADV DCM_ADV_SRAM(
		.CLKFB(SRAMCLK_FB_IBUFG),
		.CLKIN(SRAMCLK_INT),
		.DADDR(GND1[6:0]),
		.DCLK(GND3),
		.DEN(GND3),
		.DI(GND2[15:0]), 
		.DWE(GND3), 
		.PSCLK(GND3), 
		.PSEN(GND3),
		.PSINCDEC(GND3),
		.RST(~RESET | ~LOCKED_SYS),
		.CLKDV(),
		.CLKFX(),
		.CLKFX180(),
		.CLK0(SRAMCLK),
		.CLK2X(),
		.CLK2X180(),
		.CLK90(),
		.CLK180(),
		.CLK270(),
		.DO(),
		.DRDY(),
		.LOCKED(LOCKED_SRAM),
		.PSDONE()
		);
	defparam DCM_ADV_SRAM.CLK_FEEDBACK = "1X";
	defparam DCM_ADV_SRAM.CLKDV_DIVIDE = 2.0;
	defparam DCM_ADV_SRAM.CLKFX_DIVIDE = 1;
	defparam DCM_ADV_SRAM.CLKFX_MULTIPLY = 4;
	defparam DCM_ADV_SRAM.CLKIN_DIVIDE_BY_2 = "FALSE";
	defparam DCM_ADV_SRAM.CLKIN_PERIOD = 10.0;
	defparam DCM_ADV_SRAM.CLKOUT_PHASE_SHIFT = "NONE";
	defparam DCM_ADV_SRAM.DCM_AUTOCALIBRATION = "TRUE";
	defparam DCM_ADV_SRAM.DCM_PERFORMANCE_MODE = "MAX_SPEED";
	defparam DCM_ADV_SRAM.DESKEW_ADJUST = "SOURCE_SYNCHRONOUS";
	defparam DCM_ADV_SRAM.DFS_FREQUENCY_MODE = "LOW";
	defparam DCM_ADV_SRAM.DLL_FREQUENCY_MODE = "LOW";
	defparam DCM_ADV_SRAM.DUTY_CYCLE_CORRECTION = "TRUE";
	defparam DCM_ADV_SRAM.FACTORY_JF = 16'hF0F0;
	defparam DCM_ADV_SRAM.PHASE_SHIFT = 0;
	defparam DCM_ADV_SRAM.STARTUP_WAIT = "FALSE";

endmodule

 

