// Copyright (C) Microsoft Corporation. All rights reserved.
// ---------------------------------------------------------------------------------------
// emips_gdb.cpp
//
// eMIPS Debugging: Extension-to-remote protocol translator for GDB.
//
// ---------------------------------------------------------------------------------------

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

const char *MyName = "Emips2Gdb";

typedef UINT32 ADDRESS;

typedef struct {
    bool fSpecial;
    unsigned char nReg;
} EMIPSREG;


// BREAK/WATCH POINTS AVAILABLE
#define WBP_MAX 2

// HARDWARE WATCHPOINTS
#define HW_WBP true


// Gdb Mips registers coding
#define GP0_REGNUM      	0         
#define GPLAST_REGNUM   	31
#define PS_REGNUM       	32        
#define LO_REGNUM      	 	33      
#define HI_REGNUM       	34        
#define BADVADDR_REGNUM 	35        
#define CAUSE_REGNUM    	36        
#define PC_REGNUM       	37        
#define FP0_REGNUM      	38        
#define FPLAST_REGNUM   	70        
#define FPS_REGNUM      	71        
#define FPIR_REGNUM     	72        
#define UNUSED_REGNUM   	73        
#define FIRST_CP0_REGNUM 	74       
#define PRID_REGNUM     	89        
#define LAST_REGNUM     	89


/* FPSECIAL HARDWARE ACCESSIBLE REGISTER NUMBER */
#define PC_FSPECIAL_REGNUM       0   
#define HI_FSPECIAL_REGNUM       1
#define LO_FSPECIAL_REGNUM       2
#define PS_FSPECIAL_REGNUM		 3
#define BADVADDR_FSPECIAL_REGNUM 4
#define CAUSE_FSPECIAL_REGNUM	 5
#define FPLAST_FSPECIAL_REGNUM	 6
#define FPS_FSPECIAL_REGNUM		 7



#define FAKE_REG_HEX_VALUE	 "DEADBEEF"
/*
 * emips2gdb serial Protocol Opcodes
 */

#define GETREG		0x00
#define SETREG		0x01
#define FETCH		0x02
#define STORE		0x03
#define INTERRUPT	0x06
#define EXTENDED	0x07

/*
 * emips2gdb serial Protocol options
 */

#define SUSPEND		0x00
#define CONTINUE	0x08
#define WBPOINTS	0x08

#define FSPECIAL	0x80

 

/* Configuration options
 */
class ConfigInfo {
 public:
    ConfigInfo(int argc, char **argv);

    int DebugLevel;
    char *SerialLine;
    UINT BaudRate;

    /* Selective trace/debugging
     */
#define MAX_ERROR_LEVEL 3
    int ErrorLevel;
#define L_LINE 1
#define L_CPU 2
#define L_GDB 4
    int ModulesToTrace;
};

void Mask2Modules(int Mask)
{
    if (Mask & L_LINE) printf(" L_LINE");
    if (Mask & L_CPU) printf(" L_CPU");
    if (Mask & L_GDB) printf(" L_GDB");
    printf("\n");
}

ConfigInfo *Config = NULL;
#define TRACE(_m_,_l_,_x_) if (((_m_) & Config->ModulesToTrace) && (Config->ErrorLevel >= (_l_))) _x_
#define DPRINTF(_m_,_l_,_x_) if (((_m_) & Config->ModulesToTrace) && (Config->ErrorLevel >= (_l_))) printf _x_
#define EPRINTF(_x_) printf _x_

/* Asynchronous duplex I/O comm class.
 */
class Aio {
 public:
    Aio( HANDLE Line);
    virtual ~Aio(void);

    bool GetEvents(HANDLE *Events);
    bool GetNextChar(unsigned char *c);
    bool PutNextChar(unsigned char c);
    bool PostAndWait(DWORD *ErrorCode);

//#define BUFMAX 400
#define BUFMAX 		800
    BYTE InputBuffer[BUFMAX];

 private:
    HANDLE h;
    DWORD BytesAvailable;
    DWORD BytesRead, BytesWritten;
    OVERLAPPED WriteOver;
    OVERLAPPED ReadOver;
    HANDLE Events[2];
#define evRead 0
#define evWrite 1
    DWORD OpsPosted;
#define opRead  (1 << evRead)
#define opWrite (1 << evWrite)
};

bool Aio::GetEvents(HANDLE *pEvents)
{
    if ((Events[evRead] == INVALID_HANDLE_VALUE) ||
        (Events[evWrite] == INVALID_HANDLE_VALUE))
        return false;
    pEvents[0] = Events[evRead];
    pEvents[1] = Events[evWrite];
    return true;
}

bool Aio::PostAndWait(DWORD *ErrorCode)
{
    char *op;
    DWORD err;
    bool r;

    OpsPosted = 0;
    for (;;) {
        if ( ! (OpsPosted & opRead)) {
            op = "ReadFile";
            BytesAvailable = 0;
            r = ReadFile(h,
                         &InputBuffer[0],
                         BUFMAX,
                         &BytesRead,
                         &ReadOver);
            if (r) goto GotSomeRead;

        HandleErrors:
            err = GetLastError();
            *ErrorCode = err;

            if (err != ERROR_IO_PENDING)
                return false;

            OpsPosted |= opRead;
        }

        op = "WaitForMultiple";
        err = WaitForMultipleObjects(2,Events,FALSE,INFINITE);
        if (err == WAIT_FAILED) {
            *ErrorCode = err;
            return false;
        }

        if (err == evRead) {
            op = "GetOverlappedResult(r)";
            if (GetOverlappedResult(h,
                                    &ReadOver,
                                    &BytesRead,
                                    TRUE)) {        /* wait */
            GotSomeRead:            
                ReadOver.Offset += BytesRead;
                OpsPosted &= ~opRead;

                // are we there yet?
                BytesAvailable = BytesRead;
                BytesRead = 0;
                DPRINTF(L_GDB,0,("%s has %d (+%d)\n",MyName,BytesAvailable,BytesRead));
                return true;

            } else {
                goto HandleErrors;
            }
        } else if (err == evWrite) {
            op = "GetOverlappedResult(w)";
            if (GetOverlappedResult(h,
                                    &WriteOver,
                                    &BytesWritten,
                                    TRUE)) {        /* wait */
                WriteOver.Offset += BytesWritten;
                OpsPosted &= ~opWrite;
            } else {
                goto HandleErrors;
            }
        }
    }

}

bool Aio::GetNextChar(unsigned char *nc)
{
  BOOL r;
  DWORD err;
  unsigned char c;
#define RETURN(x) {c=x;goto Out;}

  if (BytesRead < BytesAvailable)
    RETURN( InputBuffer[BytesRead++]);

  if (OpsPosted & opRead) {
    if (GetOverlappedResult(h,
			    &ReadOver,
			    &BytesRead,
			    TRUE)) {        /* wait */
      OpsPosted &= ~opRead;
      goto GotMilk;
    }
  }

  //  DPRINTF(L_GDB,0,("r(%x,%x) '%s'\n",ReadOver.Internal,ReadOver.InternalHigh,InputBuffer));
  BytesAvailable = 0;
  r = ReadFile(h,
	       InputBuffer,
	       BUFMAX,
	       &BytesRead,
	       &ReadOver);
  if (r) {
  GotMilk:
    ReadOver.Offset += BytesRead;
    BytesAvailable = BytesRead;
    BytesRead = 0;
    //DPRINTF(L_GDB,0,("%s +has %d (%x,%x) '%s'\n",MyName,BytesAvailable,ReadOver.Internal,ReadOver.InternalHigh,InputBuffer));
    RETURN( InputBuffer[BytesRead++]);
  }    

  OpsPosted |= opRead;
 WaitAgain:
  //  DPRINTF(L_GDB,0,("wfm(%x,%x)\n",ReadOver.Internal,ReadOver.InternalHigh));
  err = WaitForMultipleObjects(2,Events,FALSE,INFINITE);
  //  DPRINTF(L_GDB,0,("wfm->%x\n",err));
  if (err == WAIT_FAILED) goto Done;

  if (err == evRead) {
    if (GetOverlappedResult(h,
			    &ReadOver,
			    &BytesRead,
			    TRUE)) {        /* wait */
      OpsPosted &= ~opRead;
      goto GotMilk;

    } else {
      goto HandleErrors;
    }
  } else if (err == evWrite) {
    if (GetOverlappedResult(h,
			    &WriteOver,
			    &BytesWritten,
			    TRUE)) {        /* wait */
      WriteOver.Offset += BytesWritten;
      OpsPosted &= ~opWrite;
      goto WaitAgain;
    } else {
      goto HandleErrors;
    }
  }
 Done:
  *nc = 0;
  return false;
 HandleErrors:
  *nc = 0xff;
  return false;
 Out:
  //  DPRINTF(L_GDB,0,("->%c %x '%s'\n",c,c,InputBuffer));
  *nc = c;
  return true;
}

bool Aio::PutNextChar(unsigned char c)
{
    //    DPRINTF(L_GDB,0,("%c",c));
    BOOL r = WriteFile(h,&c,1,&BytesWritten,&WriteOver);
    if (r) {
        WriteOver.Offset += BytesWritten;
    }
    return true;
}


Aio::Aio( HANDLE Line)
{
    Events[evRead] = Events[evWrite] = h = INVALID_HANDLE_VALUE;
    BytesAvailable = BytesRead = BytesWritten = OpsPosted = 0;
    memset(&WriteOver,0,sizeof WriteOver);
    memset(&ReadOver,0,sizeof ReadOver);
    memset(InputBuffer,0,BUFMAX);

    if (Line == INVALID_HANDLE_VALUE)
        return; // and good luck to you

    h = Line;
    ReadOver.hEvent = 
        Events[evRead] = 
            CreateEvent(NULL,FALSE,FALSE,NULL);
    WriteOver.hEvent = 
        Events[evWrite] =
            CreateEvent(NULL,FALSE,FALSE,NULL);
}

Aio::~Aio(void)
{
    if (h != INVALID_HANDLE_VALUE) {
        CloseHandle(h);
        CloseHandle(Events[evRead]);
        CloseHandle(Events[evWrite]);
    }
    h = INVALID_HANDLE_VALUE;
}

/* Communication interface (to eMIPS/ModelSim/Giano/...)
 * Assume synchronous for now.
 */
enum IoResult {
    Success = 0,
    Timeout,
    NotConnected,
    Failed
};

class ToEmips {
 public:
    /*
    ToEmips(ConfigInfo *Info);
    virtual ~ToEmips(void);
    */

    virtual IoResult Read (BYTE *Buffer, DWORD nBytes, DWORD *nRead) = 0;
    virtual IoResult Write(BYTE *Buffer, DWORD nBytes, DWORD *nWritten) = 0;
    virtual IoResult CurrentStatus(void) = 0;
};

/* implementations of it.. */
class ToEmipsNull : public ToEmips {
 public:
    ToEmipsNull(ConfigInfo *Info);
    virtual ~ToEmipsNull(void);

    virtual IoResult Read (BYTE *Buffer, DWORD nBytes, DWORD *nRead);
    virtual IoResult Write(BYTE *Buffer, DWORD nBytes, DWORD *nWritten);
    virtual IoResult CurrentStatus(void);
};

class ToEmipsSerial : public ToEmips {
 public:
    ToEmipsSerial(ConfigInfo *Info);
    virtual ~ToEmipsSerial(void);

    virtual IoResult Read (BYTE *Buffer, DWORD nBytes, DWORD *nRead);
    virtual IoResult Write(BYTE *Buffer, DWORD nBytes, DWORD *nWritten);
    virtual IoResult CurrentStatus(void);
 private:
    HANDLE Line;
};

/* Interface to the eMIPS Extension (through serial line ToEmips above)
 */
class EmipsCpu {
 public:
    EmipsCpu(ConfigInfo *Info, ToEmips *Line);
    virtual ~EmipsCpu(void);

    bool ExternalInterrupt(bool Suspend);
    bool GetRegister(int GdbReg, UINT32 *Value);
    bool SetRegister(int GdbReg, UINT32 Value);
    bool Fetch(BYTE *MemBuffer, ADDRESS *Address, DWORD nBytes);
    bool Store(BYTE *MemBuffer, ADDRESS *Address, DWORD nBytes);
    bool InsertWBP (char code, ADDRESS *Address);
	bool DeleteWBP (char code, ADDRESS *Address);
	bool InitWBP(void);
	bool Wait(void);

 private:
    IoResult Rpc(BYTE CommandCode,
                 BYTE *InArguments, DWORD InBytes,
                 BYTE *InArguments2, DWORD InBytes2,
                 BYTE *OutArguments, DWORD *OutBytes);
    bool RegsMap (unsigned int GdbReg, EMIPSREG *EmipsReg);
    
    ToEmips *MyBoard;
	
	// Data Structure to manage hardware watchpoints and breakpoints
	struct {
		ADDRESS Address;
		char wp_type;
		bool is_wp;
		bool enabled;
	} wbp_table [WBP_MAX];

};

/* Our main object: Interface to GDB
 */
class Emips2Gdb {
 public:
    Emips2Gdb(ConfigInfo *Info, EmipsCpu *Emips);
    virtual ~Emips2Gdb(void);

    bool StartInterface(void);

    unsigned char *GetPacket (unsigned char *buffer);
    bool           PutPacket(const unsigned char *buffer);
    
    unsigned char* Mem2Hex(const unsigned char* mem,
                           unsigned char* buf,
                           int count);
    unsigned char* Hex2Mem(unsigned char* buf,
                           unsigned char* mem,
                           int count);
    unsigned char* Reg2Hex(const unsigned char* mem,
                           unsigned char* buf,
                           int count);
    unsigned char* Hex2Reg(unsigned char* buf,
                           unsigned char* mem,
                           int count);
    void           FakeReg(unsigned char* buf);
    bool           GdbServer(int GdbTrap);
    void           DebuggerNotify(const char *ModuleName, ADDRESS LoadAddress, char Event);
    
#define DebuggerNotifyLoad(m,a,e,x) DebuggerNotify(m,a,'L')
#define DebuggerNotifyUnload(m,a,t) DebuggerNotify(m,a,'U')

 private:

    /* Our state */

    BYTE PacketBuffer[BUFMAX];
#define BUFMEMMAX   (BUFMAX/2)
    BYTE MemBuffer [BUFMEMMAX];
    BYTE ErMsg[4];

    char PipeName[256];

    BOOL DebuggerAttached;
    ADDRESS StartOfIoSpace;
    ADDRESS EndOfIoSpace;

    Aio *ToDebugger;
    EmipsCpu * MyCpu;

    ConfigInfo *MyInfo;
};

/* Methods of the EmipsCpu class
 */

IoResult EmipsCpu::Rpc(BYTE CommandCode,
                       BYTE *InArguments, DWORD InBytes,
                       BYTE *InArguments2, DWORD InBytes2,
                       BYTE *OutArguments, DWORD *OutBytes)
{
    IoResult r;
    DWORD nb = 0;

    DPRINTF(L_CPU,0,("EmipsCpu::Rpc(BYTE CommandCode=%d, UINT32 *InArguments=%x, DWORD InBytes=%d, UINT32 *OutArguments=%x, DWORD *OutBytes=%x *(%d))\n",
            CommandCode, InArguments, InBytes, OutArguments, OutBytes,
            (OutBytes) ? *OutBytes : 0));

#define retfail(_r_) (((_r_) == Success) ? Failed : (_r_))

    r = MyBoard->Write(&CommandCode,1,&nb);

	//
	//Sleep(1000);
	//

    if ((r != Success) || (nb != 1)) {
        EPRINTF(("EmipsCpu::Rpc::Write failed r=%d nb=%d\n", r, nb));
        return retfail(r);
    }
    if (InArguments && InBytes) {
        r = MyBoard->Write(InArguments,InBytes,&nb);
        if ((r != Success) || (nb != InBytes)) {
            EPRINTF(("EmipsCpu::Rpc::Write2 failed r=%d nb=%d (?%d)\n", r, nb, InBytes));
            return retfail(r);
        }

        if (InArguments2 && InBytes2) {
            r = MyBoard->Write(InArguments2,InBytes2,&nb);
            if ((r != Success) || (nb != InBytes2)) {
                EPRINTF(("EmipsCpu::Rpc::Write3 failed r=%d nb=%d (?%d)\n", r, nb, InBytes2));
                return retfail(r);
            }
        }

    }

    if (OutArguments && OutBytes) {
        r = MyBoard->Read(OutArguments,*OutBytes,&nb);
        if ((r != Success) || (nb != *OutBytes)) {
            EPRINTF(("EmipsCpu::Rpc::Read failed r=%d nb=%d (?%d)\n", r, nb,*OutBytes));
            return retfail(r);
        }
    } else {
        /* Just an Ack */
        BYTE Ack;
        r = MyBoard->Read(&Ack,1,&nb);
        if ((r != Success) || (nb != 1)) {
            EPRINTF(("EmipsCpu::Rpc::ReadAck failed r=%d nb=%d\n", r, nb));
            return retfail(r);
        }
    }
    return r;
}


bool EmipsCpu::RegsMap (unsigned int GdbReg, EMIPSREG *EmipsReg)
{
	if ((GdbReg >=GP0_REGNUM) && (GdbReg <= GPLAST_REGNUM)){
	       	DPRINTF(L_CPU,0,("EmipsCpu::RegsMap... Accessing to a GP register!)\n"));
		EmipsReg->fSpecial = 0;	
	       	EmipsReg->nReg	  = GdbReg;
	       	return true;
	}
	else {  
		EmipsReg->fSpecial = 1;
		switch (GdbReg) {
			case PC_REGNUM 	: EmipsReg->nReg = PC_FSPECIAL_REGNUM;
					  return true;
			case LO_REGNUM 	: EmipsReg->nReg = LO_FSPECIAL_REGNUM;
					  return true;
			case HI_REGNUM 	: EmipsReg->nReg = HI_FSPECIAL_REGNUM;
					  return true;
			case PS_REGNUM  : EmipsReg->nReg = PS_FSPECIAL_REGNUM;
					  return true;
			case BADVADDR_REGNUM  : EmipsReg->nReg = BADVADDR_FSPECIAL_REGNUM;
					  return true;
			case CAUSE_REGNUM  : EmipsReg->nReg = CAUSE_FSPECIAL_REGNUM;
					  return true;
			case FPLAST_REGNUM  : EmipsReg->nReg = FPLAST_FSPECIAL_REGNUM;
					  return true;
			case FPS_REGNUM  : EmipsReg->nReg = FPS_FSPECIAL_REGNUM;
					  return true;

			default		: DPRINTF(L_CPU,0,("EmipsCpu::RegsMap... The Gdb %d register does not exist in hardware!)\n",GdbReg));
					  return false; 			// Gdb Register is not accessible in eMIPS 

		}
	}
}

bool EmipsCpu::GetRegister(int GdbReg, UINT32 *Value)
{
    IoResult r;
    DWORD nb = 4;
    BYTE Command; 
    bool fSpecial;
    unsigned char nReg;

    EMIPSREG EmipsReg, *pEmipsReg = &EmipsReg;

    if (RegsMap(GdbReg, pEmipsReg)){	// Verify if Gdb Register is an eMIPS accessible one
    	fSpecial = pEmipsReg->fSpecial;
   		nReg     = pEmipsReg->nReg;

    	Command = (fSpecial) ? GETREG | (nReg << 2) | FSPECIAL : GETREG | (nReg << 2);
    
   	 DPRINTF(L_CPU,0,("EmipsCpu::GetRegister(int nReg=%d, UINT32 *Value=%x, bool fSpecial=%d, command=%d)\n",
            	nReg, Value, fSpecial,Command));

   	 r = Rpc(Command,
            	NULL, 0,
            	NULL, 0,
            	(BYTE*)Value, &nb);
   	 return (r == Success);
    }
    else
	 return false;

} 


bool EmipsCpu::SetRegister(int GdbReg, UINT32 Value)
{
    IoResult r;
    BYTE Command;
    bool fSpecial;
    unsigned char nReg;

    EMIPSREG EmipsReg, *pEmipsReg = &EmipsReg;;

    
    if (RegsMap(GdbReg, pEmipsReg)){	// Verify if Gdb Register is an eMIPS accessible one
    	fSpecial = pEmipsReg->fSpecial;
   	nReg     = pEmipsReg->nReg;

    	Command = (fSpecial) ? SETREG | (nReg << 2) | FSPECIAL : SETREG | (nReg << 2);

   	DPRINTF(L_CPU,0,("EmipsCpu::SetRegister(int nReg=%d, UINT32 *Value=%x, bool fSpecial=%d, command=%d)\n",
        	    nReg, Value, fSpecial,Command));

    	r = Rpc(Command,
           	(BYTE*)&Value, 4,
           	NULL, 0,
           	NULL, 0);
   	return (r == Success);
	}
    else
	return false;

}
 

bool EmipsCpu::Fetch(BYTE *MemBuffer, ADDRESS *Address, DWORD nBytes)
{
    IoResult r;
    BYTE Command;
    BYTE * pAddr = (BYTE *) Address;
	BYTE *p_nBytes = (BYTE *) &nBytes;
	BYTE nBytes_Addr [6];
    
    Command = FETCH;

    DPRINTF(L_CPU,0,("Command = %d, nBytes = %d, Address=%x \n", 
			Command, nBytes, *Address));
    
	if (nBytes < 32) {
		Command |= (BYTE)nBytes << 3;	// Embedds nBytes in the command byte	
	
		nBytes_Addr [0] = *(pAddr+3);
		nBytes_Addr [1] = *(pAddr+2);
		nBytes_Addr [2] = *(pAddr+1);
		nBytes_Addr [3] = *(pAddr+0);

		r = Rpc(Command,
            nBytes_Addr, 4,
            NULL, 0,
            (BYTE*)MemBuffer, &nBytes);
    }
    else {
		nBytes_Addr [0] = *(p_nBytes+1);
		nBytes_Addr [1] = *p_nBytes;
		nBytes_Addr [2] = *(pAddr+3);
		nBytes_Addr [3] = *(pAddr+2);
		nBytes_Addr [4] = *(pAddr+1);
		nBytes_Addr [5] = *(pAddr+0);	
		
		r = Rpc(Command,
            nBytes_Addr, 6,
            NULL, 0,
            (BYTE*)MemBuffer, &nBytes);
    
    }


    DPRINTF(L_CPU,0,("EmipsCpu::Fetch(ADDRESS Address=%x, void *Destination=%x, DWORD nBytes=%u)\n",
            Address, MemBuffer, nBytes));
   
	DPRINTF(L_CPU,0,("*p_nBytes, Sizeof(nBytes) = %x, %d\n", *p_nBytes, sizeof(nBytes)));
   
	DPRINTF(L_CPU,0,("nBytes_Addr [0:5] = %x, %x, %x, %x %x %x\n",
			nBytes_Addr [0], nBytes_Addr [1], nBytes_Addr [2],
			nBytes_Addr [3], nBytes_Addr [4], nBytes_Addr [5]));
   
    return (r == Success);
}

bool EmipsCpu::Store(BYTE *MemBuffer, ADDRESS *Address, DWORD nBytes)
{
    IoResult r;
    BYTE Command;
	BYTE *pAddr = (BYTE*) Address;
	BYTE *p_nBytes = (BYTE *) &nBytes;
	BYTE nBytes_Addr [6];

	Command = STORE;

	DPRINTF(L_CPU,0,("Command = %d, nBytes = %d, Address=%x \n", 
			Command, nBytes, *Address));

	if (nBytes < 32) {
		Command |= (BYTE)nBytes << 3;	// Embedds nBytes in the command byte	

		nBytes_Addr [0] = *(pAddr+3);
		nBytes_Addr [1] = *(pAddr+2);
		nBytes_Addr [2] = *(pAddr+1);
		nBytes_Addr [3] = *(pAddr+0);

		r = Rpc(Command,
            nBytes_Addr, 4,
            (BYTE*)MemBuffer, nBytes,
            NULL, 0);
	}
	else {
		nBytes_Addr [0] = *(p_nBytes+1);
		nBytes_Addr [1] = *p_nBytes;
		nBytes_Addr [2] = *(pAddr+3);
		nBytes_Addr [3] = *(pAddr+2);
		nBytes_Addr [4] = *(pAddr+1);
		nBytes_Addr [5] = *(pAddr+0);	

		r = Rpc(Command,
             nBytes_Addr, 6,
            (BYTE*)MemBuffer, nBytes,
            NULL, 0);
	}

	DPRINTF(L_CPU,0,("EmipsCpu::Store(ADDRESS Address=%x, void *MemBuffer=%x, DWORD nBytes=%u)\n",
            Address, MemBuffer, nBytes));

	DPRINTF(L_CPU,0,("nBytes_Addr [0:5] = %x, %x, %x, %x %x %x\n",
			nBytes_Addr [0], nBytes_Addr [1], nBytes_Addr [2],
			nBytes_Addr [3], nBytes_Addr [4], nBytes_Addr [5]));
   
   
    return (r == Success);
}

bool EmipsCpu::ExternalInterrupt(bool Suspend)
{
    IoResult r;
    BYTE Ack;
    DWORD nb = 1;

    DPRINTF(L_CPU,0,("EmipsCpu::ExternalInterrupt(bool Suspend=%d)\n",
            Suspend));

    r = Rpc((Suspend) ? (SUSPEND|INTERRUPT) : (CONTINUE|INTERRUPT),
            NULL, 0,
            NULL, 0,
            &Ack, &nb);

	
 /*   while (Ack != 0xff) {
        r = MyBoard->Read(&Ack,1,&nb);*/
	   while ((nb == 0) || (Ack != 0xff)) {
			Rpc((Suspend) ? (SUSPEND|INTERRUPT) : (CONTINUE|INTERRUPT),
            NULL, 0,
            NULL, 0,
            &Ack, &nb);
    }

    return (r == Success);
}


bool EmipsCpu::InsertWBP (char code, ADDRESS *Address)
{
	IoResult r;
    BYTE Command;
	BYTE ControlByte;
	char entry;
	bool found = false;
	char IsWpMask;
	char WatchpointTypeMask;
    char InsertMask = 0x20;  // Enable Mask
	char wp_type;
	BYTE *pAddr = (BYTE*) Address;
	BYTE AddrBuff [4]; 

   	Command = EXTENDED | WBPOINTS;

   		// Build ControlByte 
		// Defines WatchpointTypeMask and IsWpMask
	switch (code) {

					// Case 0 is for software breakpoint. In that case this function is not called
		case 1	:	// hardware breakpoint
					IsWpMask = 0x00;			
					WatchpointTypeMask = 0xc0;		// Dummy value
					wp_type = 3;					// Break point wp_type value
					break;
		case 2	:	// Write watchpoint
					IsWpMask = 0x10;			
					WatchpointTypeMask = 0x00;
					wp_type = 0;
					break;
		case 3	:	// Read watchpoint
					IsWpMask = 0x10;			
					WatchpointTypeMask = 0x40;
					wp_type = 1;
					break;
		case 4	:	// Read/Write watchpoint
					IsWpMask = 0x10;			
					WatchpointTypeMask = 0x80;
					wp_type = 2;
	}
	
	// Search for availability of Watchpoint-Breapoint slots and update table if there is one 
	for (entry=0; entry < WBP_MAX ; entry++)
		if (!(wbp_table[entry].enabled)){
			found = true;
			wbp_table[entry].enabled = true;
			wbp_table[entry].Address = *Address;
			if (code>1){							// Watchpoint
				wbp_table[entry].wp_type = wp_type;	
				wbp_table[entry].is_wp = true;
			}
			else {									// Breakpoint
				wbp_table[entry].wp_type = 3;		// Dummy Value
				wbp_table[entry].is_wp = false;
			}

			break;
		}

		if (!found) {
			DPRINTF(L_CPU,0,("EmipsCpu::InsertWBP : No HW Watchpoints Break Points slot available!"));
			return (false);		// No HW watchpoints/breakpoints available
		}


	ControlByte = WatchpointTypeMask | InsertMask | IsWpMask | entry;

	DPRINTF(L_CPU,0,("EmipsCpu::InsertWBP(char code=%d, ADDRESS *Address=%x, Command=%x, ControlByte=%x)\n",
        	    code, *Address, Command, ControlByte));

	// Put Address byte in the correct order
	AddrBuff[0] = *(pAddr + 3);
	AddrBuff[1] = *(pAddr + 2);
	AddrBuff[2] = *(pAddr + 1);
	AddrBuff[3] = *(pAddr + 0);

	r = Rpc(Command,
           	&ControlByte, 1,
           	AddrBuff, 4,
           	NULL, 0);
   	
	return (true);
}


bool EmipsCpu::DeleteWBP (char code, ADDRESS *Address)
{
	IoResult r;
    BYTE Command;
	BYTE ControlByte;
	char entry;
	bool found = false;
	char DeleteMask = 0x00;  // Delete Mask
	char wp_type;
	bool looking_for_wp;

   	Command = EXTENDED | WBPOINTS;

	switch (code) {

					// Case 0 is for software breakpoint. In that case this function is not called
		case 1	:	// hardware breakpoint
					looking_for_wp = false;
					wp_type = 3;				
					break;
		case 2	:	// Write watchpoint
					looking_for_wp = true;
					wp_type = 0;
					break;
		case 3	:	// Read watchpoint
					looking_for_wp = true;
					wp_type = 1;
					break;
		case 4	:	// Read/Write watchpoint
					looking_for_wp = true;
					wp_type = 2;

	}

   	// Search for the entry to delete 
	for (entry=0 ; entry < WBP_MAX ; entry++)
		if ((*Address == wbp_table[entry].Address) && (wbp_table[entry].wp_type == wp_type) 
			&& (looking_for_wp == wbp_table[entry].is_wp) && (wbp_table[entry].enabled)) {
					found = true;
					wbp_table[entry].enabled = false; // Disable 
					break;
		}

	if (!found) {
			DPRINTF(L_CPU,0,("EmipsCpu::DeleteWBP : No HW Watchpoints Break Points to delete"));
			return (false);		// No HW watchpoints/breakpoints available
		}

	// Build ControlByte 
		// Defines WatchpointTypeMask and IsWpMask

	ControlByte = DeleteMask | entry;

	DPRINTF(L_CPU,0,("EmipsCpu::DeleteWBP(char code=%d, ADDRESS *Address=%x, Command=%x, ControlByte=%x)\n",
        	    code, Address, Command, ControlByte));

	r = Rpc(Command,
           	&ControlByte, 1,
           	NULL, 0,
           	NULL, 0);
   	
	return (true);

}


bool EmipsCpu::InitWBP(void)	
{
	IoResult r;
	int entry;
	BYTE Command = EXTENDED | WBPOINTS;
	BYTE ControlByte;
	char DeleteMask = 0x00;  // Delete Mask

	// Init Data Structure table_wbp
	for (entry=0 ; entry<WBP_MAX ; entry++) {
		wbp_table[entry].Address	= 0;
		wbp_table[entry].is_wp		= false;
		wbp_table[entry].wp_type	= -1;
		wbp_table[entry].enabled	= false;
	}

	// Init Watchpoint Breakpoint hardware slots
	for (entry=0 ; entry<WBP_MAX ; entry++) {
		ControlByte = DeleteMask | entry;
		r = Rpc(Command,
           	&ControlByte, 1,
           	NULL, 0,
           	NULL, 0);
		if (r != Success) return false;

	}

	return true;
}


bool EmipsCpu::Wait(void)
{
    IoResult r;
    DWORD nb = 0;
    BYTE Ack;

    DPRINTF(L_CPU,0,("EmipsCpu::Wait()\n"));

    /* Wait for a trap indication */
   /* r = MyBoard->Read(&Ack,1,&nb);
    if ((r != Success) || (nb != 1)) {
        EPRINTF(("EmipsCpu::Wait::Read failed r=%d nb=%d\n", r, nb));
        return false;
		}
	*/
    r = MyBoard->Read(&Ack,1,&nb);
    //printf("\nACK = %d\n", Ack);

	while ((r != Success) || (nb != 1)) {
        EPRINTF(("EmipsCpu::Wait::Read failed r=%d nb=%d\n", r, nb));
        r = MyBoard->Read(&Ack,1,&nb);
	}
    return true;
}

EmipsCpu::EmipsCpu(ConfigInfo *Info, ToEmips *Line)
{
 
	/*for (int entry=0 ; entry<WBP_MAX ; entry++) {
		wbp_table[entry].Address	= 0;
		wbp_table[entry].is_wp		= false;
		wbp_table[entry].wp_type	= -1;
		wbp_table[entry].enabled	= false;
	}*/

    DPRINTF(L_CPU,0,(":EmipsCpu(ConfigInfo *Info=%x, ToEmips *Line=%x)\n",
            Info, Line));
    MyBoard = Line;
}

EmipsCpu::~EmipsCpu(void)
{
    DPRINTF(L_CPU,0,(":~EmipsCpu(void)\n"));
}

/* Methods of the Emips2Gdb class
 */
/* Constructors/destructors
 */
Emips2Gdb::Emips2Gdb(ConfigInfo *Info, EmipsCpu *Emips)
{
    DPRINTF(L_GDB,0,(":Emips2Gdb(ConfigInfo *Info=%x, EmipsCpu *Emips=%x)\n",
            Info, Emips));

    ErMsg[0] = 'E';
    ErMsg[1] = '0';
    ErMsg[2] = '0';
    ErMsg[3] = 0;

    this->MyCpu = NULL;
    ToDebugger = NULL;

    this->DebuggerAttached = FALSE;

    /* BUGBUG fixme */
    StartOfIoSpace = 0;
    EndOfIoSpace = 0;

    this->MyInfo = Info;
    this->MyCpu = Emips;

    DPRINTF(L_GDB,0,("\t:Emips2Gdb(%s %x) ->%x\n", MyName, Info, this));
}

Emips2Gdb::~Emips2Gdb(void)
{
    DPRINTF(L_GDB,0,(":~Emips2Gdb(void)\n"));
}

/*
 * CPU interface
 *
 */
	

/* Low-level interaction with client
 */
bool Emips2Gdb::StartInterface(void)
{
    char *op;
    BOOL r, fflop = 0;
    DWORD err;
    HANDLE h;
    OVERLAPPED Over;

    if (MyCpu == NULL) {
        printf("%s: Missing CPU interface", MyName);
        return false;
    }

    // Namepipe init

    sprintf(PipeName,"\\\\.\\pipe\\%s", MyName);
    printf("Debugger pipe is %s\n", PipeName);

    h = CreateNamedPipe(PipeName,
                           PIPE_ACCESS_DUPLEX |
                           FILE_FLAG_OVERLAPPED,
                           PIPE_TYPE_BYTE |
                           PIPE_READMODE_BYTE |
                           PIPE_WAIT,
                           PIPE_UNLIMITED_INSTANCES,
                           4096,     // output buffer size
                           4096,     // input buffer size
                           INFINITE,   // default timeout
                           NULL       // security attrib
                           );
    if (h == INVALID_HANDLE_VALUE) {
        EPRINTF(("CreateNamedPipe(%s) failed with err = x%x\n", 
               PipeName, GetLastError()));
        return false;
    }

    ToDebugger = new Aio(h);

    memset(&Over, 0, sizeof Over);
    Over.hEvent = CreateEvent(NULL,FALSE,FALSE,NULL);

 AllOverAgain:

    op = "ConnectNamedPipe";
    r = ConnectNamedPipe(h,
                         &Over);
    if (r) goto Done; // should fail!

    printf("Debugger listener %s waiting ...\n", MyName);

    op = "GetOverlappedResult";
    r = GetOverlappedResult(h,
                            &Over,
                            &err, /* useless */
                            TRUE);
    if (!r) goto Done;

#if 1
    /* cgwin is buggy and closes/reopens the pipe immediately.
     */
    if (!fflop) {
        fflop ^= 1;
        r = DisconnectNamedPipe(h);
        goto AllOverAgain;
    }
    fflop = 0;

#endif

   
    printf("Debugger listener %s connected ...\n", MyName);
    DebuggerAttached = TRUE;

    /* wait until we get a command
     */
    r = ToDebugger->PostAndWait(&err);
    if (r) {
        /* Switch between servicing GDB and waiting for the processor to stop
         * BUGBUG: Cannot handle ^C from the user this way.
         */
        int GdbTrap = 0;
        for (;;) {
            op = "ServerLoop()";
            r = GdbServer(GdbTrap);
            if (!r) goto Disconnect;

            op = "CpuWait()";
            r = MyCpu->Wait();
            if (!r) goto Done;

            /* Inform the remote debugger of what happened */
            {
                unsigned char Buffer[4];

                GdbTrap = 5; /* uhu? */
                Buffer[0] = 'T'; 
                Buffer[1] = '0';
                Buffer[2] = '5';
                Buffer[3] = 0;
      
                PutPacket(Buffer);
            }
        }
    } else {
        switch (err) {
        case ERROR_HANDLE_EOF:
        case ERROR_BROKEN_PIPE:
        Disconnect:
            printf("%s: Debugger client is gone. ", MyName);

            op = "DisconnectNamedPipe";
            r = DisconnectNamedPipe(h);
            if (!r) goto Done;

        case ERROR_PIPE_LISTENING:
            // get back to the beginning..
            printf("%s: Waiting for new connection...\n",MyName);
            goto AllOverAgain;

        default: 
            break;
        }
    }

 Done:
    EPRINTF(("Last %s failed with err = x%x\n", op, GetLastError()));
    delete ToDebugger;
    return true;
}

/*
 * GDB protocol proper
 */
#define PUBLIC
#define PRIVATE static

/* packets encode bytes as hexnum pairs
 */
PRIVATE const unsigned char hexchars[]="0123456789abcdef";


/* All regs are one word
 */
#define MIPS_REGISTER_SIZE 4

/* atoh only smaller ;-)
 */
PRIVATE int 
hex(unsigned char ch)
{
    if ((ch >= 'a') && (ch <= 'f')) return (ch-'a'+10);
    if ((ch >= '0') && (ch <= '9')) return (ch-'0');
    if ((ch >= 'A') && (ch <= 'F')) return (ch-'A'+10);
    return (-1);
}

/* scan for the sequence $<data>#<checksum>    
 */
unsigned char *
Emips2Gdb::GetPacket (unsigned char *buffer)
{
    unsigned char checksum;
    unsigned char xmitcsum;
    int count;
    unsigned char ch;

    for (;;) {
        /* wait around for the start character, ignore all other characters
         */
        for (;;) {
            if (!ToDebugger->GetNextChar(&ch))
                return NULL;
            if (ch == '$')
                break;
        }

    retry:
        checksum = 0;
        xmitcsum = (unsigned char)-1;
        count = 0;

        /* now, read until a # or end of buffer is found */
        while (count < BUFMAX) {
            if (!ToDebugger->GetNextChar(&ch))
                return NULL;
            if (ch == '$')
                goto retry;
            if (ch == '#')
                break;
            checksum = (unsigned char)(checksum + ch);
            buffer[count] = ch;
            count = count + 1;
        }
        buffer[count] = 0;

        if (ch == '#') {
            if (!ToDebugger->GetNextChar(&ch))
                return NULL;
            xmitcsum = (unsigned char)(hex (ch) << 4);
            if (!ToDebugger->GetNextChar(&ch))
                return NULL;
            xmitcsum = (unsigned char)(xmitcsum + hex (ch));

            if (checksum != xmitcsum) {
                /* Negative acknowledge
                 */
                ToDebugger->PutNextChar ('-');
            } else {
                ToDebugger->PutNextChar ('+'); /* successful transfer */

                /* if a sequence char is present, reply the sequence ID */
                if (buffer[2] == ':')
                {
                    ToDebugger->PutNextChar (buffer[0]);
                    ToDebugger->PutNextChar (buffer[1]);

                    return &buffer[3];
                }

                return &buffer[0];
            }
        }
    }
}

/* send the packet in the comm buffer.  
 */
bool 
Emips2Gdb::PutPacket(const unsigned char *buffer)
{
    unsigned char checksum;
    int  count;
    unsigned char ch;

    DPRINTF(L_GDB,0,("PutPacket(%s)\n", buffer));
    /*  $<packet info>#<checksum>. */
    do {
        ToDebugger->PutNextChar('$');
        checksum = 0;
        count    = 0;

        while ((ch=buffer[count]) != 0) {
            ToDebugger->PutNextChar(ch);
            checksum = (unsigned char)(checksum + ch);
            count += 1;
        }

        ToDebugger->PutNextChar('#');
        ToDebugger->PutNextChar(hexchars[checksum >> 4]);
        ToDebugger->PutNextChar(hexchars[checksum % 16]);

        if (!ToDebugger->GetNextChar(&ch))
            return false;

    } while (ch != '+');

    return true;
}

/* convert the memory pointed to by mem into hex, placing result in buf
 * return a pointer to the last char put in buf (zero, unless faulted).
 */
unsigned char* 
Emips2Gdb::Mem2Hex(const unsigned char* mem,
                      unsigned char* buf,
                      int count)
{
    int i;
    unsigned char ch;

    for (i=0;i<count;i++) {
        ch = *(mem + i);
        *buf++ = hexchars[ch >> 4];
        *buf++ = hexchars[ch % 16];
    }
    *buf = 0;
    return(buf);
}

/* convert the hex array pointed to by buf into binary to be placed in mem
 * return a pointer to the character AFTER the last byte written
 */
unsigned char*
Emips2Gdb::Hex2Mem(unsigned char* buf,
                      unsigned char* mem,
                      int count)
{
    int i;
    unsigned char ch;

    for (i=0;i<count;i++) {
        ch = (unsigned char)(hex(*buf++) << 4);
        ch = (unsigned char)(ch + hex(*buf++));
        *mem++ = ch;
    }
    return(mem);
}

/* while we find nice hex chars, build an int
 * return number of chars processed          
 */
PRIVATE int 
Hex2Int(unsigned char **ptr, int *intValue)
{
    int numChars = 0;
    int hexValue;

    *intValue = 0;

    while (**ptr) {
        hexValue = hex(**ptr);
        if (hexValue >=0) {
            *intValue = (*intValue <<4) | hexValue;
            numChars ++;
        }
        else
            break;

        (*ptr)++;
    }

    return (numChars);
}

/* convert the LOCAL memory pointed to by mem into hex, placing result in buf
 * return a pointer to the last char put in buf (zero, unless faulted).
 */
unsigned char* 
Emips2Gdb::Reg2Hex(const unsigned char* mem,
		   unsigned char* buf,
		   int count)
{ 
    int i;
    unsigned char ch;Emips2Gdb:

    for (i=0;i<count;i++) {
        ch = *mem++;
        *buf++ = hexchars[ch >> 4]; 
        *buf++ = hexchars[ch % 16];
    }
    *buf = 0;
    return(buf);
}

/* convert the hex array pointed to by buf into binary to be placed in LOCAL mem
 * return a pointer to the character AFTER the last byte written
 */
unsigned char*
Emips2Gdb::Hex2Reg(unsigned char* buf,
		   unsigned char* mem,
		   int count) 
{
    int i;
    unsigned char ch;

    for (i=0;i<count;i++) {
        ch = (unsigned char)(hex(*buf++) << 4);
        ch = (unsigned char)(ch + hex(*buf++));
        *mem++ = ch;
    }
    return(mem);
}


/* 
 * This function write a constant string of 8 byte on the buffer pointed by buf
 * It's called if the GdbReg is not an eMIPS accessible register
 */
void	   
Emips2Gdb::FakeReg(unsigned char* buf) {
        const char* fakereg= {FAKE_REG_HEX_VALUE};
	int i;
	
	DPRINTF(L_GDB,0,("\nPutting fake value on GdbReg \n"));
	for (i=0 ; i<8 ; i++)
		*buf++ = fakereg[i];
}

	
/* Protocol handler
 * This function does all command processing for interfacing to gdb.
 */
PRIVATE const unsigned char OkMsg[3] = {'O', 'K', 0};
PRIVATE const unsigned char NoHwWatchBreakMsg[3] = {'`','\'', 0};

bool
Emips2Gdb::GdbServer(int GdbTrap)
{
    int stepping, i;
    ADDRESS addr;
    int    length;
    unsigned int regno;
    unsigned char * ptr, *ptrSave;
    UINT32 Value;
    unsigned int GdbReg;
	char Ztype;


#define ErrorReply(_c_) {ErMsg[2] = _c_;goto Bummer;}


    /* If not from a trap must first stop the processor
     */
    if (GdbTrap == 0) {
	    MyCpu->ExternalInterrupt(true);
		
		// If Hardware Watchpoints and Breakpoints are present initialize them
		
	// Init Data Structure table_wbp
	
		if (HW_WBP)
			MyCpu->InitWBP();
    }


    /* We are not single-stepping, unless GDB says so.
     */
    stepping = 0;

    /* Enter command loop, accept packet from debugger and act on it.
     * Until instructed to continue execution.
     */

    for (;;) {
        /* Get next command
         */
        ptr = GetPacket(&PacketBuffer[0]);
        if (ptr == NULL) return false;
        i = *ptr++;
        DPRINTF(L_GDB,0,("gnc %c%s\n",i,ptr));

        /* Default reply -- none
         */
        PacketBuffer[0] = 0;

        /* See whatnot
         */
        switch (i) {

        case '?' :
            /* re-synchronize, or get the list of loaded modules.
             */
            if (*ptr == 'L') {
                DPRINTF(L_GDB,0,("DebuggerNotifyModuleList(); ???? \n"));
                /* Return OK, no matter what */
                PacketBuffer[0] = 0;
            } else {
                PacketBuffer[0] = 'S';
                PacketBuffer[1] =  hexchars[GdbTrap >> 4];
                PacketBuffer[2] =  hexchars[GdbTrap % 16];
                PacketBuffer[3] = 0;
            }
            break;

        case 'g' :
            /* return the value of all the CPU registers
             */
    
            ptr = PacketBuffer;
            for (i = 0; i <= LAST_REGNUM; i++) {
 		
                if (MyCpu->GetRegister(i, &Value)) {
                	Reg2Hex((const unsigned char*) &Value,
                        	ptr,
                        	MIPS_REGISTER_SIZE);
                }
                else
                    FakeReg (ptr); 
		
                ptr += MIPS_REGISTER_SIZE*2; /* 2 hex(es) per byte */
            }

            break;

        case 'G' :
            /* set the value of all the CPU registers - return OK 
             */

            for (i = 0; i <= LAST_REGNUM; i++) {
                Hex2Reg (ptr,
                         (unsigned char *)&Value,
                         MIPS_REGISTER_SIZE);
                MyCpu->SetRegister(i, Value);
                ptr += MIPS_REGISTER_SIZE*2;
            }

            goto Itsok;

#if 1 /* It works without this one too */
        case 'p' :
            /* get the value of a single CPU register - return OK
             */

            Hex2Int (&ptr, (int *)&GdbReg);
            ptr = PacketBuffer;
            
            UINT32 Value;               
		
            if (MyCpu->GetRegister(GdbReg, &Value)) {
                Reg2Hex((const unsigned char*) &Value,
                        ptr,
                        MIPS_REGISTER_SIZE);
            }
            else {
                FakeReg (ptr);
                //ErrorReply('1');
            }	
            break;
#endif

        case 'P' :
            /* set the value of a single CPU register - return OK
            */ 
	   
            if (Hex2Int (&ptr, (int *)&GdbReg) && *ptr++ == '=') {
                    Hex2Reg (ptr,
                             (unsigned char *)&Value, 
                             MIPS_REGISTER_SIZE);
		    
                    MyCpu->SetRegister(GdbReg, Value);
			   
                    goto Itsok;
            }
	    
            //ErrorReply('1');
	    
            break;
	

        case 'm' :
            /* mAA..AA,LLLL  Read LLLL bytes at address AA..AA
             */
            ptrSave = ptr;
            /* TRY TO READ %x,%x.  IF SUCCEED, SET PTR = 0
             * NB: eval is left to right in C, case you forgot.
             */
            if (Hex2Int(&ptr,(int*)&addr) &&
                (*(ptr++) == ',') &&
                Hex2Int(&ptr,&length) &&
                (length < BUFMAX/2)) {
                ptr = 0;
//#if 0
//                if (StartOfIoSpace &&
//                    (addr >= StartOfIoSpace) &&
//                    (addr < EndOfIoSpace)) {
//                    io2hex( addr, PacketBuffer, length);
//                } else
//#endif
                {
                    MyCpu->Fetch(MemBuffer, &addr, length);
                    Mem2Hex((unsigned char*) MemBuffer, PacketBuffer, length);
                }
//#if 0
//                if (GetSetChar_faulted) {
//                    ErrorReply('3');
//                }
//#endif
            }

            if (ptr) {
                ErrorReply('1');
            }

            break;

        case 'M' :
            /* MAA..AA,LLLL: Write LLLL bytes at address AA.AA return OK
             */
            if (Hex2Int(&ptr,(int*)&addr) &&
                (*(ptr++) == ',') &&
                Hex2Int(&ptr,&length) &&
                (*(ptr++) == ':')) {
#if 0
                if (StartOfIoSpace &&
                    (addr >= StartOfIoSpace) &&
                    (addr < EndOfIoSpace)) {
                    hex2io( ptr, addr, length);
                } else
#endif
				{
                    Hex2Mem(ptr, (unsigned char*) MemBuffer, length);
					MyCpu->Store(MemBuffer, &addr, length);
                }

#if 0
                if (GetSetChar_faulted) {
                    ErrorReply('3');
                }
#endif

                goto Itsok;

            }

            ErrorReply('2');
            break;

        case 's' :
            /* sAA..AA   Step one instruction from AA..AA(optional)
             */
            stepping = 1;
            /* fall through
             */

        case 'c' :
            /* cAA..AA    Continue at address AA..AA(optional)
             */

            /* try to read optional parameter, pc unchanged if no parm
            */
            if (Hex2Int(&ptr,(int*)&addr))
                MyCpu->SetRegister(PC_REGNUM, addr);

            /* Tell processor to continue, wait for a trap.
             */
            goto Out;

        case 'D' :
            /* Debugger is detaching, wont speak again
             */
            PutPacket(PacketBuffer);
            DebuggerAttached = FALSE;
            goto Out;

        case 'k' :
            /* Kill program
             */
            DebuggerAttached = FALSE;
            DPRINTF(L_GDB,0,("BaseDelete(); * wont come back * ??\n"));
            goto Out;

		case 'Z' :
            /* Insert WATCHPOINT AND BREAKPOINT
             */
			/* Zt,LLLL
             */
            
			Ztype = *(ptr++)-'0'; // Compute Z type
			
			if (*(ptr++) == ',') { 
			
				Hex2Int(&ptr,(int*)&addr); // Get the Address

				DPRINTF(L_GDB,0,("\nEmips2Gdb::GdbServer --> HW_WBP = %d,  BREAKPOINT OR WATCHPOINT FUNCTION: %d\n", HW_WBP, Ztype));
	            
				if (HW_WBP)
					// Hardware watchpoint: 
					if ((Ztype > 0) && (Ztype <= 4)) { // A hw breakpoint or watchpoint insertion is requested 
					
						MyCpu->InsertWBP(Ztype, &addr);		
						goto Itsok;
					}
					else
						// Software breakpoint: do nothing!
						break;
				else { // No Support to Hardware Breakpoints and Watchpoints
					break;
						}
			}
			
		
			ErrorReply('2');
			break;
			
		case 'z' :
            /* Remove WATCHPOINT AND BREAKPOINT
             */
            	/* zt,LLLL
             */
            
			Ztype = *(ptr++)-'0'; // Compute Z type
			
			if (*(ptr++) == ',') { 
			
				Hex2Int(&ptr,(int*)&addr); // Get the Address

				DPRINTF(L_GDB,0,("\nEmips2Gdb::GdbServer --> HW_WBP = %d,  BREAKPOINT OR WATCHPOINT FUNCTION: %d\n", HW_WBP, Ztype));
	            
				if (HW_WBP)
					// Hardware watchpoint: 
					if ((Ztype > 0) && (Ztype <= 4)) {// A hw breakpoint or watchpoint insertion is requested 
						MyCpu->DeleteWBP(Ztype, &addr);		
						goto Itsok;
					}
					else
						// Software breakpoint: do nothing!
						break;
				else { // No Support to Hardware Breakpoints and Watchpoints
					PutPacket(NoHwWatchBreakMsg);
					continue;
						}
			}
			
		
			ErrorReply('2');
			break;

        default:
            break;              /* silently ignore */
        }

        /* Send reply back.  Its either an error, a standard reply,
         * or a reply with some information.
         */
        PutPacket(PacketBuffer);
        continue;
 
    Itsok:
        /* Reply OK to request
         */
        PutPacket(OkMsg);
        continue;

    Bummer:
        /* Something wrong
         */
        PutPacket(ErMsg);
        continue;
    }

 Out:
	/* Resume the CPU
     */
    MyCpu->ExternalInterrupt(false);

    return (DebuggerAttached == TRUE);
}

/* Notify the debugger in case of load/unloads of additional modules.
 */
void
Emips2Gdb::DebuggerNotify(const char *ModuleName, ADDRESS LoadAddress, char Event)
{
    size_t count;
    unsigned char *p = PacketBuffer;

    if (!DebuggerAttached)
        return;

    count = strlen(ModuleName);

    *p++ = Event;
    Mem2Hex((const unsigned char *)ModuleName,p,(int)count);
    p += count*2;
    *p++ = ':';
    for (count = 0; count < 8; count++) {
        *p++ = hexchars[(LoadAddress>>28) & 0xf];
        LoadAddress <<= 4;
    }
    *p++ = 0;

    PutPacket(PacketBuffer);
}


/* 
 * Methods of the ToEmipsNull class
 */

IoResult ToEmipsNull::Read (BYTE *Buffer, DWORD nBytes, DWORD *nRead)
{
    DPRINTF(L_LINE,0,("ToEmipsNull::Read(BYTE *Buffer=%x, DWORD nBytes=%d, DWORD *nRead=%x (*%d))\n",
                      Buffer, nBytes, nRead, (nRead) ? *nRead : 0));

    memset(Buffer,0xff,nBytes);
    *nRead = nBytes;
    return Success;
}

IoResult ToEmipsNull::Write(BYTE *Buffer, DWORD nBytes, DWORD *nWritten)
{
    DPRINTF(L_LINE,0,("ToEmipsNull::Write(BYTE *Buffer=%x, DWORD nBytes=%d, DWORD *nWritten=%x (*%d))\n",
                      Buffer, nBytes, nWritten, (nWritten) ? *nWritten : 0));

    *nWritten = nBytes;
    return Success;
}

IoResult ToEmipsNull::CurrentStatus(void)
{
    return Success;
}

ToEmipsNull::ToEmipsNull(ConfigInfo *Info)
{
    DPRINTF(L_LINE,0,(":ToEmipsNull(ConfigInfo *Info=%x)\n",
            Info));
}

ToEmipsNull::~ToEmipsNull(void)
{
    DPRINTF(L_LINE,0,(":~ToEmipsNull(void)\n"));
}



/* 
 * Methods of the ToEmipsSerial class
 */

IoResult ToEmipsSerial::Read (BYTE *Buffer, DWORD nBytes, DWORD *nRead)
{
    BOOL b;
    IoResult r;

	int i;
	DWORD nReadTemp; 
	DWORD* PnReadTemp = &nReadTemp;
	BYTE* BufferElement;

    DPRINTF(L_LINE,0,("ToEmipsSerial::Read(BYTE *Buffer=%x, DWORD nBytes=%d, DWORD *nRead=%x (*%d))..",
                      Buffer, nBytes, nRead, (nRead) ? *nRead : 0));

    if (Line == INVALID_HANDLE_VALUE)
        return NotConnected;

	*nRead = 0;
	for (i=0 ; i<nBytes ; i++){
		BufferElement = Buffer + i;
		b = ReadFile(Line,
			         BufferElement, 1,
				     PnReadTemp, NULL);
		*nRead += nReadTemp;
		//printf("\nnRead = %d\n", *nRead);
	}
		

    if (b) {
        TRACE(L_LINE,0,
        {
            printf("gets");
            for (int i=0;i<*nRead;i++) printf(" %x",Buffer[i]);
        } );
        r = Success;
        goto Out;
    }
    EPRINTF(("rf le=%x\n",GetLastError()));
    r = Failed;
 Out:
    DPRINTF(L_LINE,0,(" ->%d,%d\n",r,*nRead));
    return r;
}

IoResult ToEmipsSerial::Write(BYTE *Buffer, DWORD nBytes, DWORD *nWritten)
{
    BOOL b;
    IoResult r;

    DPRINTF(L_LINE,0,("ToEmipsSerial::Write(BYTE *Buffer=%x, DWORD nBytes=%d, DWORD *nWritten=%x (*%d))..",
                      Buffer, nBytes, nWritten, (nWritten) ? *nWritten : 0));
    TRACE(L_LINE,0,
    {
        printf("gives");
        for (int i=0;i<nBytes;i++) printf(" %x",Buffer[i]);
    } );

    if (Line == INVALID_HANDLE_VALUE)
        return NotConnected;

    b = WriteFile(Line,
                  Buffer, nBytes,
                  nWritten, NULL);

    if (b) { r = Success; goto Out;}
    EPRINTF(("wf le=%x\n",GetLastError()));
    r = Failed;
 Out:
    DPRINTF(L_LINE,0,(" ->%d,%d\n",r,*nWritten));
    return r;
}

IoResult ToEmipsSerial::CurrentStatus(void)
{
    return (Line == INVALID_HANDLE_VALUE) ? NotConnected : Success;
}

ToEmipsSerial::ToEmipsSerial(ConfigInfo *Info)
{
    UINT InterByteTime;
    DCB ControlBlock;
    COMMTIMEOUTS Timeouts;
    char *Name = Info->SerialLine;
    UINT BaudRate = Info->BaudRate;

    DPRINTF(L_LINE,0,(":ToEmipsSerial(ConfigInfo *Info=%x) {%s,%d}\n",
            Info, Name, BaudRate));

    /* 
     *    Open & check the comm line
     */
    Line = CreateFile( Name, GENERIC_READ | GENERIC_WRITE,
                        0, /* exclusive access */
                        NULL, /* no security attributes */
                        OPEN_EXISTING,
						0,
						//FILE_FLAG_NO_BUFFERING |
						//FILE_FLAG_OVERLAPPED,
                        //Laser-programmed gate arraysLaser-programmed gate arraysLaser-programmed gate arraysLaser-programmed gate arraysLaser-programmed gate arraysFILE_FLAG_NO_BUFFERING,
                        NULL  /* hTemplate - must be NULL for comms */
                        );

    if (Line == INVALID_HANDLE_VALUE) {
        EPRINTF(("Could not open %s (le=%x)\n",Name,GetLastError()));
        return;
    }

    /* 
     *    Get what we wont change
     */
    ControlBlock.DCBlength = sizeof(ControlBlock);
    if (!GetCommState(Line,&ControlBlock)) {
        EPRINTF(("Warning:GetCommState() failed on %s (x%x)\n",Name,GetLastError()));
        /* 
         *    might be a named pipe, ignore
         */
    }

    /* 
     *    No parity, no flow..
     */
    ControlBlock.BaudRate = BaudRate;
    ControlBlock.fBinary = TRUE;     /* no EOF check */
    ControlBlock.fParity = FALSE;
    ControlBlock.fOutxCtsFlow = FALSE;
    ControlBlock.fOutxDsrFlow = FALSE;
    ControlBlock.fDtrControl = DTR_CONTROL_ENABLE;
    ControlBlock.fDsrSensitivity = FALSE;
    ControlBlock.fTXContinueOnXoff = TRUE;
    ControlBlock.fOutX = FALSE;
    ControlBlock.fInX = FALSE;
    ControlBlock.fErrorChar = FALSE;
    ControlBlock.fNull = FALSE;
    ControlBlock.fRtsControl = RTS_CONTROL_ENABLE;
    ControlBlock.fAbortOnError = FALSE;
    ControlBlock.ByteSize = 8;
    ControlBlock.Parity = NOPARITY;
    ControlBlock.StopBits = ONESTOPBIT;
    if (!SetCommState(Line,&ControlBlock)) {
        EPRINTF(("SetCommState() failed on %s (x%x), ignoring..\n",Name,GetLastError()));
        /* 
         *    might be a named pipe, ignore
         */
    }

    /* 
     *    By default, it will wait to fillup completely the read buffer
     *    before signalling completion.  So we have to set a timeout.
     *    Timeouts are in milliseconds units (none on xmit)
     *    Luckily, we can do the optimal thing of setting a timeout that
     *    (a) only starts after the first byte is received
     *    (b) renews itself for as long as bytes keep coming in
     *    (c) expires after that much time elapses between bytes
     */

    InterByteTime = BaudRate / 10; /* approx byte rate */
    InterByteTime = 1000 / InterByteTime; /* in msecs, most likely 0 */
    InterByteTime++;  /* roundup, avoid zero */
    DPRINTF(L_LINE,0,("SerialRead inter-byte timeout is %d msecs\n",InterByteTime));
    Timeouts.ReadIntervalTimeout = InterByteTime;
#if 1 /* turn this on for non-blocking reads */
//    Timeouts.ReadTotalTimeoutConstant = 2 * InterByteTime;
     Timeouts.ReadTotalTimeoutConstant = 1000;
     Timeouts.ReadTotalTimeoutMultiplier = 1;

//    Timeouts.ReadTotalTimeoutMultiplier = InterByteTime;
#else
    Timeouts.ReadTotalTimeoutConstant = 0;
    Timeouts.ReadTotalTimeoutMultiplier = 0;
#endif

	Timeouts.WriteTotalTimeoutMultiplier = 0;
    Timeouts.WriteTotalTimeoutConstant = 0;

   if (!SetCommTimeouts( Line, &Timeouts)) {
        EPRINTF(("Warning: SetCommTimeouts() failed (x%x), ignoring..\n",
               GetLastError()));
		//EPRINTF(("Warning: Connecting to a blocking NamedPipe ...\n"));  


    }

    /* The only indication all is fine is the handle is valid */
}

ToEmipsSerial::~ToEmipsSerial(void)
{
    DPRINTF(L_LINE,0,(":~ToEmipsSerial(void)\n"));
    if (Line != INVALID_HANDLE_VALUE)
        CloseHandle(Line);
    Line = INVALID_HANDLE_VALUE;

}



/* Arguments -> Configuration parameters
 */
ConfigInfo::ConfigInfo(int argc, char **argv)
{
    char *pgmname = argv[0];

    DebugLevel = 0;
    SerialLine = NULL;
    BaudRate = 115200;
    ErrorLevel = MAX_ERROR_LEVEL;
    ModulesToTrace = 0;

    /* 
     *    Parse invocation arguments, if any
     */
    static const char *Usage = "%s [-b<BAUDRATE>] [-verbose*] [-t<TRACEBITS>] COM1|COM2|...";

    while (argc > 1) {
        if (argv[1][0] == '-') {
            switch (argv[1][1]) {
            case 'b':
                BaudRate = atoi(argv[1]+2);
                printf("Baudrate will be %d baud\n", BaudRate);
                break;                
            case 't':
                ModulesToTrace = strtoul(argv[1]+2,NULL,16);
                printf("Will trace: "); Mask2Modules(ModulesToTrace);
                break;                
            case 'V': DebugLevel++;/*fallthrough*/
            case 'v': DebugLevel++;
                printf("DebugLevel = %d\n", DebugLevel);
                break;
            default:
                printf(Usage,pgmname);
                exit(-1);
            }
        } else {
            SerialLine = argv[1];
        }
        argc--,argv++;
    }

    ErrorLevel = (DebugLevel > MAX_ERROR_LEVEL) ?
                      0 : (MAX_ERROR_LEVEL - DebugLevel);

}

/* Entry point for this program
 */

int main(int argc, char ** argv)
{
    ConfigInfo Args = ConfigInfo(argc,argv);

    ToEmips   * CommLine = NULL;
    EmipsCpu  * Emips = NULL;
    Emips2Gdb * Interface = NULL;

    Config = &Args;

    if (Args.SerialLine)
        CommLine = new ToEmipsSerial(&Args);
    else {
        printf("Using NULL serial line (testing only)\n");
        CommLine = new ToEmipsNull(&Args);
    }
    if ((CommLine == NULL) || (CommLine->CurrentStatus() != Success)) {
        printf("Failed to create comm interface (to board)\n");
        return -1;
    }

    Emips = new EmipsCpu(&Args,CommLine);
    if (Emips == NULL) {
        printf("Failed to create eMIPS interface\n");
        return -1;
    }

    Interface = new Emips2Gdb(&Args,Emips);
    if (Interface == NULL) {
        printf("Failed to create GDB interface\n");
        return -1;
    }

    Interface->StartInterface();

    delete Interface;
    delete Emips;

    return 0;
}

