/* Extensible Peripheral Load/Unload test
 */
#include "print.h"
#include <mips\ml40x.h>

/* The following is lifted from PMT.c. BUGBUG Should #include it instead
 */

typedef char *STRING;
typedef struct _CONTROLLER {
    STRING TagName;
    UINT (*DumpController)(struct _Pmt *PmtEntry, STRING TagName);
} CONTROLLER, *PCONTROLLER;

/* Dumpers for specific controllers
 * Returns the number of  peripheral of its kind found in that range
 */
UINT AnyController(struct _Pmt *PmtEntry, STRING TagName)
{
    return 0;
}

/* A few look like the SRAM one */
UINT GenericController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Sram *pF = (struct _Sram *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;(pF->BaseAddressAndTag & SRAMBT_TAG) == PmtEntry->Tag; pF++) {
        Puts("\n\t");Puts(TagName);Puts("["); PutWord(Found);
        Puts("].AddresssAndTag = "); PutWord(pF->BaseAddressAndTag);
        Puts("\n\t");Puts(TagName);Puts("["); PutWord(Found);
        Puts("].Control = "); PutWord(pF->Control);

        Found++;
    }
    return Found;
}

UINT AicController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Aic *pF = (struct _Aic *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tAIC["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tAIC["); PutWord(Found);
        Puts("].IrqStatus = "); PutWord(pF->IrqStatus);

        Puts("\n\tAIC["); PutWord(Found);
        Puts("].IrqRawStatus = "); PutWord(pF->IrqRawStatus);

        Puts("\n\tAIC["); PutWord(Found);
        Puts("].IrqEnable = "); PutWord(pF->IrqEnable);

        Puts("\n\tAIC["); PutWord(Found);
        Puts("].IrqEnableClear = "); PutWord(pF->IrqEnableClear);

        Puts("\n\tAIC["); PutWord(Found);
        Puts("].IrqSoft = "); PutWord(pF->IrqSoft);

        Found++;
    }
    return Found;
}

UINT PioController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Pio *pF = (struct _Pio *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tPIO["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].Enable = "); PutWord(pF->Enable);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].Disable = "); PutWord(pF->Disable);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].Direction = "); PutWord(pF->Direction);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].OutDisable = "); PutWord(pF->OutDisable);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].PinData = "); PutWord(pF->PinData);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].PinData = "); PutWord(pF->PinData);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].ClearData = "); PutWord(pF->ClearData);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].PinStatus = "); PutWord(pF->PinStatus);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].IntrStatus = "); PutWord(pF->IntrStatus);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].IntrEnable = "); PutWord(pF->IntrEnable);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].IntrDisable = "); PutWord(pF->IntrDisable);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].IntrTrigger = "); PutWord(pF->IntrTrigger);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].reserved[0] = "); PutWord(pF->reserved[0]);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].reserved[1] = "); PutWord(pF->reserved[1]);

        Puts("\n\tPIO["); PutWord(Found);
        Puts("].reserved[2] = "); PutWord(pF->reserved[2]);

        Found++;
    }
    return Found;
}

UINT UsartController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Usart *pF = (struct _Usart *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tUSART["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].Control = "); PutWord(pF->Control);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].IntrEnable = "); PutWord(pF->IntrEnable);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].IntrDisable = "); PutWord(pF->IntrDisable);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].IntrMask = "); PutWord(pF->IntrMask);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].ChannelStatus = "); PutWord(pF->ChannelStatus);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].RxData = "); PutWord(pF->RxData);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].TxData = "); PutWord(pF->TxData);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].Baud = "); PutWord(pF->Baud);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].Timeout = "); PutWord(pF->Timeout);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].reserved[0] = "); PutWord(pF->reserved[0]);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].reserved[1] = "); PutWord(pF->reserved[1]);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].reserved[2] = "); PutWord(pF->reserved[2]);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].reserved[3] = "); PutWord(pF->reserved[3]);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].reserved[4] = "); PutWord(pF->reserved[4]);

        Puts("\n\tUSART["); PutWord(Found);
        Puts("].reserved[5] = "); PutWord(pF->reserved[5]);

        Found++;
    }
    return Found;
}

UINT TcController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Tc *pF = (struct _Tc *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tTC["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tTC["); PutWord(Found);
        Puts("].Control = "); PutWord(pF->Control);

        Puts("\n\tTC["); PutWord(Found);
        Puts("].FreeRunning = "); PutWord(pF->FreeRunning>>16);PutWord((UINT32)pF->FreeRunning);

        Puts("\n\tTC["); PutWord(Found);
        Puts("].DownCounter = "); PutWord(pF->DownCounterHigh);PutWord(pF->DownCounter);

        Puts("\n\tTC["); PutWord(Found);
        Puts("].reserved[0] = "); PutWord(pF->reserved[0]);

        Puts("\n\tTC["); PutWord(Found);
        Puts("].reserved[1] = "); PutWord(pF->reserved[1]);

        Found++;
    }
    return Found;
}

UINT WdController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Wd *pF = (struct _Wd *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tWD["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].OvflMode = "); PutWord(pF->OvflMode);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].ClockMode = "); PutWord(pF->ClockMode);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].Control = "); PutWord(pF->Control);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].Status = "); PutWord(pF->Status);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].reserved[0] = "); PutWord(pF->reserved[0]);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].reserved[1] = "); PutWord(pF->reserved[1]);

        Puts("\n\tWD["); PutWord(Found);
        Puts("].reserved[2] = "); PutWord(pF->reserved[2]);

        Found++;
    }
    return Found;
}

UINT PmcController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Pmc *pF = (struct _Pmc *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tPMC["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].SystemPowerEnable = "); PutWord(pF->SystemPowerEnable);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].SystemPowerDisable = "); PutWord(pF->SystemPowerDisable);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].PeripheralPowerEnable = "); PutWord(pF->PeripheralPowerEnable);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].PeripheralPowerDisable = "); PutWord(pF->PeripheralPowerDisable);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].reserved[0] = "); PutWord(pF->reserved[0]);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].reserved[1] = "); PutWord(pF->reserved[1]);

        Puts("\n\tPMC["); PutWord(Found);
        Puts("].reserved[2] = "); PutWord(pF->reserved[2]);

        Found++;
    }
    return Found;
}

UINT AceController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Sac *pF = (struct _Sac *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tACE["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].Control = "); PutWord(pF->Control);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].BUSMODEREG = "); PutWord(pF->BUSMODEREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].STATUS = "); PutWord(pF->STATUS);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].ERRORREG = "); PutWord(pF->ERRORREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].CFGLBAREG = "); PutWord(pF->CFGLBAREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].MPULBAREG = "); PutWord(pF->MPULBAREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].VERSIONREG = "); PutWord(pF->VERSIONREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].SECCNTCMDREG = "); PutWord(pF->SECCNTCMDREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].CONTROLREG = "); PutWord(pF->CONTROLREG);

        Puts("\n\tACE["); PutWord(Found);
        Puts("].FATSTATREG = "); PutWord(pF->FATSTATREG);

        Found++;
    }
    return Found;
}

UINT EcController(struct _Pmt *PmtEntry, STRING TagName)
{
    struct _Ec *pF = (struct _Ec *) (PmtEntry->TopOfPhysicalAddress << 16);
    UINT Found = 0;

    for (;pF->Tag == PmtEntry->Tag; pF++) {
        Puts("\n\tEC["); PutWord(Found);
        Puts("].Tag = "); PutWord(pF->Tag);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].Control = "); PutWord(pF->Control);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].SlotStatusAndTag = "); PutWord(pF->SlotStatusAndTag);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].BatOrSize[0] = "); PutWord(pF->BatOrSize[0]);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].BatOrSize[1] = "); PutWord(pF->BatOrSize[1]);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].BatOrSize[2] = "); PutWord(pF->BatOrSize[2]);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].BatOrSize[3] = "); PutWord(pF->BatOrSize[3]);

        Puts("\n\tEC["); PutWord(Found);
        Puts("].BatOrSize[4] = "); PutWord(pF->BatOrSize[4]);

        Found++;
    }
    return Found;
}

CONTROLLER AllControllers[PMTTAG_LAST_TAG_DEFINED+1] = {
    { "BRAM", AnyController },
    { "PMT", AnyController },
    { "SRAM", GenericController },
    { "DDRAM", AnyController },
    { "FLASH", GenericController },
    { "INTERRUPT_CONTROLLER", AicController },
    { "USART", UsartController },
    { "TIMER", TcController },
    { "WATCHDOG", WdController },
    { "GPIO", PioController },
    { "SYSTEM_ACE", AceController },
    { "LCD", AnyController },
    { "PS2", AnyController },
    { "VGA", AnyController },
    { "ETHERNET", AnyController },
    { "AC97", AnyController },
    { "POWER_MGR", PmcController },
    { "EXTENSION_CONTROLLER", EcController }
};

void PmtDump(void)
{
    struct _Pmt *Pmt = ThePmt;
    int maxe;
    PCONTROLLER pC;

    Puts("Peripheral Mapping Table, from the top at "); PutWord((UINT32) Pmt); PutChar('\n');

    for (maxe=0;maxe<100;maxe++,Pmt--) {
        Puts("Pmt["); PutWord(maxe); Puts("]: "); PutWord(*(UINT32*)Pmt);
        Puts(" -- Address = "); PutWord(Pmt->TopOfPhysicalAddress << 16);
        Puts(" Tag = "); PutWord(Pmt->Tag);
        
        if (Pmt->Tag < (PMTTAG_LAST_TAG_DEFINED+1)) {
            pC = &AllControllers[Pmt->Tag];
            PutChar(' '); Puts(pC->TagName);
            pC->DumpController(Pmt,pC->TagName);
        }
        if (Pmt->Tag == PMTTAG_END_OF_TABLE) {
            Puts(" END_OF_TABLE");
        }
        PutChar('\n');
        if (Pmt->Tag == PMTTAG_END_OF_TABLE)
            break;
    }
}

/* End of code lifted from PMT.c */

#define TheEc ((struct _Ec *)EXTENSION_CONTROLLER_DEFAULT_ADDRESS)

void DumpPeripheral(UINT32 Address, UINT Tag, char *TagName)
{
    struct _Pmt e;
    PCONTROLLER pC;

    e.Tag = Tag;
    e.TopOfPhysicalAddress = Address >> 16;

    if (Tag < (PMTTAG_LAST_TAG_DEFINED+1)) {
        pC = &AllControllers[Tag];
        pC->DumpController(&e,TagName);
    } else {
        Puts("Unknown Tag="); PutWord(Tag);
    }
    PutChar('\n');
}

void DumpTaggedPeripherals(UINT Tag)
{
    struct _Pmt *Pmt = ThePmt;
    int maxe;
    PCONTROLLER pC;

    if (Tag < (PMTTAG_LAST_TAG_DEFINED+1))
        pC = &AllControllers[Tag];
    else {
        Puts("Cannot dump unknown tag="); PutWord(Tag);PutChar('\n');
        return;
    }

    Puts("Currently enabled peripherals with tag="); PutWord(Tag);PutChar('\n');

    for (maxe=0;maxe<100;maxe++,Pmt--) {
        if (Pmt->Tag == Tag) {
            /* This dumps the whole segment */
            pC->DumpController(Pmt,pC->TagName);
            PutChar('\n');
        }
        if (Pmt->Tag == PMTTAG_END_OF_TABLE)
            break;
    }    
}

void DumpEc(void)
{
    DumpPeripheral(EXTENSION_CONTROLLER_DEFAULT_ADDRESS,
                   PMTTAG_EXTENSION_CONTROLLER,
                   "EC");
}

UINT32 FindSpaceForPeripheral(UINT Tag, UINT32 Size)
{
    struct _Pmt *Pmt = ThePmt;
    int maxe;
    UINT32 *pAddress;

    Puts("Making space for tag="); PutWord(Tag);
    Puts(" and size="); PutWord(Size); Puts("...\n");
    /* Start by looking if we have a segment available */
    for (maxe=0;maxe<100;maxe++,Pmt--) {
        if (Pmt->Tag == Tag) {
            Puts("\nTrying matching segment at "); PutWord(Pmt->TopOfPhysicalAddress << 16); Puts("...\n");
            
            for (pAddress = (UINT32 *)(Pmt->TopOfPhysicalAddress << 16);
                 (((UINT32)pAddress >> 16) == Pmt->TopOfPhysicalAddress);
                 ) {
                /* busy? */
                if ((*pAddress /*& 0x0000ffff*/) == Tag) {
                    pAddress = (UINT32*) ((char *)pAddress + Size); /* NONONONNONO!!! */
                } else
                    goto Found;
            }
        }
    }    

    return ~0;
 Found:
    Puts("\tFound room at "); PutWord((UINT32)pAddress); PutChar('\n');
    return (UINT32)pAddress;
}

char LoadTest(int SlotNo)
{
    int i;
    UINT32 Bat;
    UINT Tag;

    Puts("\n\tSlot #"); PutWord(SlotNo); Puts(" is not currently loaded, will perform a LOAD test\n\n");

    Puts("Waiting for the peripheral to be loaded (please use the JTAG or DBG module)...\n");
    for (i = 0; i < 1000; i++) {
        if (TheEc->Control & ECC_INT_LOAD)
            break;
        Delay(20 * 0x10000); // NB: about 1 sec
    }

    if (!(TheEc->Control & ECC_INT_LOAD) ||
        ((TheEc->Control & ECC_SLOTNO) != (SlotNo << ECC_SLOTNO_SHIFT))) {
        DumpEc();
        Puts("Extension not loaded or wrong slot#\n");
        return 'a';
    }

    Puts("Peripheral was loaded:\n");
    DumpEc();

    Tag = TheEc->SlotStatusAndTag & ECS_TAG;
    DumpTaggedPeripherals(Tag);

    if (TheEc->Control & ECC_PRIVILEDGED)
        Puts("Extension requires priviledged interface\n");

    /* Does it want BATs? (duh) */
    if ((TheEc->SlotStatusAndTag & ECS_STATUS) == ECS_CONFIG) {
        if (TheEc->BatOrSize[0] != 0) {
            Puts("Needs BAT config.\n");
            Bat = FindSpaceForPeripheral(TheEc->SlotStatusAndTag & ECS_TAG,
                                         TheEc->BatOrSize[0]);  
            if (Bat == ~0) {
                Puts("Could not find room for it\n");
                return 'c';
            }

            TheEc->BatOrSize[0] = Bat | ECB_BAT_VALID;
            TheEc->SlotStatusAndTag = ECS_RUN;
            DumpEc();
        }
        
        if (TheEc->Control & ECC_WANTS_INTERRUPT) {
            /* Should enable them when loading the device driver */
            Puts("Extension requires interrupt #");
            PutWord(31-SlotNo);
            PutChar('\n');
        }

        Puts("Configured ok.\n");
    }

    /* See if its visible now */
    Bat = TheEc->BatOrSize[0] & ECB_BAT;
    Puts("Checking peripherals at ");PutWord(Bat & 0xffff0000);PutChar('\n');
    DumpPeripheral(Bat, /* will round it down to 64k */
                   Tag,
                   "NewPeripheral");

    if ((*((UINT32*)Bat)) != Tag) {
        Puts("Wrong tag, mapping failed? "); PutWord(*((UINT32*)Bat)); PutChar('\n');
        return 'd';
    }

    /* Clear interrupt */
    TheEc->Control |= ECC_IRQ;

    return 'y';
}

char UnloadTest(int SlotNo)
{
    int i;
    UINT32 Bat;
    UINT Tag;

    Puts("\n\tSlot #"); PutWord(SlotNo); Puts(" is loaded, will perform an UNLOAD test\n\n");

    /* Say what its like today */
    Tag = TheEc->SlotStatusAndTag & ECS_TAG;
    DumpTaggedPeripherals(Tag);

    /* Check & Remove all active mappings */
    if ((TheEc->SlotStatusAndTag & ECS_STATUS) > ECS_CONFIG) {

        /* Check that the peripheral base mapping was working ok */
        Bat = TheEc->BatOrSize[0];
        if (Bat & ECB_BAT_VALID) {
            UINT32 *Registers = (UINT32 *) (Bat & ECB_BAT);
            if (*Registers != Tag) {
                Puts("Peripheral not answering to BAT[0]? BAT=");
                PutWord(Bat);
                Puts(" *BAT=");
                PutWord(*Registers);
                Puts(" != Tag=");
                PutWord(Tag);
                PutChar('\n');
                return 'a';
            }
        }

        /* Remove all BATs */
        for (i = 0; i < EC_MAX_BATS; i++) {
            TheEc->BatOrSize[i] = 0;
        }

        /* Verify mappings are removed */
        if (Bat & ECB_BAT_VALID) {
            UINT32 *Registers = (UINT32 *) (Bat & ECB_BAT);
            if (*Registers == Tag) {
                Puts("Peripheral still answering to BAT[0]? BAT=");
                PutWord(Bat);
                Puts(" *BAT=");
                PutWord(*Registers);
                Puts(" == Tag=");
                PutWord(Tag);
                PutChar('\n');
                return 'b';
            }
        }
    }


    /* Force unload */
    TheEc->SlotStatusAndTag = (TheEc->SlotStatusAndTag & ~ECS_STATUS) | ECS_ABSENT;

    /* Wait for unload complete */
    Puts("Waiting for the peripheral to unload...\n");
    for (i = 0; i < 10; i++) {
        if (TheEc->Control & ECC_INT_UNLOAD)
            break;
        Delay(20 * 0x10000); // NB: about 1 sec
    }

    if (!(TheEc->Control & ECC_INT_UNLOAD) ||
        ((TheEc->Control & ECC_SLOTNO) != (SlotNo << ECC_SLOTNO_SHIFT))) {
        DumpEc();
        Puts("Extension not unloaded or wrong slot#\n");
        return 'c';
    }

    Puts("Peripheral was unloaded:\n");
    DumpEc();

    /* Say what its like now */
    DumpTaggedPeripherals(Tag);

    /* Clear interrupt */
    TheEc->Control |= ECC_IRQ;

    return 'y';
}

BOOL SlotIsLoaded(int SlotNo)
{
    /* Do we even have an extension controller.
     */
    if (TheEc->Tag != PMTTAG_EXTENSION_CONTROLLER) {
        Puts("EC has wrong tag=");PutWord(TheEc->Tag);PutChar('\n');
        return '?';
    }

    /* Select slot # */
    TheEc->Control = SlotNo << ECC_SLOTNO_SHIFT; /* Size is r/o */
    DumpEc();
    if ((TheEc->SlotStatusAndTag & ECS_STATUS) == ECS_ABSENT)
        return 'n';
    return 'y';
}

void main(char *StackPointer)
{
    char Step = 'a'-1;

    /* Wait fro go
     */
    Delay(20 * 0x10000); // NB: about 1 sec
    Step++;

    /* See what we'll do:
     * If Slot#0 is loaded we'll unload it
     * Otherwise we wait for a signal that its loaded and configure it
     */
    Puts("Extensible Peripheral Test (using slot #0)\n");
    Step = SlotIsLoaded(0);
    if (Step == 'y')
        Step = UnloadTest(0);
    else if (Step == 'n')
        Step = LoadTest(0);

    /* END: all is well?
     */ 
    if (Step != 'y')
        goto error;
    Puts("TEST PASSED SUCCESSFULLY\n");
    return;

error:
    Puts("TEST FAILED at Step=");
    PutChar(Step);
    Puts("\n");
    return;
}
