/* Copyright (c) Microsoft Corporation. All rights reserved. */
#include "snake.h"

#include <mips\ml40x.h>
#define TheUsart ((struct _Usart *)USART_DEFAULT_ADDRESS)
#define TheTc  ((struct _Tc  *)TIMER_DEFAULT_ADDRESS)
typedef UINT32 TIME;

/* This one assumes hw division does work */
void PutDecWord(UINT v)
{
#define MAXDIGS 10 /* 32 bits! */
    char num[MAXDIGS+1];
    int i = MAXDIGS;
    UINT r;

    num[i--] = 0;
    for (;;i--) {
        r = v % 10;
        v = v / 10;
        num[i] = '0' + r;
        if ((v == 0) || (i == 0)) break;
    }

    Puts(num+i);
}

#define PutWord PutDecWord  /* delete if hw still broken */

/* Button handling
 */
int CurrentDirection = PAUSE;

void OnButtonPress(int ButtonPressed)
{
    switch (ButtonPressed) {
    case Button1: /* oause */
        CurrentDirection = PAUSE;
        break;
    case Button2: /* exit */
        CurrentDirection = TERMINATE;
        break;
    case Button4: /* left */
        CurrentDirection = LEFT;
        break;
    case Button5: /* right */
        CurrentDirection = RIGHT;
        break;
    case Button3: /* up */
        CurrentDirection = UP;
        break;
    case Button6: /* down */
        CurrentDirection = DOWN;
        break;
    }
}

/* Pseudo-random number generator
 */
TIME CurrentTime(void)
{
    UINT64 Frc = TheTc->FreeRunning;
    UINT32 v = (UINT32)Frc;
    return v;
}

/* Stupid but quick */
UINT16 Random(void)
{
    static UINT16 Previous = 0;
    TIME Now = CurrentTime();
    UINT16 Next;

    /* Should overflow in ~2 secs */
    Next = (UINT16)(Now >> 8);
    if (Previous == 0) {
        Previous = Next;
        return Random();
    }
    Previous = Previous + Next;
    return Previous;
}

/* Game plans
 */
#define MAX_LEVEL 7
#define MAX_GRADE 2
typedef struct {
    UINT16 NumGoodies;
    UINT16 Speed;
} LEVEL_INFO;
typedef LEVEL_INFO GRADE_INFO[MAX_LEVEL];
const GRADE_INFO LevelInfo[MAX_GRADE] = {
  { /* normal */
    { 1, 400 },
    { 2, 200 },
    { 3, 150 },
    { 4, 100 },
    { 5, 80 },
    { 6, 80 },
    { 7, 60 }
  },
  { /* harder */
    { 3, 150 },
    { 4, 100 },
    { 5, 100 },
    { 6, 80 },
    { 7, 80 },
    { 8, 80 },
    { 9, 60 }
  }
};

#define CurSpeed LevelInfo[CurGrade][CurLevel].Speed
#define CurGoodies LevelInfo[CurGrade][CurLevel].NumGoodies

UINT8 GameMap[NUM_PX];
UINT8 CurLevel = 0;
UINT8 CurGrade = 0;

/* Game status
 */
UINT16 GoodiesLeft = 0;
UINT16 CurPos = 0;
UINT16 CurTime;
UINT CurScore, HighScore;
UINT TimeLimit;

void CheckHighScore(UINT Score)
{
    if (Score > HighScore) {
        Puts("High Score!! [Previous HighScore ");
        PutWord(HighScore);
        Puts("] ");
        HighScore = Score;
    }
}

void YouWin(void)
{
    Smile();
    Puts("\r\nYou win! Score ");
    PutWord(CurScore);
    Puts("\033[K\r\n");
    CheckHighScore(CurScore);
}

void YouLose(int Level)
{
    Puts("\r\nYou lost on level ");
    PutWord(Level);
    Puts(", score ");
    PutWord(CurScore);
    Puts("\033[K\r\n");
    CheckHighScore(CurScore);
}

void ShowStatus(void)
{
    Puts("\033[25;1H\rScore: ");
    PutWord(CurScore);
    Puts(" Goodies left: ");
    PutWord(GoodiesLeft);
    Puts(" Level ");
    PutWord(CurLevel);
    Puts(" Up/Down/Left/Right?");
}

BOOL AskIfPlayAgain(int GameNumber)
{
    int yn;

    Delay(0x10000);

    Puts("\r\nWanna play");
    if (GameNumber) Puts(" again");
    Puts("[yn]?\033[K");
 retry:
    yn = GetChar();
    if ((yn == _T('y')) || (yn == _T('Y')))
        yn = 'y';
    else
    if ((yn == _T('n')) || (yn == _T('N')))
        yn = 'n';
    else
        goto retry;
    PutChar(yn);
    return (yn == 'y');
}

void NewMap(void)
{
    int i;
    UINT16 r = 0;

    ZeroMap(GameMap,PX_FREE);
    NOISE(Puts("Map cleared\n"));

    GoodiesLeft = CurGoodies;
    for (i = GoodiesLeft; i >= 0; i--) {
        r = Random() % NUM_PX;
        if (GetPixel(GameMap,r) != PX_FREE) {
            i++;
            continue;
        }
        SetPixel(GameMap,r, (UINT8)((i == 0) ? PX_BUSY : PX_GOODY));
    }
    CurPos = r;
    CurrentDirection = PAUSE;
    TimeLimit = NUM_PX/8;
    CurTime = 0;
    ShowStatus();
}

/* TRUE -> not done */
BOOL GameTick(void)
{
    UINT8 CurX, CurY;

    CurY = CurPos / MAP_WIDTH;
    CurX = CurPos - (CurY * MAP_WIDTH);

    switch (CurrentDirection) {
    case LEFT: if (CurX) CurX--; break;
    case RIGHT: if (CurX < (MAP_WIDTH-1)) CurX++; break;
    case UP: if (CurY < (MAP_HEIGHT-1)) CurY++; break;
    case DOWN: if (CurY) CurY--; break;
    case PAUSE: CurTime--; return TRUE;
    default:
        return FALSE;
    }
    CurPos = (CurY * MAP_WIDTH) + CurX;

    switch (GetPixel(GameMap,CurPos)) {
    case PX_BUSY:
        /* you lose */
        return FALSE;
    case PX_GOODY:
        CurScore += (TimeLimit > CurTime) ?
            (TimeLimit - CurTime) * (CurLevel + 1) : 1;
        GoodiesLeft--;
        ShowStatus();
        if (GoodiesLeft == 0) 
            /* level clear */
            return FALSE;
        /* fall through */
    case PX_FREE:
        SetPixel(GameMap,CurPos,PX_BUSY);
        return TRUE;
    }
    return FALSE;
}

void Sleep(int nMsecs)
{
    /* poll serial line, if any char pickup */
    int i;
    for (i = 0; i < nMsecs; i++) {
        if (TheUsart->ChannelStatus & USI_RXRDY) {
            char c = TheUsart->RxData;
            OnButtonPress(c);
            return;
        }
        Delay(0x1000);
    }

}

/* TRUE -> level complete */
BOOL NewGame(void)
{
    NOISE(Puts("NewMap()..."));
    NewMap();
    NOISE(Puts("Done.\n"));

    do {
        ShowMap(GameMap);
        NOISE(Puts("MapSown, sleeping.."));
        Sleep(CurSpeed);
        NOISE(Puts("Done.\n"));
        CurTime++;
    } while (GameTick());

    if (GoodiesLeft)
        YouLose(CurLevel);
    return (GoodiesLeft == 0);
}

int main(char *StackPointer)
{
    UINT i = 0;

    TheTc->Control = TCCT_ENABLE | TCCT_INT_ENABLE;

    for (;;) {

        if (!AskIfPlayAgain(i))
            break;

        i = 1;

        InitScreen();

        CurScore = 0;
        CurLevel = 0;
        while (NewGame()) {
            if (++CurLevel < MAX_LEVEL)
                continue;
            YouWin();
            break;
        }
    }

    Puts("TEST PASSED SUCCESSFULLY\n");
    return 0;
}

/* Inlining teh console handler */


void InitScreen(void)
{
    Puts("\033[2J");
}

void ShowMap(MAP Map)
{
    UINT8 p;
    int i, x = 0;

    /* Home cursor */
    Puts("\033[H\r"); 
    for (i = 0; i < NUM_PX; i++) {
        p = GetPixel(Map,i);
        PutChar( (p == PX_FREE) ? ' ' :
               ((p == PX_GOODY) ? 'X' : 'o'));
            
        if (++x == MAP_WIDTH) {
            x = 0;
            PutChar('|');
            if (i < (NUM_PX-1))
                Puts("\r\n");
        }
    }
}

void Smile(void)
{
    /* not yet */
}


int memset(void *d, int v, UINT c)
{
    char *bd = (char *)d;
    char *be = bd + c;

    while (bd < be)
        *(bd++) = v;

    return c;
}
