/* Tests the behavior of the downcounter
 */

#include "print.h"
#include <mips\ml40x.h>

#define TheAic ((struct _Aic *)INTERRUPT_CONTROLLER_DEFAULT_ADDRESS)
#define TheTc  ((struct _Tc  *)TIMER_DEFAULT_ADDRESS)

void DumpUint32(char *What, UINT32 Value)
{
    Puts(What);
    PutWord(Value);
    PutChar('\n');
}

void DumpAicRegisters(void)
{
    Puts("-- AIC --\n");
    DumpUint32("Tag=",TheAic->Tag);
    DumpUint32("IrqStatus=",TheAic->IrqStatus);
    DumpUint32("IrqRawStatus=",TheAic->IrqRawStatus);
    DumpUint32("IrqEnable=",TheAic->IrqEnable);
    DumpUint32("IrqEnableClear=",TheAic->IrqEnableClear);
}

void DumpTcRegisters(void)
{
    UINT64 v;
#define DumpRegister64(_n_,_h_,_l_) { \
    Puts(_n_); \
    PutWord(_h_); PutChar('.'); PutWord(_l_); \
    PutChar('\n'); \
}
    Puts("-- TC --\n");
    DumpUint32("Tag=",TheTc->Tag);
    DumpUint32("Control=",TheTc->Control);
    DumpRegister64("DownCounter=",TheTc->DownCounterHigh,TheTc->DownCounter);
    v = TheTc->FreeRunning;
    DumpRegister64("FreeCounter=",(UINT32)(v>>32), (UINT32)v);
#undef DumpRegister64
}

void DumpPsr(void)
{
    Puts("-- COPROC-0 --\n");
    DumpUint32("psr=", GetPsr());
    DumpUint32("cause=", GetCause());
    DumpUint32("epc=", GetEpc());
}



void main(char *StackPointer)
{
    char Step = 'a'-1;
    int i, n;
    volatile UINT32 x, y;
    UINT32 psr;

    /* Wait for go
     */
    Delay(20 * 0x10000); // NB: about 1 sec

    /* Test utils */
    Step++;
    DumpUint32("HiMom! sp=", (UINT32)StackPointer);

    /* Check the tags, bail out if wrong
     */
    Step++;
    if (TheTc->Tag != PMTTAG_TIMER) {
        DumpUint32("?TimerTag=", TheTc->Tag);
        goto error;
    }
    if (TheAic->Tag != PMTTAG_INTERRUPT_CONTROLLER) {
        DumpUint32("?AicTag=", TheAic->Tag);
        goto error;
    }

    /* Dump all initial register values
     */
    Step++;
    DumpTcRegisters();
    DumpPsr();
    Puts("Testing begins.\n");

    /*
     * SPEC0: We should be able to repeat at will
     */
    for (n = 1000; n < 1000000; n *= 8) {

        /*
         * SPEC1: We should be able to stop the counter
         */
        Step++;
        Puts("\tHalting the counter...");

        TheTc->Control = TCCT_RESET; /* because we do not enable it */
        Delay(10);       /* make sure it takes, not clear if needed */

        /* CHECK: Control should have the expected value
         */
        if (TheTc->Control != 0) {
            DumpUint32("Non-zero Control=",TheTc->Control);
            goto error;
        }

        /* CHECK: Counter should not move
         */
        for (x = 0; x < 0x10000; x++)
            if (TheTc->DownCounter != 0) {
                DumpUint32("Counter not stopped=",TheTc->DownCounter);
                goto error;
                break;
            }
        Puts("ok.\n");


        /*
         * SPEC2: While stopped, we should be able to read-write from it
         */
        Step++;
        Puts("\tRead/Write to a stopped counter...");

        TheTc->DownCounterHigh = 0x0badf00d;
        TheTc->DownCounter = 0xfadefade;

        /* CHECK: we should read back what we wrote
         */
        if ((TheTc->DownCounterHigh != 0x0badf00d) ||
            (TheTc->DownCounter != 0xfadefade)) {
            Puts("FAILED: should be high=0xbadf00d low=0xfadefade");
            DumpTcRegisters();
            goto error;
        }
        Puts("ok.\n");


        /*
         * SPEC3: It should count down from where we start it
         */
        Step++;
        Puts("\tCount down tests...");

        TheTc->Control = TCCT_ENABLE | TCCT_INT_ENABLE;

        /* CHECK: Control should tell us its running
         */
        if (TheTc->Control != (TCCT_ENABLE | TCCT_INT_ENABLE)) {
            Puts("FAILED: Control != ENABLE+INTENABLE");
            DumpTcRegisters();
            goto error;
        }

        /* CHECK: should have started to move down
         */
        Delay(100);
        x = TheTc->DownCounter;
        if ((TheTc->DownCounterHigh > 0x0badf00d) ||
            (x >= 0xfadefade)) {
            Puts("FAILED: should be less-than high=0xbadf00d low=0xfadefade");
            DumpTcRegisters();
            goto error;
        }

        /* CHECK: should keep on moving down
         */
        for (i = 0; i < 0x10000; i++) {
            Delay(10);
            y = TheTc->DownCounter;
            if ((TheTc->DownCounterHigh > 0x0badf00d) ||
                (y >= x)) {
                Puts("FAILED: not moving down");
                DumpTcRegisters();
                goto error;
            }
            x = y;
        }

        Puts("ok.\n");

        /*
         * SPEC4: We can change it on the fly, without stopping it
         */
        Step++;
        Puts("\tRestart test...");

        /* SPEC4a: The correct procedure is high-first, then low
         */
        TheTc->DownCounterHigh = 0;
        TheTc->DownCounter = n;

        if ((TheTc->DownCounterHigh != 0) ||
            (TheTc->DownCounter > n)) {
                Puts("FAILED: did not take");
                DumpTcRegisters();
                goto error;
            }

        Puts("ok.\n");

        /*
         * SPEC5: It should trigger TCCT_INTERRUPT when wrapping around zero
         */
        Step++;
        Puts("\tWrap-around test...");

        /* CHECK: Should wrap
         */
        for (i = 0; i < 0x1000000; i++) {
            x = TheTc->DownCounter;
            y = TheTc->DownCounterHigh;
            if (x > n)
                break;
        }

        /* CHECK: Should keep counting down from ff...ff
         */
        if ((x <= n) || (y != 0xffffffff || (0 == (x & 0x80000000)))) {
            Puts("FAILED: did not wrap-around properly");
            DumpTcRegisters();
            DumpPsr();
            goto error;
        }

        /* CHECK: Should signal wrap-around
         */
        y = TheAic->IrqRawStatus;
        x = TheTc->Control;
        if (x != (TCCT_ENABLE|TCCT_INT_ENABLE|TCCT_INTERRUPT)) {
            DumpUint32("FAILED: did not signal wrap-around, Control=", x);
            DumpTcRegisters();
            DumpPsr();
            goto error;
        }

        /* CHECK: Wrap-around should trigger an interrupt
         */
        if (0 == (y & AIC_TIMER_BIT)) {
            DumpUint32("FAILED: did not trigger interrupt #0, Aic->IrqRawStatus=",y);
            DumpTcRegisters();
            DumpPsr();
            goto error;
        }

        Puts("ok.\n");

    }

    /* END: all is well
     */ 
    Puts("Tested ok.\n");
    DumpTcRegisters();

    Step++;
    Puts("TEST PASSED SUCCESSFULLY\n");
    return;

error:
    Puts("TEST FAILED at Step=");
    PutChar(Step);
    Puts("\n");
    return;
}
