/* WINDOWS test driver
 */

#include <windows.h>
#include <stdio.h>

int debug_level = 0;
int Bytes2Send = 16;

/* 
 *    Board interface
 */
typedef struct _BOARD {
    BOOL bInitialized;

    /* NT handles for sending&receiving. */
    char *Name;
    HANDLE hFile;

    /* The board speed&address. */
    ULONG BaudRate;

} BOARD, * PBOARD;

/*
 *    Constructor for board over serial line.
 */
PBOARD BoardInitialize(PBOARD Serial, BOOL Quietly)
{
    UINT InterByteTime;
    DCB ControlBlock;
    COMMTIMEOUTS Timeouts;
    HANDLE hLine, IoCompletionPort;
    BOOL IsPipe = FALSE;

    /* 
     *    Open & check the comm line
     */
    hLine = CreateFile( Serial->Name, GENERIC_READ | GENERIC_WRITE,
                        FILE_SHARE_WRITE, /* exclusive access (except PnP..) */
                        NULL, /* no security attributes */
                        OPEN_EXISTING,
                        0, //FILE_FLAG_OVERLAPPED,
                        NULL  /* hTemplate - must be NULL for comms */
                        );

    if (hLine == INVALID_HANDLE_VALUE) {
        if (!Quietly)
            printf("Could not open %s\n",Serial->Name);
        return NULL;
    }

    Serial->hFile = hLine;
    IsPipe = Serial->Name[0] == '\\';

    /* 
     *    Get what we wont change
     */
    ControlBlock.DCBlength = sizeof(ControlBlock);
    if (!GetCommState(hLine,&ControlBlock) && !IsPipe) {
        printf("GetCommState() failed on %s (x%x)\n",Serial->Name,GetLastError());
        /* 
         *    might be a named pipe, ignore
         */
    }

    /* 
     *    No parity, no flow..
     */
    ControlBlock.BaudRate = Serial->BaudRate;
    ControlBlock.fBinary = TRUE;     /* no EOF check */
    ControlBlock.fParity = FALSE;
    ControlBlock.fOutxCtsFlow = FALSE;
    ControlBlock.fOutxDsrFlow = FALSE;
    ControlBlock.fDtrControl = DTR_CONTROL_ENABLE;
    ControlBlock.fDsrSensitivity = FALSE;
    ControlBlock.fTXContinueOnXoff = TRUE;
    ControlBlock.fOutX = FALSE;
    ControlBlock.fInX = FALSE;
    ControlBlock.fErrorChar = FALSE;
    ControlBlock.fNull = FALSE;
    ControlBlock.fRtsControl = RTS_CONTROL_ENABLE;
    ControlBlock.fAbortOnError = FALSE;
    ControlBlock.ByteSize = 8;
    ControlBlock.Parity = NOPARITY;
    ControlBlock.StopBits = ONESTOPBIT;
    if (!SetCommState(hLine,&ControlBlock) && !IsPipe) {
        printf("SetCommState() failed on %s (x%x)\n",Serial->Name,GetLastError());
        /* 
         *    might be a named pipe, ignore
         */
    }

    /* 
     *    Doing overlapped I/O on the serial line under NT is funny.
     *    By default, it will wait to fillup completely the read buffer
     *    before signalling completion.  So we have to set a timeout.
     *    Timeouts are in milliseconds units (none on xmit)
     *    Luckily, we can do the optimal thing of setting a timeout that
     *    (a) only starts after the first byte is received
     *    (b) renews itself for as long as bytes keep coming in
     *    (c) expires after that much time elapses between bytes
     */

    InterByteTime = Serial->BaudRate / 10; /* approx byte rate */
    InterByteTime = 1000 / InterByteTime; /* in msecs, most likely 0 */
    InterByteTime++;  /* roundup, avoid zero */
    if (debug_level)
        printf("SerialRead inter-byte timeout is %d msecs\n",InterByteTime);
    Timeouts.ReadIntervalTimeout = InterByteTime;
    Timeouts.ReadTotalTimeoutConstant = 0;
    Timeouts.ReadTotalTimeoutMultiplier = 0;

    Timeouts.WriteTotalTimeoutMultiplier = 0;
    Timeouts.WriteTotalTimeoutConstant = 0;

    if (!SetCommTimeouts( hLine, &Timeouts) && !IsPipe) {
        printf("SetCommTimeouts() failed (x%x), ignoring..\n",
               GetLastError());
    }

    /*
     *    Done initializing
     */
    Serial->bInitialized = TRUE;

    return Serial;
}

void BoardFinalize( PBOARD Board)
{
    CloseHandle(Board->hFile);
    Board->hFile = INVALID_HANDLE_VALUE;
    Board->bInitialized = FALSE;
}

/* We want to run tests that are compiled in or that come from a file
 */
typedef char *STRING;
typedef struct _TEST_LIST TEST_LIST, *PTEST_LIST;
struct _TEST_LIST {
    BOOL (*Initialize)(PTEST_LIST This);
#define MAXLINE 128
    BOOL (*Next)(PTEST_LIST This, char *CmdLine);
    STRING Name;
    void *State;
};

/* Builtin test list
 */
STRING AllTests[] = {
    "treset.bin",
    "tflash.bin",
    /*"tsysace.bin",*/
    "tusart1.bin aaaaaaaaaaaa",
    "tusart.bin",
    "thfs2.bin 1%234567890abcd",
    "tthreads.bin",
    "tth1.bin #2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345#2345",
    "tsb.bin",
    "tprintf.bin",
    "pmt.bin",
    "techo.bin TEST PASSED SUCCESSFULLYq",
    "tdown.bin",
    "tdiv2.bin",
    "tdiv.bin",
    "ctint.bin",
    "ctimer.bin",   // bug
    "neg1.bin",
    "tgpio.bin",
    "snake.bin y                             qqqqqqn",
};
#define NTESTS ((sizeof AllTests) / (sizeof AllTests[0]))

BOOL BuiltinInitialize(PTEST_LIST This)
{
    This->State = (void *)0;
    return TRUE;
}

BOOL BuiltinNext(PTEST_LIST This, char *CmdLine)
{
    UINT Index = (UINT)This->State;

    if (Index >= NTESTS)
        return FALSE;
    strcpy(CmdLine,AllTests[Index]);
    This->State = (void*)(Index + 1);

    return TRUE;
}

TEST_LIST BuiltinList = {
    BuiltinInitialize,
    BuiltinNext,
    "DefaultTestsList",
    NULL
};

/* Tests in a script
 */
BOOL FileInitialize(PTEST_LIST This)
{
    This->State = fopen(This->Name,"rb");

    return (This->State != NULL);
}

BOOL FileNext(PTEST_LIST This, char *CmdLine)
{
    FILE *f = (FILE *)This->State;
    int c, nchars = 0;

    for (;f != NULL;) {
        c = fgetc(f);
        if (c == EOF) {
            fclose(f);
            f = NULL;
            This->State = NULL;
        } else {
            CmdLine[nchars++] = (char)c;
            if (debug_level) printf("%x ",c);
        }
        switch (c) {
        case '\n':
        case '\r':
            /* ignore them at start of line */
            if (nchars == 1) {
                nchars = 0;
                continue;
            }
        case 0:
            CmdLine[nchars-1] = 0;
            f = NULL;
            break;
        default:
            break;
        }
    }

    return (nchars > 0);
}

TEST_LIST FileList = {
    FileInitialize,
    FileNext,
    NULL,
    NULL
};

PTEST_LIST Tests = &BuiltinList;

/* Download a binary file to the board
 * BUGBUG Should probably just do a system() instead BUGBUG
 */
BOOL WaitForLeaderByte = FALSE;
BOOL WaitForEcho = TRUE;
BOOL Verbose = FALSE;

BOOL Download(HANDLE Serial, char *Filename)
{
    HANDLE b;
    int n = 0, le;
    DWORD Filesize = 0, Wordsize;
    int Byte = 0, Ack = 0;
    UINT nRead, nWritten;
    char *msg;
#define FAIL(_m_) {msg = _m_;goto Bad;}

    b = CreateFile(Filename, 
                   GENERIC_READ,
                   FILE_SHARE_READ,
                   NULL,
                   OPEN_EXISTING,
                   FILE_ATTRIBUTE_NORMAL,
                   NULL);
    if (b == INVALID_HANDLE_VALUE) {
        printf("Cant open %s, le=x%x\n", Filename, GetLastError());
        return FALSE;
    }

    Filesize = GetFileSize(b,NULL);
    if (Verbose)
        printf("%s is %d bytes long (%08x)\n", Filename, Filesize, Filesize);

    /* Wait for leader?
     */
 WaitForTheLeader:
#define LEADER_BYTE 'l' /*x6c*/
    if (WaitForLeaderByte) {
        printf("Waiting for leader byte...\n");
        if (!ReadFile(Serial,&Byte,1,&nRead,NULL) ||
            (nRead != 1))
            FAIL("Cant read leader byte");
        if (Byte != LEADER_BYTE)
            FAIL("Bad leader byte");
    }

    /* Send size over
     */
    Wordsize = ((Filesize + 3) & ~3) >> 2;
    if (Verbose)
        printf("Sending size (as %d words)...\n", Wordsize);
    for (n = 0; n < 4; n++) {
        Byte = (Wordsize >> (8*(3-n))) & 0xff;
        if (!WriteFile(Serial,&Byte,1,&nWritten,NULL) ||
            (nWritten != 1))
            FAIL("Cant write data byte");

        if (WaitForEcho) {
            if (!ReadFile(Serial,&Ack,1,&nRead,NULL) ||
                (nRead != 1))
                FAIL("Cant read ack byte");

            if (Verbose)
                printf("{%x %x}\n",Byte,Ack);

            if (Byte != Ack)
                FAIL("Bad ACK byte");
        }
    }

    /* Wait for the 's' to ACK the size
     */
    if (WaitForEcho) {
        if (Verbose)
            printf("Waiting for 's' ACK of size...\n");
        if (!ReadFile(Serial,&Ack,1,&nRead,NULL) ||
            (nRead != 1))
            FAIL("Cant read ack byte");
        if (Verbose)
            printf("{%x}\n",Ack);

        if (Ack != 's'/*x73*/)
            FAIL("Bad size-ACK byte");
    }

    /* Loop reading one byte, sending it, waiting for ACK
     */
    for (n = 0; n < Filesize; n++) {
        if (!ReadFile(b,&Byte,1,&nRead,NULL) ||
            (nRead != 1)) {
            if (GetLastError() == 0)
                break;
            FAIL("Cant read data byte");
        }

        if (!WriteFile(Serial,&Byte,1,&nWritten,NULL) ||
            (nWritten != 1))
            FAIL("Cant write data byte");

        if (0 == (n & 0xff))
            printf(".");

        if (WaitForEcho) {
            if (!ReadFile(Serial,&Ack,1,&nRead,NULL) ||
                (nRead != 1))
                FAIL("Cant read ack byte");

            if (Verbose)
                printf("{%x %x}\n",Byte,Ack);

            if (Byte != Ack)
                FAIL("Bad ACK byte");
        }
    }

    /* Roundup to words, if we have to
     */
    Filesize = (Filesize + 3) & ~3;
    for (;n < Filesize; n++) {
        Byte = 0;
        if (!WriteFile(Serial,&Byte,1,&nWritten,NULL) ||
            (nWritten != 1))
            FAIL("Cant write data byte");

        if (WaitForEcho) {
            if (!ReadFile(Serial,&Ack,1,&nRead,NULL) ||
                (nRead != 1))
                FAIL("Cant read ack byte");

            if (Verbose)
                printf("{%x %x}\n",Byte,Ack);

            if (Byte != Ack)
                FAIL("Bad ACK byte");
        }
    }

    if (WaitForEcho)
        printf("Download complete, ");
    printf("%d bytes sent\n", n);
    CloseHandle(b);
    return TRUE;
 Bad:
    le = GetLastError();
    printf("After %d bytes, le=x%x, download failed: %s\n",n,le,msg);
    if ((n == 0) && (le == 0) && WaitForLeaderByte)
        goto WaitForTheLeader;
    CloseHandle(b);
    return FALSE;

}

/* Util: matches a string somewhere in another
 */
UINT Match(char *Src, char *Target)
{
    UINT Matched = 0;
    char s, t;

    s = *Src++;
    t = Target[Matched];
    for (;s && t;) {
        if (s == t) {
            Matched++;
        }
        else {
            Matched = 0;
        }
        s = *Src++;
        t = Target[Matched];
    }

    return Matched;
}

/* Runs one test, returns TRUE iff passed
 */
#define BUFFER_SIZE 1024
BOOL OneTest(PBOARD Board, char *CmdLine)
{
    BYTE InBuffer[BUFFER_SIZE];
    UINT nBytes, nBytesRead = 0, nMatched = 0;
    BOOL bOk, bDone = FALSE;
    char *p, *input, *match;
    static char EndOfTest[] = "TEST PASSED SUCCESSFULLY";

    /* Split teh command line into <test><input>
     */
    input = strchr(CmdLine,' ');
    if (input) {
        *input++ = 0;
    }

    /* Start by downlading the test to the board
     */
    bOk = Download(Board->hFile,CmdLine);
    if (!bOk) {
        printf("Failed to download test '%s'\n", CmdLine);
        return FALSE;
    }

    /* Init */
    memset(InBuffer,0,sizeof InBuffer);
    match = &EndOfTest[0];

    for (;;) {
        /* Give it the required input
         * NB: If a program needs a blast of input use the % character
         */
        if (input && *input) {
            nBytes = 0;
            bOk = WriteFile(Board->hFile,
                            input,
                            (*input == '%') ? strlen(input) : 1,
                            &nBytes,
                            NULL);
            if (debug_level) printf("{%.*s}\n", (*input == '%') ? strlen(input) : 1, input);
            if (*input == '#') Sleep(100);
            input += nBytes;
        }

        /* Get some output
         */
        nBytesRead = 0;
        bOk = ReadFile(Board->hFile,
                       InBuffer,
                       (sizeof InBuffer) - 1,
                       &nBytesRead,
                       NULL);

        InBuffer[nBytesRead] = 0;
        printf("%s",InBuffer);

        /* Complete? 
         */
        nBytes = Match(InBuffer,match);
        if (nBytes) {
            nMatched += nBytes;
            match += nBytes;
            if (*match == 0)
                break;
        } else {
            nMatched = 0;
            match = &EndOfTest[0];
        }
    }

    /* Give it ALL the required input
     */
    while (input && *input) {
        bOk = WriteFile(Board->hFile,
                        input,
                        1,
                        &nBytes,
                        NULL);
        if (debug_level) printf("{%.*s}", 1, input);
        input++;

        /* Get some output
         */
        nBytesRead = 0;
        bOk = ReadFile(Board->hFile,
                       InBuffer,
                       (sizeof InBuffer) - 1,
                       &nBytesRead,
                       NULL);

        InBuffer[nBytesRead] = 0;
        printf("%s",InBuffer);

    }

    /* Did we see the leader byte at the end?
     * [Deals with some screwed up cases]
     */
    if (nBytesRead && (InBuffer[nBytesRead-1] == LEADER_BYTE))
        WaitForLeaderByte = FALSE;
    else
        WaitForLeaderByte = TRUE;

    return TRUE;
}

typedef struct {
    void *Next;
    char *Test;
} FAILURE_RECORD, *PFAILURE_RECORD;

PFAILURE_RECORD RecordFailure(PFAILURE_RECORD SoFar, char *Test)
{
    PFAILURE_RECORD Failure = malloc(sizeof(FAILURE_RECORD));
    int Len = strlen(Test);

    Failure->Next = SoFar;
    Failure->Test = malloc(Len+1);
    strcpy(Failure->Test,Test);
    return Failure;
}

int main(int argc, char **argv)
{
    BOARD Board;
    PBOARD s;
    int Passed = 0, Failed = 0;
    PFAILURE_RECORD Failures = NULL;
    char Buffer[MAXLINE];

    /* Parse options, if any
     */
 Recheck:
    if (argc > 1) {
        /* Any options?
         */
        if (argv[1][0] == '-') {
            switch (argv[1][1]) {
                /* List of tests to run is in a file
                 */
            case 'f':
            case 'F':
                Tests = &FileList;
                FileList.Name = argv[2];
                argc--;
                argv++;
                break;
                /* Be verbose
                 */
            case 'v':
            case 'V':
                debug_level++;
                break;
            default:
                printf("Unknown option %s, ignoring.\n", argv[1]);
            }
            argc--;
            argv++;
            goto Recheck;
        }
    }

    /* Initialize the list of tests
     */
    if (!Tests->Initialize(Tests)) {
        printf("Could not initialize test list '%s'\n", Tests->Name);
        return -1;
    }

    /* Open the serial line connection to the board
     */
    memset(&Board,0,sizeof Board);
    Board.BaudRate = 38400;
    Board.Name = "com3";
    if (argc > 1)
        Board.Name = argv[1];
    if (argc > 2)
        Board.BaudRate = atoi(argv[2]);

    s = BoardInitialize(&Board,FALSE);
    if (s == NULL) {
        printf("Could not initialize serial line '%s'\n", Board.Name);
        return -1;
    }

    /* Run all tests
     */
    printf("Running tests from %s\n", Tests->Name);
    while (Tests->Next(Tests,Buffer)) {
        printf("\n\tRunning test '%s'\n\n", Buffer);
        if (OneTest(s,Buffer)) {
            Passed++;
        } else {
            Failed++;
            Failures = RecordFailure(Failures,Buffer);
        }
    }

    printf("Ran %d tests, %d pass %d fail\n", Passed+Failed, Passed, Failed);
    while (Failures) /* yes we'll leak */
    {
        printf("\t%s\n", Failures->Test);
        Failures = Failures->Next;
    }

    /* Done
     */
    BoardFinalize(s);
    return 0;
}
