/* WINDOWS side of the serial line blast-test for Hostfs 
*/

#include <windows.h>

int debug_level = 1;
int Bytes2Send = 16;

/* 
 *    Network interface
 */
typedef struct _NETWORK {
    BOOL bInitialized;

    /* NT handles for sending&receiving. */
    char *Name;
    HANDLE hFile;

    /* The network speed&address. */
    ULONG BaudRate;

} NETWORK, * PNETWORK;

/*
 *    Constructor for serial line network object.
 */
PNETWORK SerialInitialize(PNETWORK Serial, BOOL Quietly)
{
    UINT InterByteTime;
    DCB ControlBlock;
    COMMTIMEOUTS Timeouts;
    HANDLE hLine, IoCompletionPort;

    /* 
     *    Open & check the comm line
     */
    hLine = CreateFile( Serial->Name, GENERIC_READ | GENERIC_WRITE,
                        FILE_SHARE_WRITE, /* exclusive access (except PnP..) */
                        NULL, /* no security attributes */
                        OPEN_EXISTING,
                        0, //FILE_FLAG_OVERLAPPED,
                        NULL  /* hTemplate - must be NULL for comms */
                        );

    if (hLine == INVALID_HANDLE_VALUE) {
        if (!Quietly)
            printf("Could not open %s\n",Serial->Name);
        return NULL;
    }

    Serial->hFile = hLine;

    /* 
     *    Get what we wont change
     */
    ControlBlock.DCBlength = sizeof(ControlBlock);
    if (!GetCommState(hLine,&ControlBlock)) {
        printf("GetCommState() failed on %s (x%x)\n",Serial->Name,GetLastError());
        /* 
         *    might be a named pipe, ignore
         */
    }

    /* 
     *    No parity, no flow..
     */
    ControlBlock.BaudRate = Serial->BaudRate;
    ControlBlock.fBinary = TRUE;     /* no EOF check */
    ControlBlock.fParity = FALSE;
    ControlBlock.fOutxCtsFlow = FALSE;
    ControlBlock.fOutxDsrFlow = FALSE;
    ControlBlock.fDtrControl = DTR_CONTROL_ENABLE;
    ControlBlock.fDsrSensitivity = FALSE;
    ControlBlock.fTXContinueOnXoff = TRUE;
    ControlBlock.fOutX = FALSE;
    ControlBlock.fInX = FALSE;
    ControlBlock.fErrorChar = FALSE;
    ControlBlock.fNull = FALSE;
    ControlBlock.fRtsControl = RTS_CONTROL_ENABLE;
    ControlBlock.fAbortOnError = FALSE;
    ControlBlock.ByteSize = 8;
    ControlBlock.Parity = NOPARITY;
    ControlBlock.StopBits = ONESTOPBIT;
    if (!SetCommState(hLine,&ControlBlock)) {
        printf("SetCommState() failed on %s (x%x)\n",Serial->Name,GetLastError());
        /* 
         *    might be a named pipe, ignore
         */
    }

    /* 
     *    Doing overlapped I/O on the serial line under NT is funny.
     *    By default, it will wait to fillup completely the read buffer
     *    before signalling completion.  So we have to set a timeout.
     *    Timeouts are in milliseconds units (none on xmit)
     *    Luckily, we can do the optimal thing of setting a timeout that
     *    (a) only starts after the first byte is received
     *    (b) renews itself for as long as bytes keep coming in
     *    (c) expires after that much time elapses between bytes
     */

    InterByteTime = Serial->BaudRate / 10; /* approx byte rate */
    InterByteTime = 1000 / InterByteTime; /* in msecs, most likely 0 */
    InterByteTime++;  /* roundup, avoid zero */
    if (debug_level)
        printf("SerialRead inter-byte timeout is %d msecs\n",InterByteTime);
    Timeouts.ReadIntervalTimeout = InterByteTime;
    Timeouts.ReadTotalTimeoutConstant = 0;
    Timeouts.ReadTotalTimeoutMultiplier = 0;

    Timeouts.WriteTotalTimeoutMultiplier = 0;
    Timeouts.WriteTotalTimeoutConstant = 0;

    if (!SetCommTimeouts( hLine, &Timeouts)) {
        printf("SetCommTimeouts() failed (x%x), ignoring..\n",
               GetLastError());
    }

    /*
     *    Done initializing
     */
    Serial->bInitialized = TRUE;

    return Serial;
}

void SerialFinalize( PNETWORK Serial)
{
    CloseHandle(Serial->hFile);
    Serial->hFile = INVALID_HANDLE_VALUE;
    Serial->bInitialized = FALSE;
}

#define BUFFER_SIZE 128
void test(PNETWORK Serial)
{
    BYTE OutBuffer[BUFFER_SIZE];
    BYTE InBuffer[BUFFER_SIZE];
    UINT i, nBytes, k;
    BOOL bOk;

    /* Init */
    memset(InBuffer,0,sizeof InBuffer);
    for (i = 0; i < sizeof OutBuffer; i++)
        OutBuffer[i] = i+1;

    for (k = 0; k < 6; k++) {
        /* Blast one */
#define NN Bytes2Send
        nBytes = 0;
        bOk = WriteFile(Serial->hFile,
                        OutBuffer,
                        NN,
                        &nBytes,
                        NULL);

        Sleep(1000);

        nBytes = 0;
        bOk = ReadFile(Serial->hFile,
                       InBuffer,
                       NN,
                       &nBytes,
                       NULL);

        InBuffer[nBytes] = 0;
        printf("Got %d.%d:\n %s\n", bOk, nBytes,InBuffer);
        for (i = 0; i < nBytes; i++)
            printf(" %02x", InBuffer[i]);
        printf("\n");
    }
}

int main(int argc, char **argv)
{
    NETWORK Serial;
    PNETWORK s;

    memset(&Serial,0,sizeof Serial);
    Serial.BaudRate = 38400;
    Serial.Name = "com3";
    if (argc > 1)
        Serial.Name = argv[1];
    if (argc > 2) {
        Bytes2Send = atoi(argv[2]);
        if (Bytes2Send > BUFFER_SIZE)
            Bytes2Send = BUFFER_SIZE;
    }
    if (argc > 3)
        Serial.BaudRate = atoi(argv[3]);

    s = SerialInitialize(&Serial,FALSE);
    if (s == NULL) {
        printf("Ciao.\n");
        return -1;
    }

    test(s);

    SerialFinalize(s);
}
