; ---------------------
; HL system
; ---------------------


(set-option :print-success false)

(declare-sort InputType)
(declare-sort OutputType)
(declare-sort PublicStateType)
(declare-sort PrivateStateType)
(define-sort InputSet () (Array InputType Bool))
(define-sort OutputSet () (Array OutputType Bool))


(declare-datatypes () ((ServiceResultType 
                       (ServiceResult (newPublicState PublicStateType) 
                                      (newPrivateState PrivateStateType)
                                      (output OutputType)))))

(declare-fun Service (InputType PublicStateType PrivateStateType) ServiceResultType)

(declare-const InitialAvailableInputs InputSet)
(declare-const InitialPublicState     PublicStateType)
(declare-const InitialPrivateState    PrivateStateType)
(declare-const DeadPublicState        PublicStateType)
(declare-const DeadPrivateState       PrivateStateType)

(declare-datatypes () ((HLVars (HLVars 
                             (HLAlive Bool) 
			     (HLAvailableInputs InputSet)
			     (HLObservedOutputs OutputSet)
			     (HLPublicState PublicStateType)
			     (HLPrivateState PrivateStateType)))))

(declare-const hstate1 HLVars) 
(declare-const hstate2 HLVars)


(define-fun HLMakeInputAvailable ((pre HLVars) (post HLVars))  Bool
          (exists ((input InputType))
            (and (not (select (HLAvailableInputs pre) input))
                 (= (HLAvailableInputs pre) (store (HLAvailableInputs post) input true))
                 (= (HLObservedOutputs pre) (HLObservedOutputs post))
		 (= (HLAlive pre) (HLAlive post))
		 (= (HLPublicState pre) (HLPublicState post))
		 (= (HLPrivateState pre) (HLPrivateState post)))))


(define-fun HLAdvanceService ((pre HLVars) (post HLVars))  Bool        
      (exists ((input InputType))
        (let ((hlsResult (Service input (HLPublicState pre) (HLPrivateState pre))))
            (and (select (HLAvailableInputs pre) input)
                 (HLAlive pre)
                 (= (HLAvailableInputs pre) (HLAvailableInputs post))
                 (= (HLObservedOutputs post) (store (HLObservedOutputs pre) (output hlsResult) true))
                 (= (HLPublicState post) (newPublicState hlsResult))
                 (= (HLAlive pre) (HLAlive post))
                 (= (HLPrivateState post) (newPrivateState hlsResult))
            ))))

(define-fun HLDie ((pre HLVars) (post HLVars))  Bool 
          (and 
              (not (HLAlive post))
                 (= (HLAvailableInputs pre) (HLAvailableInputs post))
                 (= (HLObservedOutputs pre) (HLObservedOutputs post))
		 (= (HLPublicState post) DeadPublicState)
		 (= (HLPrivateState post) DeadPrivateState)))

(define-fun HLInit ((state HLVars)) Bool 
          (and (HLAlive state) 
               (= (HLAvailableInputs state) InitialAvailableInputs)
               (= (HLObservedOutputs state) ((as const OutputSet) false))
               (= (HLPublicState state) InitialPublicState)
               (= (HLPrivateState state) InitialPrivateState)))


(define-fun HLNext ((pre HLVars) (post HLVars)) Bool
          (or (HLDie pre post)
              (HLAdvanceService pre post)
              (HLMakeInputAvailable pre post)))


; ---------------------
; Cryptographic primitives
; ---------------------


(declare-sort SymmetricKeyType)
(declare-datatypes () ((HashType BaseHashValue
                              (Hash (hash1 HashType) (hash2 HashType))
                              (InputType (get_InputType InputType))
                              (PublicStateType (get_PublicStateType PublicStateType))
                              (PrivateStateEncType (get_PrivateStateEncType PrivateStateEncType)))
                    (PrivateStateEncType 
                              (SymmetricEncrypt (key SymmetricKeyType) (privateState PrivateStateType)))
                    (MACType
                              (GenerateMAC (get_MACkey SymmetricKeyType) (get_MACHash HashType)))))
                              
(define-sort HashDomain () HashType)

(declare-fun SymmetricDecrypt (SymmetricKeyType PrivateStateEncType) PrivateStateType)

(define-fun ValidateMAC ((key SymmetricKeyType) (h HashType) (m MACType)) Bool (= m (GenerateMAC key h)))

; SymmetricCryptoCorrect 
(assert (forall ((key SymmetricKeyType) (privateState PrivateStateType))
                 (= (SymmetricDecrypt key (SymmetricEncrypt key privateState)) privateState)))


; ---------------------
; 2.4 Specification of the Memoir-Basic System
; ---------------------


(define-sort MACSet () (Array MACType Bool))
(define-fun emptyMACSet () MACSet ((as const MACSet) false))

(declare-datatypes () ((LL1UntrustedStorageType 
                       (LL1UntrustedStorage (publicState PublicStateType)
                                            (privateStateEnc PrivateStateEncType)
                                            (historySummary HashType)
                                            (authenticator MACType)))))

(declare-datatypes () ((LL1TrustedStorageType 
                      (LL1TrustedStorage (historySummary HashType)
                                         (symmetricKey SymmetricKeyType)))))

(declare-datatypes () ((LL1Vars
                       (LL1Vars (LL1AvailableInputs InputSet)
                                (LL1ObservedOutputs OutputSet)
                                (LL1ObservedAuthenticators MACSet)
                                (LL1Disk LL1UntrustedStorageType)
                                (LL1RAM LL1UntrustedStorageType)
                                (LL1NVRAM LL1TrustedStorageType)))))


(define-fun LL1MakeInputAvailable ((input InputType) (pre LL1Vars) (post LL1Vars)) Bool
       (and
           (not (select (LL1AvailableInputs pre) input))
           (= (LL1AvailableInputs post) (store (LL1AvailableInputs pre) input true))
           (= (LL1Disk pre) (LL1Disk post))
           (= (LL1RAM pre) (LL1RAM post))
           (= (LL1NVRAM pre) (LL1NVRAM post))
	   (= (LL1ObservedOutputs pre) (LL1ObservedOutputs post))
           (= (LL1ObservedAuthenticators pre) (LL1ObservedAuthenticators post))))


(define-fun LL1PerformOperation ((input InputType) (pre LL1Vars) (post LL1Vars)) Bool
         (and 
          (select (LL1AvailableInputs pre) input)
          (let ((symmetricKey (symmetricKey (LL1NVRAM pre))))
          (let ((publicState (publicState (LL1RAM pre))))
          (let ((privateStateEnc (privateStateEnc (LL1RAM pre))))
          (let ((stateHash (Hash (PublicStateType publicState) (PrivateStateEncType privateStateEnc))))
          (let ((historyStateBinding (Hash (historySummary (LL1RAM pre)) stateHash)))
          (let ((privateState (SymmetricDecrypt symmetricKey privateStateEnc)))
          (let ((sResult (Service input publicState privateState)))
          (let ((newPrivateStateEnc (SymmetricEncrypt symmetricKey (newPrivateState sResult))))
          (let ((newHistorySummary (Hash (historySummary (LL1NVRAM pre)) (InputType input))))
          (let ((newStateHash (Hash (PublicStateType (newPublicState sResult)) 
		                   (PrivateStateEncType newPrivateStateEnc))))
          (let ((newHistoryStateBinding (Hash newHistorySummary newStateHash)))
          (let ((newAuthenticator (GenerateMAC symmetricKey newHistoryStateBinding)))
          (and
           (ValidateMAC symmetricKey historyStateBinding (authenticator (LL1RAM pre)))
           (= (historySummary (LL1NVRAM pre)) (historySummary (LL1RAM pre)))
           (= (LL1RAM post) (LL1UntrustedStorage (newPublicState sResult) newPrivateStateEnc newHistorySummary newAuthenticator))
           (= (LL1NVRAM post) (LL1TrustedStorage newHistorySummary symmetricKey))
           (= (LL1ObservedOutputs post) (store (LL1ObservedOutputs pre) (output sResult) true))
           (= (LL1Disk post) (LL1Disk pre))
           (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
           (= (LL1ObservedAuthenticators post) (store (LL1ObservedAuthenticators pre) newAuthenticator true))
          )))))))))))))))

(define-fun LL1RepeatOperation ((input InputType) (pre LL1Vars) (post LL1Vars)) Bool
    (and 
     (select (LL1AvailableInputs pre) input)
     (let ((stateHash (Hash (PublicStateType (publicState (LL1RAM pre))) (PrivateStateEncType (privateStateEnc (LL1RAM pre))))))
     (let ((historyStateBinding (Hash (historySummary (LL1RAM pre)) stateHash)))
     (let ((privateState (SymmetricDecrypt (symmetricKey (LL1NVRAM pre)) (privateStateEnc (LL1RAM pre)))))
     (let ((sResult (Service input (publicState (LL1RAM pre)) privateState)))
     (let ((newPrivateStateEnc (SymmetricEncrypt (symmetricKey (LL1NVRAM pre)) (newPrivateState sResult))))
     (let ((newStateHash (Hash (PublicStateType (newPublicState sResult)) (PrivateStateEncType newPrivateStateEnc))))
     (let ((newHistoryStateBinding (Hash (historySummary (LL1NVRAM pre)) newStateHash)))
     (let ((newAuthenticator (GenerateMAC (symmetricKey (LL1NVRAM pre)) newHistoryStateBinding)))
     (and
       (ValidateMAC (symmetricKey (LL1NVRAM pre)) historyStateBinding (authenticator (LL1RAM pre)))
       (= (historySummary (LL1NVRAM pre)) (Hash (historySummary (LL1RAM pre)) (InputType input)))
       (= (LL1RAM post) (LL1UntrustedStorage (newPublicState sResult) newPrivateStateEnc (historySummary (LL1NVRAM pre)) newAuthenticator))
       (= (LL1ObservedOutputs post) (store (LL1ObservedOutputs pre) (output sResult) true))
       (= (LL1NVRAM post) (LL1NVRAM pre))
       (= (LL1Disk post) (LL1Disk pre))
       (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
       (= (LL1ObservedAuthenticators post) (store (LL1ObservedAuthenticators pre) newAuthenticator true)))))))))))))


(define-fun LL1Restart ((pre LL1Vars) (post LL1Vars)) Bool
     (exists ((randomSymmetricKey SymmetricKeyType) (hash HashType))
        (and 
         (not (= randomSymmetricKey (symmetricKey (LL1NVRAM pre))))
         (= (GenerateMAC randomSymmetricKey hash) (authenticator (LL1RAM post)))
         (= (LL1NVRAM post) (LL1NVRAM pre))
         (= (LL1Disk post) (LL1Disk pre))
         (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
         (= (LL1ObservedOutputs post) (LL1ObservedOutputs pre))
         (= (LL1ObservedAuthenticators post) (LL1ObservedAuthenticators pre)))))


(define-fun LL1ReadDisk ((pre LL1Vars) (post LL1Vars)) Bool
    (and
     (= (LL1RAM post) (LL1Disk pre))
     (= (LL1NVRAM post) (LL1NVRAM pre))
     (= (LL1Disk post) (LL1Disk pre))
     (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
     (= (LL1ObservedOutputs post) (LL1ObservedOutputs pre))
     (= (LL1ObservedAuthenticators post) (LL1ObservedAuthenticators pre))))



(define-fun LL1WriteDisk ((pre LL1Vars) (post LL1Vars)) Bool
    (and
     (= (LL1Disk post) (LL1RAM pre))
     (= (LL1NVRAM post) (LL1NVRAM pre))
     (= (LL1RAM post) (LL1RAM pre))
     (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
     (= (LL1ObservedOutputs post) (LL1ObservedOutputs pre))
     (= (LL1ObservedAuthenticators post) (LL1ObservedAuthenticators pre))))


(define-fun LL1CorruptRAM ((pre LL1Vars) (post LL1Vars)) Bool
   (exists ((fakeSymmetricKey SymmetricKeyType) (hash HashType))
         (let ((untrustedStorage (LL1RAM post)))
         (and
          (not (= fakeSymmetricKey (symmetricKey (LL1NVRAM pre))))
          (or (select (LL1ObservedAuthenticators pre) (authenticator untrustedStorage))
              (= (authenticator untrustedStorage) (GenerateMAC fakeSymmetricKey hash)))
          (= (LL1Disk post) (LL1Disk pre))
          (= (LL1NVRAM post) (LL1NVRAM pre))
          (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
          (= (LL1ObservedOutputs post) (LL1ObservedOutputs pre))
          (= (LL1ObservedAuthenticators post) (LL1ObservedAuthenticators pre))))))
   

(define-fun LL1RestrictedCorruption ((pre LL1Vars) (post LL1Vars)) Bool
   (and
     ; nvram
     (exists ((garbageHistorySummary HashType))
       (and
         (forall ((stateHash HashType) (authenticator MACType))
             (=> (select (LL1ObservedAuthenticators pre) authenticator)
                 (let ((historyStateBinding  (Hash garbageHistorySummary stateHash)))
                      (not (ValidateMAC (symmetricKey (LL1NVRAM pre)) historyStateBinding authenticator)))))
         (forall ((stateHash HashType) (authenticator MACType) (someHistorySummary HashType) (someInput InputType))
             (=> (select (LL1ObservedAuthenticators pre) authenticator)
                 (let ((historyStateBinding  (Hash someHistorySummary stateHash)))
                  (=> (= garbageHistorySummary (Hash someHistorySummary (InputType someInput)))
                      (not (ValidateMAC (symmetricKey (LL1NVRAM pre)) historyStateBinding authenticator))))))
         (= (LL1NVRAM post) (LL1TrustedStorage garbageHistorySummary (symmetricKey (LL1NVRAM pre))))))
     ; ram
       (or 
            (= (LL1RAM post) (LL1RAM pre))
            (exists ((randomSymmetricKey SymmetricKeyType) (hash HashType))
              (and (not (= randomSymmetricKey (symmetricKey (LL1NVRAM pre))))
                   (= (authenticator (LL1RAM post)) (GenerateMAC randomSymmetricKey hash)))))
     (= (LL1Disk post) (LL1RAM pre))
     (= (LL1AvailableInputs post) (LL1AvailableInputs pre))
     (= (LL1ObservedOutputs post) (LL1ObservedOutputs pre))
     (= (LL1ObservedAuthenticators post) (LL1ObservedAuthenticators pre))))


; TBD
(define-fun LL1Init ((state LL1Vars)) Bool 
   (exists ((symmetricKey SymmetricKeyType))
     (let ((initialPrivateStateEnc (SymmetricEncrypt symmetricKey InitialPrivateState)))
     (let ((initialStateHash (Hash (PublicStateType InitialPublicState) (PrivateStateEncType initialPrivateStateEnc))))
     (let ((initialHistoryStateBinding (Hash BaseHashValue initialStateHash)))
     (let ((initialAuthenticator (GenerateMAC symmetricKey initialHistoryStateBinding)))
     (let ((initialUntrustedStorage 
          (LL1UntrustedStorage InitialPublicState initialPrivateStateEnc BaseHashValue initialAuthenticator)))
     (let ((initialTrustedStorage  (LL1TrustedStorage BaseHashValue symmetricKey)))
     (and 
       (= (LL1Disk state) initialUntrustedStorage)
       (= (LL1RAM state) initialUntrustedStorage)
       (= (LL1NVRAM state) initialTrustedStorage)
       (= (LL1AvailableInputs state) InitialAvailableInputs)
       (= (LL1ObservedOutputs state) ((as const OutputSet) false))
       (= (LL1ObservedAuthenticators state) (store emptyMACSet initialAuthenticator true)))))))))))
     
(define-fun LL1Next ((pre LL1Vars) (post LL1Vars)) Bool
   (or
      (exists ((input InputType)) (LL1MakeInputAvailable input pre post))
      (exists ((input InputType)) (LL1PerformOperation input pre post))
      (exists ((input InputType)) (LL1RepeatOperation input pre post))
      (LL1Restart pre post)
      (LL1ReadDisk pre post)
      (LL1WriteDisk pre post)
      (LL1CorruptRAM pre post)
      (LL1RestrictedCorruption pre post)))


; -------------------
; Refinement 1: Mapping Memoir-Basic State to High-Level State


(define-fun LL1HistoryStateBindingAuthenticated ((historyStateBinding HashType) (state LL1Vars)) Bool
      (exists ((authenticator MACType))
        (and (select (LL1ObservedAuthenticators state) authenticator)
             (ValidateMAC (symmetricKey (LL1NVRAM state)) historyStateBinding authenticator))))
          

(define-fun LL1NVRAMHistorySummaryUncorrupted ((state LL1Vars)) Bool
    (exists ((stateHash HashType))
        (LL1HistoryStateBindingAuthenticated (Hash (historySummary (LL1NVRAM state)) stateHash) state)))
            


(define-fun LL1Refinement ((hstate HLVars) (lstate LL1Vars)) Bool
    (and
     (= (HLAvailableInputs hstate) (LL1AvailableInputs lstate))
     (= (HLObservedOutputs hstate) (LL1ObservedOutputs lstate))
     (ite (LL1NVRAMHistorySummaryUncorrupted lstate)
           (let ((refPrivateStateEnc (SymmetricEncrypt (symmetricKey (LL1NVRAM lstate)) (HLPrivateState hstate))))
           (let ((refStateHash (Hash (PublicStateType (HLPublicState hstate)) (PrivateStateEncType refPrivateStateEnc))))
           (let ((refHistoryStateBinding (Hash (historySummary (LL1NVRAM lstate)) refStateHash)))
           (and
             (LL1HistoryStateBindingAuthenticated refHistoryStateBinding lstate)
             (HLAlive hstate)))))
          ; else
          (and
           (= (HLPublicState hstate) DeadPublicState)
           (= (HLPrivateState hstate) DeadPrivateState)
           (not (HLAlive hstate))))))

; ----------------------------------------------
; Invariants

(define-fun UnforgeabilityInvariant ((state LL1Vars)) Bool 
      (forall ((historyStateBinding HashType))
              (=> (ValidateMAC (symmetricKey (LL1NVRAM state)) historyStateBinding (authenticator (LL1RAM state)))
                  (select (LL1ObservedAuthenticators state) (authenticator (LL1RAM state))))))


(define-fun InclusionInvariant ((state LL1Vars)) Bool 
      (forall 
        ((input InputType) 
        (historySummary1 HashType) 
        (publicState PublicStateType) 
        (privateStateEnc PrivateStateEncType))
          (let ((stateHash (Hash (PublicStateType publicState) (PrivateStateEncType privateStateEnc))))
          (let ((historyStateBinding (Hash historySummary1 stateHash)))
          (let ((privateState (SymmetricDecrypt (symmetricKey (LL1NVRAM state)) privateStateEnc)))
          (let ((sResult (Service input publicState privateState)))
          (let ((newPrivateStateEnc (SymmetricEncrypt (symmetricKey (LL1NVRAM state)) (newPrivateState sResult))))
          (let ((newStateHash (Hash (PublicStateType (newPublicState sResult)) (PrivateStateEncType newPrivateStateEnc))))
          (let ((newHistoryStateBinding (Hash (historySummary (LL1NVRAM state)) newStateHash)))
          (=> 
           (and
            (= (historySummary (LL1NVRAM state)) (Hash historySummary1 (InputType input)))
            (LL1HistoryStateBindingAuthenticated historyStateBinding state))
           (and
            (select (LL1ObservedOutputs state) (output sResult))
            (LL1HistoryStateBindingAuthenticated newHistoryStateBinding state))))))))))))


(define-fun UniquenessInvariant ((state LL1Vars)) Bool
     (forall ((h1 HashType) (h2 HashType))
          (let ((historyStateBinding1 (Hash (historySummary (LL1NVRAM state)) h1)))
          (let ((historyStateBinding2 (Hash (historySummary (LL1NVRAM state)) h2)))
          (=>
           (and
            (LL1HistoryStateBindingAuthenticated historyStateBinding1 state)
            (LL1HistoryStateBindingAuthenticated historyStateBinding2 state))
           (= h1 h2))))))

(define-fun UnforgeabilityInvariantInductive ((state LL1Vars)) Bool 
      (and 
       (UnforgeabilityInvariant state)
       (forall ((historyStateBinding HashType))
              (=> (ValidateMAC (symmetricKey (LL1NVRAM state)) historyStateBinding (authenticator (LL1Disk state)))
                  (select (LL1ObservedAuthenticators state) (authenticator (LL1Disk state)))))))

(define-fun LL1Inv ((state LL1Vars)) Bool
   (and 
     (UnforgeabilityInvariantInductive state)
     (InclusionInvariant state)
     (UniquenessInvariant state)))

; verify UnforgeabilityInvariant:

(declare-fun state1 () LL1Vars) 
(declare-fun state2 () LL1Vars)

; Initial condition
(push)
(assert (LL1Init state1))
(assert (not (LL1Inv state1)))
(check-sat)
(pop)

; prove unforgeability invariant
(push)
(assert (LL1Next state1 state2))
(assert (UnforgeabilityInvariantInductive state1))
(assert (not (UnforgeabilityInvariantInductive state2)))
(check-sat)
(pop)

; prove that inclusion invariant is inductive:
(push)
(assert (LL1Next state1 state2))
(assert (LL1Inv state1))
(assert (not (InclusionInvariant state2)))
;(check-sat)
(pop)

; prove that uniqueness invariant is inductive:
; TBD: I think it requires the version of the cardinality invariant.
;(push)
;(assert (LL1Next state1 state2))
;(assert (LL1Inv state1)
;(assert (not (UniquenessInvariant state2)))
;(check-sat)
;model
;(pop)

;
; non-theorem
;
(push)
(assert (LL1Next state1 state2))
(assert (LL1Inv state1))
(assert (LL1Inv state2))
(assert (LL1Refinement hstate1 state1))
(assert (LL1Refinement hstate2 state2))
(assert (not (HLNext hstate1 hstate2)))
;(check-sat)
;(get-model)
(pop)


; ---------------------------------------------
; Specification of the Memoir-Efficient System
; ---------------------------------------------


(declare-datatypes () ((HistorySummaryType (HistorySummary (anchor HashType) (extension HashType)))))

(declare-datatypes () ((LL2UntrustedStorageType 
              (LL2UntrustedStorage
                     (publicState PublicStateType)
                     (privateStateEnc PrivateStateEncType)
                     (historySummary HistorySummaryType)
                     (authenticator MACType)))))

(declare-datatypes () ((LL2TrustedStorageType 
              (LL2TrustedStorage
                     (historySummaryAnchor HashType)
                     (symmetricKey SymmetricKeyType)
                     (hashBarrier HashType)
                     (extensionInProgress Bool)))))


(declare-datatypes () ((LL2Vars 
                     (LL2Vars 
                       (LL2AvailableInputs InputSet)
                       (LL2ObservedOutputs OutputSet)
                       (LL2ObservedAuthenticators MACSet)
                       (LL2Disk LL2UntrustedStorageType)
                       (LL2RAM LL2UntrustedStorageType)
                       (LL2NVRAM LL2TrustedStorageType)
                       (LL2SPCR HashType)))))


(define-fun Checkpoint ((historySummary HistorySummaryType)) HistorySummaryType
    (let ((checkpointedAnchor (Hash (anchor historySummary) (extension historySummary))))
    (let ((checkpointedHistorySummary (HistorySummary checkpointedAnchor BaseHashValue)))
    (ite (= (extension historySummary) BaseHashValue)
         historySummary
         checkpointedHistorySummary))))


(define-fun Successor ((historySummary HistorySummaryType) (input InputType) (hashBarrier HashType)) HistorySummaryType 
    (let ((securedInput (Hash hashBarrier (InputType input))))
    (let ((newAnchor (anchor historySummary)))
    (let ((newExtension (Hash (extension historySummary) securedInput)))
    (HistorySummary newAnchor newExtension)))))


(define-fun LL2MakeInputAvailable ((pre LL2Vars) (post LL2Vars)) Bool
   (exists ((input InputType))
      (and
        (not (select (LL2AvailableInputs pre) input))
        (= (LL2AvailableInputs post) (store (LL2AvailableInputs pre) input true))
        (= (LL2Disk post) (LL2Disk pre))
        (= (LL2RAM post) (LL2RAM pre))
        (= (LL2NVRAM post) (LL2NVRAM pre))
        (= (LL2SPCR post) (LL2SPCR pre))
        (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
        (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre)))))

(define-fun LL2PerformOperation ((pre LL2Vars) (post LL2Vars)) Bool
    (exists ((input InputType))
      (and (select (LL2AvailableInputs pre) input)
       (let ((nvram (LL2NVRAM pre)))
       (let ((ram (LL2RAM pre)))
       (let ((historySummaryHash (Hash (anchor (historySummary ram)) (extension (historySummary ram)))))
       (let ((stateHash (Hash (PublicStateType (publicState ram)) (PrivateStateEncType (privateStateEnc ram)))))
       (let ((historyStateBinding (Hash historySummaryHash stateHash)))
       (let ((privateState (SymmetricDecrypt (symmetricKey nvram) (privateStateEnc ram))))
       (let ((sResult (Service input (publicState ram) privateState)))
       (let ((newPrivateStateEnc (SymmetricEncrypt (symmetricKey nvram) (newPrivateState sResult))))
       (let ((currentHistorySummary (HistorySummary (historySummaryAnchor nvram) (LL2SPCR pre))))
       (let ((newHistorySummary (Successor currentHistorySummary  input (hashBarrier nvram))))
       (let ((newHistorySummaryHash (Hash (anchor newHistorySummary) (extension newHistorySummary))))
       (let ((newStateHash  (Hash (PublicStateType (newPublicState sResult)) (PrivateStateEncType newPrivateStateEnc))))
       (let ((newHistoryStateBinding (Hash newHistorySummaryHash newStateHash)) )
       (let ((newAuthenticator (GenerateMAC (symmetricKey nvram) newHistoryStateBinding)))
       (and 
         (ValidateMAC (symmetricKey nvram) historyStateBinding (authenticator ram))
         (ite (extensionInProgress nvram)
              (and (not (= (LL2SPCR pre) BaseHashValue))
                   (= currentHistorySummary (historySummary ram)))
              (and (= (LL2SPCR pre) BaseHashValue)
                   (= currentHistorySummary (Checkpoint (historySummary ram)))))
         (= (LL2RAM post) (LL2UntrustedStorage 
                             (newPublicState sResult)
                             newPrivateStateEnc
                    	     newHistorySummary
                    	     newAuthenticator))
         (= (LL2NVRAM post) (LL2TrustedStorage 
                             (historySummaryAnchor nvram)
                             (symmetricKey nvram)
                    	     (hashBarrier nvram)
                    	     true))
         (= (LL2SPCR post) (extension newHistorySummary))
         (= (LL2ObservedOutputs post) (store (LL2ObservedOutputs pre) (output sResult) true))
         (= (LL2Disk post) (LL2Disk pre))
         (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
         (= (LL2ObservedAuthenticators post) (store (LL2ObservedAuthenticators pre) newAuthenticator true))))))))))))))))))))

(define-fun LL2RepeatOperation ((pre LL2Vars) (post LL2Vars)) Bool
     (exists ((input InputType))
      (and (select (LL2AvailableInputs pre) input)
       (let ((nvram (LL2NVRAM pre)))
       (let ((ram (LL2RAM pre)))
       (let ((historySummaryHash (Hash (anchor (historySummary ram)) (extension (historySummary ram)))))
       (let ((stateHash (Hash (PublicStateType (publicState ram)) (PrivateStateEncType (privateStateEnc ram)))))
       (let ((historyStateBinding (Hash historySummaryHash stateHash)))
       (let ((newHistorySummary (Successor (historySummary ram) input (hashBarrier nvram))))
       (let ((checkpointedHistorySummary (Checkpoint (historySummary ram))))
       (let ((newCheckpointedHistorySummary (Successor checkpointedHistorySummary input (hashBarrier nvram))))
       (let ((checkpointedNewHistorySummary (Checkpoint newHistorySummary)))
       (let ((checkpointedNewCheckpointedHistorySummary (Checkpoint newCheckpointedHistorySummary)))
       (let ((privateState (SymmetricDecrypt (symmetricKey nvram) (privateStateEnc ram))))
       (let ((sResult (Service input (publicState ram) privateState)))
       (let ((newPrivateStateEnc (SymmetricEncrypt (symmetricKey nvram) (newPrivateState sResult))))
       (let ((currentHistorySummary (HistorySummary (historySummaryAnchor nvram) (LL2SPCR pre))))
       (let ((currentHistorySummaryHash (Hash (historySummaryAnchor nvram) (LL2SPCR pre))))
       (let ((newStateHash (Hash (PublicStateType (newPublicState sResult)) (PrivateStateEncType newPrivateStateEnc))))
       (let ((newHistoryStateBinding (Hash currentHistorySummaryHash newStateHash)))
       (let ((newAuthenticator (GenerateMAC (symmetricKey nvram) newHistoryStateBinding)))
       (and
         (ValidateMAC (symmetricKey nvram) historyStateBinding (authenticator ram))
         (not (extensionInProgress nvram))
         (= (LL2SPCR pre) BaseHashValue)
         (or (= currentHistorySummary checkpointedNewHistorySummary)
             (= currentHistorySummary checkpointedNewCheckpointedHistorySummary))
         (= (LL2RAM pre) (LL2UntrustedStorage
                             (newPublicState sResult)
                             newPrivateStateEnc
                    	     currentHistorySummary
                    	     newAuthenticator))
         (= (LL2ObservedOutputs post) (store (LL2ObservedOutputs pre) (output sResult) true))
         (= (LL2NVRAM post) (LL2NVRAM pre))
         (= (LL2SPCR post) (LL2SPCR pre))
         (= (LL2Disk post) (LL2Disk pre))
         (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
         (= (LL2ObservedAuthenticators post) (store (LL2ObservedAuthenticators pre) newAuthenticator true))))))))))))))))))))))))

(define-fun LL2TakeCheckpoint ((pre LL2Vars) (post LL2Vars)) Bool
    (let ((newHistorySummaryAnchor (Hash (historySummaryAnchor (LL2NVRAM pre)) (LL2SPCR pre))))
    (and
       (extensionInProgress (LL2NVRAM pre))
       (not (= (LL2SPCR pre) BaseHashValue))
       (= (LL2RAM post) (LL2RAM pre))
       (= (LL2SPCR post) (LL2SPCR pre))
       (= (LL2Disk post) (LL2Disk pre))
       (= (LL2NVRAM post) (LL2TrustedStorage 
                               newHistorySummaryAnchor
                               (symmetricKey (LL2NVRAM pre))
                               (hashBarrier (LL2NVRAM pre))
                               false))
       (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
       (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
       (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre)))))

(define-fun LL2Restart ((pre LL2Vars) (post LL2Vars)) Bool
    (exists ((randomSymmetricKey SymmetricKeyType) (hash HashType))
       (let ((untrustedStorage (LL2RAM post)))
       (and 
        (not (= randomSymmetricKey (symmetricKey (LL2NVRAM pre))))
        (= (authenticator untrustedStorage) (GenerateMAC randomSymmetricKey hash))
        (= (LL2SPCR post) BaseHashValue)
        (= (LL2Disk post) (LL2Disk pre))
        (= (LL2NVRAM post) (LL2NVRAM pre))
        (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
        (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
        (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre))))))


(define-fun LL2ReadDisk ((pre LL2Vars) (post LL2Vars)) Bool
   (and 
     (= (LL2RAM post) (LL2Disk pre))
     (= (LL2NVRAM post) (LL2NVRAM pre))
     (= (LL2SPCR post) (LL2SPCR pre))
     (= (LL2Disk post) (LL2Disk pre))
     (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
     (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
     (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre))))


(define-fun LL2WriteDisk ((pre LL2Vars) (post LL2Vars)) Bool
   (and 
     (= (LL2Disk post) (LL2RAM pre))
     (= (LL2NVRAM post) (LL2NVRAM pre))
     (= (LL2SPCR post) (LL2SPCR pre))
     (= (LL2RAM post) (LL2RAM pre))
     (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
     (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
     (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre))))

(define-fun LL2CorruptRAM ((pre LL2Vars) (post LL2Vars)) Bool
    (exists ((fakeSymmetricKey SymmetricKeyType) (hash HashType))
     (let ((untrustedStorage (LL2RAM post)))
     (and 
      (or (select (LL2ObservedAuthenticators pre) (authenticator untrustedStorage))
          (= (GenerateMAC fakeSymmetricKey hash) (authenticator untrustedStorage)))
      (= (LL2Disk post) (LL2Disk pre))
      (= (LL2NVRAM post) (LL2NVRAM pre))
      (= (LL2SPCR post) (LL2SPCR pre))
      (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
      (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
      (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre))))))

(define-fun LL2CorruptSPCR ((pre LL2Vars) (post LL2Vars)) Bool
    (exists ((fakeHash HashDomain))
      (let ((newHistorySummaryExtension (Hash (LL2SPCR pre) fakeHash)))
      (and
       (forall ((fakeInput InputType)) (not (= fakeHash (Hash (hashBarrier (LL2NVRAM pre)) (InputType fakeInput)))))
       (= (LL2Disk post) (LL2Disk pre))
       (= (LL2NVRAM post) (LL2NVRAM pre))
       (= (LL2SPCR post) newHistorySummaryExtension)
       (= (LL2AvailableInputs post) (LL2AvailableInputs pre))
       (= (LL2ObservedOutputs post) (LL2ObservedOutputs pre))
       (= (LL2ObservedAuthenticators post) (LL2ObservedAuthenticators pre))))))
       

;LL2Init == \*`^\label{DEF-LL2Init}^'
;    \E symmetricKey \in SymmetricKeyType, hashBarrier \in HashType :
;        LET
;            initialPrivateStateEnc == SymmetricEncrypt(symmetricKey, InitialPrivateState)
;            initialStateHash == Hash(InitialPublicState, initialPrivateStateEnc)
;            initialHistorySummary == [
;                anchor |-> BaseHashValue,
;                extension |-> BaseHashValue ]
;            initialHistorySummaryHash == Hash(BaseHashValue, BaseHashValue)
;            initialHistoryStateBinding == Hash(initialHistorySummaryHash, initialStateHash)
;            initialAuthenticator == GenerateMAC(symmetricKey, initialHistoryStateBinding)
;            initialUntrustedStorage == [
;                publicState |-> InitialPublicState,
;                privateStateEnc |-> initialPrivateStateEnc,
;                historySummary |-> initialHistorySummary,
;                authenticator |-> initialAuthenticator ]
;            initialTrustedStorage == [
;                historySummaryAnchor |-> BaseHashValue,
;                symmetricKey |-> symmetricKey,
;                hashBarrier |-> hashBarrier,
;                extensionInProgress |-> FALSE ]
;        IN
;            /\  LL2Disk = initialUntrustedStorage
;            /\  LL2RAM = initialUntrustedStorage
;            /\  LL2NVRAM = initialTrustedStorage
;            /\  LL2SPCR = BaseHashValue
;            /\  LL2AvailableInputs = InitialAvailableInputs
;            /\  LL2ObservedOutputs = {}
;            /\  LL2ObservedAuthenticators = {initialAuthenticator}

(define-fun LL2Next ((pre LL2Vars) (post LL2Vars)) Bool
   (or 
     (LL2MakeInputAvailable pre post)
     (LL2PerformOperation pre post)
     (LL2RepeatOperation pre post)
     (LL2TakeCheckpoint pre post)
     (LL2Restart pre post)
     (LL2ReadDisk pre post)
     (LL2WriteDisk pre post)
     (LL2CorruptRAM pre post)
     (LL2CorruptSPCR pre post)))


; ------------------------------------------------------
; LL2Refinement
; ------------------------------------------------------


(define-fun LL2HistorySummaryIsSuccessor 
      ((historySummary HistorySummaryType) 
       (previousHistorySummary HistorySummaryType)
       (input InputType)
       (hashBarrier HashType)) Bool
   (let ((successorHistorySummary (Successor previousHistorySummary input hashBarrier)))
   (let ((checkpointedSuccessorHistorySummary (Checkpoint successorHistorySummary)))
   (or (= historySummary successorHistorySummary)
       (= historySummary checkpointedSuccessorHistorySummary)))))


(declare-datatypes () ((Depth Zero (Succ (pred Depth)))))
(define-fun initDepth () Depth (Succ (Succ (Succ Zero))))

(declare-fun HistorySummariesMatchN (HashType HistorySummaryType HashType Depth) Bool)

(define-fun HistorySummariesMatch ((ll1 HashType) (ll2 HistorySummaryType) (hashBarrier HashType)) Bool
       (HistorySummariesMatchN ll1 ll2 hashBarrier initDepth))


(define-fun HistorySummariesMatchRecursion ((ll1HistorySummary HashType)
                                            (ll2HistorySummary HistorySummaryType) 
                                            (hashBarrier HashType)
                                            (depth Depth)) Bool
   (exists 
    ((input InputType) 
    (previousLL1HistorySummary HashType)
    (previousLL2HistorySummary HistorySummaryType))
    (and 
      (HistorySummariesMatchN previousLL1HistorySummary previousLL2HistorySummary hashBarrier depth)
      (= ll1HistorySummary (Hash previousLL1HistorySummary (InputType input)))
      (LL2HistorySummaryIsSuccessor ll2HistorySummary previousLL2HistorySummary input hashBarrier))))

(assert
  (forall 
    ((ll1HistorySummary HashType)
    (ll2HistorySummary HistorySummaryType)
    (hashBarrier HashType)
    (depth Depth))
    (let ((ll2InitialHistorySummary (HistorySummary BaseHashValue BaseHashValue)))
     (= (HistorySummariesMatchN ll1HistorySummary ll2HistorySummary hashBarrier (Succ depth)) 
     (ite 
      (= ll2HistorySummary ll2InitialHistorySummary)
        (= ll1HistorySummary BaseHashValue)
        (HistorySummariesMatchRecursion ll1HistorySummary ll2HistorySummary hashBarrier depth))))))





(define-fun AuthenticatorsMatch ((ll1Authenticator MACType) (ll2Authenticator MACType) 
                                 (symmetricKey SymmetricKeyType) (hashBarrier HashType)) Bool 
   (exists 
    ((stateHash HashType)
    (ll1HistorySummary HashType)
    (ll2HistorySummary HistorySummaryType))
      (let ((ll1HistoryStateBinding (Hash ll1HistorySummary stateHash)))
      (let ((ll2HistorySummaryHash  (Hash (anchor ll2HistorySummary) (extension ll2HistorySummary))))
      (let ((ll2HistoryStateBinding (Hash ll2HistorySummaryHash stateHash)))
      (and
        (ValidateMAC symmetricKey ll2HistoryStateBinding ll2Authenticator)
        (= ll1Authenticator (GenerateMAC symmetricKey ll1HistoryStateBinding))
        (HistorySummariesMatch ll1HistorySummary ll2HistorySummary hashBarrier)))))))

(define-fun AuthenticatorSetsMatch1 ((ll1Authenticators MACSet) (ll2Authenticators MACSet) 
                                    (symmetricKey SymmetricKeyType) (hashBarrier HashType)) Bool
   (forall ((l1 MACType))  
       (=> (select ll1Authenticators l1)
           (exists ((l2 MACType))
              (and (select ll2Authenticators l2) (AuthenticatorsMatch l1 l2 symmetricKey hashBarrier))))))

 
(define-fun AuthenticatorSetsMatch ((ll1Authenticator MACSet) (ll2Authenticator MACSet) 
                                    (symmetricKey SymmetricKeyType) (hashBarrier HashType)) Bool
   (and (AuthenticatorSetsMatch1 ll1Authenticator ll2Authenticator symmetricKey hashBarrier)
        (AuthenticatorSetsMatch1 ll2Authenticator ll1Authenticator symmetricKey hashBarrier)))

(declare-const CrazyHashValue HashType)

;(assert 
;     (not (is_Hash CrazyHashValue)))
;     (not (= CrazyHashValue BaseHashValue)))

;     (forall ((h1 HashType) (h2 HashDomain)) (not (= CrazyHashValue (Hash h1 h2))))

(define-fun LL2NVRAMLogicalHistorySummary ((state LL2Vars)) HistorySummaryType
   (ite (extensionInProgress (LL2NVRAM state))
     (ite (= (LL2SPCR state) BaseHashValue)
          (HistorySummary (historySummaryAnchor (LL2NVRAM state)) CrazyHashValue)
          (HistorySummary (historySummaryAnchor (LL2NVRAM state)) (LL2SPCR state)))
     (HistorySummary (historySummaryAnchor (LL2NVRAM state)) BaseHashValue)))

(define-fun LL2Refinement ((state1 LL1Vars) (state2 LL2Vars)) Bool
   (and
    (= (LL1AvailableInputs state1) (LL2AvailableInputs state2))
    (= (LL1ObservedOutputs state1) (LL2ObservedOutputs state2))
    (AuthenticatorSetsMatch
            (LL1ObservedAuthenticators state1)
            (LL2ObservedAuthenticators state2)
            (symmetricKey (LL2NVRAM state2))
            (hashBarrier (LL2NVRAM state2)))
    (= (publicState (LL1Disk state1)) (publicState (LL2Disk state2)))
    (= (privateStateEnc (LL1Disk state1)) (privateStateEnc (LL2Disk state2)))
    (HistorySummariesMatch (historySummary (LL1Disk state1))
                           (historySummary (LL2Disk state2))
                           (hashBarrier (LL2NVRAM state2)))
    (exists ((symmetricKey SymmetricKeyType))
       (AuthenticatorsMatch
         (authenticator (LL1Disk state1))
         (authenticator (LL2Disk state2))
         symmetricKey
         (hashBarrier (LL2NVRAM state2))))
    (= (publicState (LL1RAM state1)) (publicState (LL2RAM state2)))
    (= (privateStateEnc (LL1RAM state1)) (privateStateEnc (LL2RAM state2)))
    (HistorySummariesMatch (historySummary (LL1RAM state1))
                           (historySummary (LL2RAM state2))
                           (hashBarrier (LL2NVRAM state2)))
    (exists ((symmetricKey SymmetricKeyType))
       (AuthenticatorsMatch
         (authenticator (LL1RAM state1))
         (authenticator (LL2RAM state2))
         symmetricKey
         (hashBarrier (LL2NVRAM state2))))
    (HistorySummariesMatch (historySummary (LL1NVRAM state1))
                           (LL2NVRAMLogicalHistorySummary state2)
                           (hashBarrier (LL2NVRAM state2)))
    (= (symmetricKey (LL1NVRAM state1)) (symmetricKey (LL2NVRAM state2)))))


; Another non-theorem. Z3 produces a counter-example.
(declare-const l2state1 LL2Vars) 
(declare-const l2state2 LL2Vars)
(push)
(set-option :smt.mbqi false)
(assert (LL2Next l2state1 l2state2))
(assert (LL2Refinement state1 l2state1))
(assert (LL2Refinement state2 l2state2))
(assert (not (LL1Next state1 state2)))
(check-sat)
(get-model)
(pop)