﻿// -----------------------------------------------------------------------
// <copyright file="ClineArguments.cs" company="Microsoft">
// Copyright (c) Microsoft. All Rights Reserved
// Author: Karthick Jayaraman (karjay@microsoft.com)
// </copyright>
// -----------------------------------------------------------------------

namespace Microsoft.Cis.Security.Tools.FirewallBenchmarks
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;

    /// <summary>
    /// TODO: Update summary.
    /// </summary>
    public class ClineArguments
    {
        /// <summary>
        /// Dictionary of commandline arguments. The key is
        /// the name of the parameter, and value is a string
        /// array of parameters.
        /// </summary>
        private Dictionary<string, string[]> parameters;

        /// <summary>
        /// Help message for the command-line tool
        /// </summary>
        private string usage;

        /// <summary>
        /// Initializes a new instance of the ClineArguments class.
        /// Takes a string array of arguments and parses them in 
        /// to key-value pairs of the type that the dictionary of 
        /// parameters accepts
        /// </summary>
        /// <param name="args">String array of arguments passed 
        /// into the main function</param>
        public ClineArguments(string[] args, string helpMessage)
        {
            this.parameters = new Dictionary<string, string[]>();
            this.Parse(args);
            this.Usage = helpMessage;
        }
        
        /// <summary>
        /// Gets or sets help message for command-line tool
        /// </summary>
        public string Usage
        {
            get { return this.usage; }
            set { this.usage = value; }
        }

        /// <summary>
        /// Indexer for ClineArguments
        /// </summary>
        /// <param name="parameter">They key string</param>
        /// <returns>string array of values for the key</returns>
        public object this[string parameter]
        {
            get
            {
                return this.parameters[parameter];
            }
        }

        /// <summary>
        /// Check whether the specified parameter was
        /// used in the commandline
        /// </summary>
        /// <param name="parameter">Parameter to check</param>
        /// <returns>True or False</returns>
        public bool Contains(string parameter)
        {
            return this.parameters.Keys.Contains<string>(parameter);
        }

        /// <summary>
        /// Retrieve the nth value provided for the specified 
        /// parameter in the commandline
        /// </summary>
        /// <param name="parameter">Name of the parameter</param>
        /// <param name="n">The index of the value in the list </param>
        /// <returns>The n'th value for the specified parameter
        /// in the commandline</returns>
        public string GetNthValue(string parameter, int n)
        {
            try
            {
                string[] values = this.parameters[parameter];

                return values[n];
            }
            catch (KeyNotFoundException)
            {
                Console.WriteLine("Cannot find mandatory parameter " + parameter);
                this.PrintUsage();
                return null;
            }
        }

        /// <summary>
        /// Parse the string array of arguments into
        /// into key-value pairs of the form (string, string[])
        /// </summary>
        /// <param name="args">String array of arguments 
        /// passed into the main function</param>
        public void Parse(string[] args)
        {
            Regex keyValueSplitter = new Regex(@"^-{1,2}|^/|=|:");
            Regex valueSplitter = new Regex(@"\s|,|;");
            string prevKey = null;
            List<string> list = new List<string>();

            foreach (var arg in args)
            {
                string[] parts = keyValueSplitter.Split(arg);

                switch (parts.Length)
                {
                    case 1:
                        {
                            string[] values = valueSplitter.Split(parts[0]);
                            list.AddRange(values);
                            break;
                        }

                    case 2:
                        {
                            this.StoreKeyValueSet(prevKey, list);
                            prevKey = parts[1];
                            list.RemoveAll(item => true);
                            break;
                        }

                    case 3:
                        {
                            this.StoreKeyValue(parts[1], parts[2]);
                            break;
                        }
                }
            }

            this.StoreKeyValueSet(prevKey, list);
        }

        /// <summary>
        /// Validate whether the commandline arguments have all
        /// the needed mandatory arguments. Prints the help message
        /// if the commandline does not have the needed mandatory 
        /// arguments. 
        /// </summary>
        /// <returns>Returns true if all the mandatory arguments 
        /// are available in the command line, and false otherwise. 
        /// </returns>
        public bool ValidateArgs(string[] mandatoryArgs)
        {
            bool containsAllMandatoryArgs = false;

            if (mandatoryArgs.Length == 0)
            {
                return true;
            }

            // Check if the command-line contains each of the mandatory arguments
            foreach (var arg in mandatoryArgs)
            {
                if (!this.Contains(arg))
                {
                    containsAllMandatoryArgs = false;
                }
            }

            // Display help message if the validation fails. 
            if (!containsAllMandatoryArgs)
            {
                PrintUsage();
            }

            return containsAllMandatoryArgs;
        }

        /// <summary>
        /// Print usage information
        /// </summary>
        private void PrintUsage()
        {
            Console.WriteLine("Does not have all mandatory arguments \n");
            Console.WriteLine(this.Usage);
            System.Environment.Exit(0);
        }

        /// <summary>
        /// Add key and list of values to the parameters dictionary
        /// </summary>
        /// <param name="prevKey">Key name</param>
        /// <param name="list">List of values associated witht he key</param>
        private void StoreKeyValueSet(string prevKey, List<string> list)
        {
            if (prevKey != null)
            {
                if (list.Count == 0)
                {
                    list.Add("true");
                }

                this.parameters.Add(prevKey, list.ToArray());
            }
        }

        /// <summary>
        /// Add key and value to the parameters dictionary
        /// </summary>
        /// <param name="key">Key name to be added</param>
        /// <param name="value">Value to be added</param>
        private void StoreKeyValue(string key, string value)
        {
            this.parameters.Add(key, new string[] { value });
        }
    }
}
